/* $Id: namespace.c,v 1.7 2005/12/14 08:22:32 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * Postgres namespace interface type
 */
#include <setjmp.h>
#include <postgres.h>
#include <fmgr.h>
#include <access/heapam.h>
#include <access/htup.h>
#include <access/tupdesc.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_namespace.h>
#include <catalog/pg_conversion.h>
#include <catalog/pg_operator.h>
#include <catalog/pg_opclass.h>
#include <catalog/namespace.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/relcache.h>
#include <pypg/environment.h>
#include <pypg/postgres.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/type/object.h>
#include <pypg/type/system.h>
#include <pypg/function.h>
#include <pypg/error.h>
#include <pypg/namespace.h>

static PyObj
ns_type(PyObj self, PyObj name)
{
	PyObj str, rob = NULL;
	Oid oid = InvalidOid;

	str = PyObject_Str(name);

	PG_TRY();
	{
		oid = GetSysCacheOid(TYPENAMENSP,
								PointerGetDatum(PyString_AS_STRING(str)),
								ObjectIdGetDatum(PyPgObject_FetchOid(self)),
								0, 0);
	}
	PYPG_CATCH_END(Py_DECREF(str); return(NULL));

	if (oid == InvalidOid)
	{
		PyErr_Format(PyExc_LookupError,
			"failed to find type with name \"%s\" in namespace \"%s\"",
			PyString_AS_STRING(str),
			NameStr(PyPgNamespace_FetchStructure(self)->nspname));
	}
	else
	{
		rob = PyPgType_FromOid(oid);
	}
	Py_DECREF(str);

	return(rob);
}

static PyObj
ns_relation(PyObj self, PyObj name)
{
	PyObj str, rob;
	Oid oid = InvalidOid;

	str = PyObject_Str(name);

	PG_TRY();
	{
		oid = GetSysCacheOid(RELNAMENSP,
								PointerGetDatum(PyString_AS_STRING(str)),
								ObjectIdGetDatum(PyPgObject_FetchOid(self)),
								0, 0);
		Py_DECREF(str);
	}
	PYPG_CATCH_END(Py_DECREF(str); return(NULL));

	rob = Py_None;
	Py_INCREF(rob);
	return(rob);
}

static PyObj
ns_conversion(PyObj self, PyObj name)
{
	static PyObj cont = NULL;
	HeapTuple ht = NULL;
	PyObj str, rob;

	if (cont == NULL)
	{
		Oid to;
		PG_TRY();
		{
			Relation r;
#if PGV_MM == 80
			r = RelationSysNameGetRelation("pg_conversion");
#else
			r = RelationIdGetRelation(ConversionRelationId);
#endif
			to = r->rd_rel->reltype;
			RelationClose(r);
		}
		PYPG_CATCH_END(return(NULL));

		cont = PyPgType_FromOid(to);
		if (cont == NULL) return(NULL);
	}

	str = PyObject_Str(name);
	if (str == NULL) return(NULL);

	PG_TRY();
	{
		HeapTupleHeader hth;
		ht = SearchSysCache(CONNAMENSP,
								PointerGetDatum(PyString_AS_STRING(str)),
								ObjectIdGetDatum(PyPgObject_FetchOid(self)),
								0, 0);
		Py_DECREF(str);
		str = NULL;
		hth = HeapTupleHeader_FromHeapTuple(ht);
		ReleaseSysCache(ht);
		ht = NULL;
		rob = PyPgObject_New(cont, PointerGetDatum(hth));
	}
	PYPG_CATCH_END(if (ht) ReleaseSysCache(ht);; Py_XDECREF(str); return(NULL));

	return(rob);
}

static PyObj
ns_operatorclass(PyObj self, PyObj args)
{
	static PyObj oct = NULL;
	HeapTuple ht = NULL;
	PyObj str, rob;
	Oid opcamid = InvalidOid;

	if (oct == NULL)
	{
		Oid to;
		PG_TRY();
		{
			Relation r;
#if PGV_MM == 80
			r = RelationSysNameGetRelation("pg_opclass");
#else
			r = RelationIdGetRelation(OperatorClassRelationId);
#endif
			to = r->rd_rel->reltype;
			RelationClose(r);
		}
		PYPG_CATCH_END(return(NULL));

		oct = PyPgType_FromOid(to);
		if (oct == NULL) return(NULL);
	}

	if (PyTuple_GET_SIZE(args) != 2)
	{
		PyErr_Format(PyExc_TypeError,
			"namespace based operator class lookup requires exactly 2 arguments");
		return(NULL);
	}

	str = PyObject_Str(PyTuple_GET_ITEM(args, 0));
	opcamid = Oid_FromPyObject(PyTuple_GET_ITEM(args, 1));

	PG_TRY();
	{
		ht = SearchSysCache(CLAAMNAMENSP,
								ObjectIdGetDatum(opcamid),
								PointerGetDatum(PyString_AS_STRING(str)),
								ObjectIdGetDatum(PyPgObject_FetchOid(self)),
								0);
		if (ht == NULL)
		{
			PyErr_Format(PyExc_LookupError,
				"failed to find operator class with AMID \"%d\" and "
				"name \"%s\"", opcamid, PyString_AS_STRING(str));
			rob = NULL;
		}
		else
		{
			HeapTupleHeader hth;
			hth = HeapTupleHeader_FromHeapTuple(ht);
			ReleaseSysCache(ht);
			ht = NULL;
			rob = PyPgObject_New(oct, PointerGetDatum(hth));
		}
		Py_DECREF(str);
		str = NULL;
	}
	PYPG_CATCH_END(if (ht) ReleaseSysCache(ht);; Py_XDECREF(str); return(NULL));

	return(rob);
}

static PyObj
ns_operator(PyObj self, PyObj args)
{
	static PyObj opert = NULL;
	HeapTuple ht = NULL;
	PyObj str, rob;
	Oid left = InvalidOid, right = InvalidOid;
	int16 ac = PyTuple_GET_SIZE(args);

	if (opert == NULL)
	{
		Oid to;
		PG_TRY();
		{
			Relation r;
#if PGV_MM == 80
			r = RelationSysNameGetRelation("pg_operator");
#else
			r = RelationIdGetRelation(OperatorRelationId);
#endif
			to = r->rd_rel->reltype;
			RelationClose(r);
		}
		PYPG_CATCH_END(return(NULL));

		opert = PyPgType_FromOid(to);
		if (opert == NULL) return(NULL);
	}

	if (ac < 2)
	{
		PyErr_Format(PyExc_TypeError,
			"namespace based operator lookup requires at least 2 arguments");
		return(NULL);
	}

	str = PyObject_Str(PyTuple_GET_ITEM(args, 0));
	left = Oid_FromPyObject(PyTuple_GET_ITEM(args, 1));
	if (ac > 2)
		right = Oid_FromPyObject(PyTuple_GET_ITEM(args, 2));

	PG_TRY();
	{
		HeapTupleHeader hth;
		ht = SearchSysCache(OPERNAMENSP,
								PointerGetDatum(PyString_AS_STRING(str)),
								ObjectIdGetDatum(left),
								ObjectIdGetDatum(right),
								ObjectIdGetDatum(PyPgObject_FetchOid(self)));
		Py_DECREF(str);
		str = NULL;
		hth = HeapTupleHeader_FromHeapTuple(ht);
		ReleaseSysCache(ht);
		ht = NULL;
		rob = PyPgObject_New(opert, PointerGetDatum(hth));
	}
	PYPG_CATCH_END(if (ht) ReleaseSysCache(ht);; Py_XDECREF(str); return(NULL));

	return(rob);
}

static PyObj
ns_function(PyObj self, PyObj args)
{
	PyObj str, rob;
	Oid oid = InvalidOid;
	int16 ac, ca;
#if PGV_MM == 80
	oidvector ov = {0,};

	ac = PyTuple_GET_SIZE(args);
	if (ac < 1)
	{
		PyErr_Format(PyExc_TypeError,
			"namespace based function lookup requires at least 1 argument");
		return(NULL);
	}

	str = PyObject_Str(PyTuple_GET_ITEM(args, 0));
	for (ca = 1; ca < ac; ++ca)
	{
		ov[ca-1] = Oid_FromPyObject(PyTuple_GET_ITEM(args, ca));
	}

	PG_TRY();
	{
		oid = GetSysCacheOid(PROCNAMENSP,
									PointerGetDatum(PyString_AS_STRING(str)),
									Int16GetDatum(ac - 1),
									PointerGetDatum(ov),
									ObjectIdGetDatum(PyPgObject_FetchOid(self)));
	}
	PYPG_CATCH_END(Py_DECREF(str); return(NULL));
	Py_DECREF(str);
#endif

	rob = PyPgFunction_FromOid(oid);
	return(rob);
}

static PyMethodDef PyPgNamespace_Methods[] = {
	{"Type", (PyCFunction) ns_type, METH_O,
		"fetch the type with the given name in the namespace"},
	{"Relation", (PyCFunction) ns_relation, METH_O,
		"fetch the pg_class with the given name in the namespace"},
	{"Conversion", (PyCFunction) ns_conversion, METH_O,
		"fetch the conversion with the given name in the namespace"},

	{"OperatorClass", (PyCFunction) ns_operatorclass, METH_VARARGS,
		"fetch the operator class with the given name in the namespace"},
	{"Operator", (PyCFunction) ns_operator, METH_VARARGS,
		"fetch the operator with the given name in the namespace"},
	{"Function", (PyCFunction) ns_function, METH_VARARGS,
		"fetch the function with the given name in the namespace"},
	/* XXX: Constraint? */
	{NULL}
};

static PyMemberDef PyPgNamespace_Members[] = {
	{NULL}
};

static PyObj
ns_repr(PyObj self)
{
	Form_pg_namespace ns = PyPgNamespace_FetchStructure(self);
	PyObj rob;

	rob = PyString_FromFormat("<%s %s>", self->ob_type->tp_name,
													NameStr(ns->nspname));
	return(rob);
}

static PyObj
ns_new_from_oid(PyTypeObject *subtype, Oid ns_oid)
{
	HeapTuple ht = NULL;
	HeapTupleHeader hth;
	PyObj no, pg_namespace_type, rob = NULL;

	pg_namespace_type = pg_namespace_PyPgType();
	if (pg_namespace_type == NULL)
		return(NULL);

	no = PyLong_FromUnsignedLong(ns_oid);
	if (PyMapping_HasKey(PyPgNamespace_Cache, no))
	{
		rob = PyObject_GetItem(PyPgNamespace_Cache, no);
		Py_DECREF(no);
		return(rob);
	}

	PG_TRY();
	{
		ht = SearchSysCache(NAMESPACEOID, ns_oid, 0, 0, 0);
		if (ht == NULL)
		{
			ereport(ERROR,(
				errmsg("namespace with Oid \"%d\" exists", ns_oid)
			));
		}

		hth = HeapTupleHeader_FromHeapTuple(ht);
		if (hth == NULL)
		{
			ereport(ERROR,(
				errmsg("failed to duplicate namespace into Python context")
			));
		}
	}
	PYPG_CATCH_END(if (ht) ReleaseSysCache(ht); goto fail);
	ReleaseSysCache(ht);

	rob = subtype->tp_alloc(subtype, 0);
	if (rob == NULL) goto fail;

	PyPgObject_FixType(rob, pg_namespace_type);
	PyPgObject_FixDatum(rob, PointerGetDatum(hth));

	if (PyObject_SetItem(PyPgNamespace_Cache, no, rob) < 0)
		goto fail;

	return(rob);

fail:
	Py_DECREF(no);
	XDECREF(rob);
	return(NULL);
}

static PyObj
ns_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	Oid ns_oid = InvalidOid;
	PyObj no;
	PyObj rob;

	no = PySequence_GetItem(args, 0);
	if (no == NULL)
	{
		PyErr_SetString(PyExc_TypeError,
			"Postgres.Namespace requires at least one argument");
		return(NULL);
	}

	ns_oid = Oid_FromPyObject(no);
	if (ns_oid == InvalidOid)
	{
		if (!PyErr_Occurred())
		{
			PyErr_Format(PyExc_TypeError,
				"unable to fetch Oid from '%s' object",
				no->ob_type->tp_name);
		}

		return(NULL);
	}

	rob = ns_new_from_oid(subtype, ns_oid);
	return(rob);
}

const char doc[] =
"Python interface type to a Postgres namespaces";

PyTypeObject PyPgNamespace_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Namespace",			/* tp_name */
	sizeof(struct PyPgNamespace),	/* tp_basicsize */
	0,										/* tp_itemsize */
	NULL,									/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	ns_repr,								/* tp_repr */
	NULL,									/* tp_as_number */
	NULL,									/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	NULL,									/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) doc,						/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	NULL,									/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgNamespace_Methods,			/* tp_methods */
	PyPgNamespace_Members,			/* tp_members */
	NULL,									/* tp_getset */
	&PyPgObject_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	ns_new,								/* tp_new */
};

PyObj
PyPgNamespace_FromOid(Oid po)
{
	return(ns_new_from_oid(&PyPgNamespace_Type, po));
}

PyObj
pg_namespace_PyPgType(void)
{
	static PyObj rob = NULL;
	if (rob == NULL)
	{
		Relation r;
#if PGV_MM == 80
		r = RelationSysNameGetRelation("pg_namespace");
#else
		r = RelationIdGetRelation(NamespaceRelationId);
#endif
		rob = PyPgType_FromOid(r->rd_rel->reltype);
		RelationClose(r);
	}

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
