/* $Id: pl.c,v 1.10 2005/08/10 18:52:20 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * be/src/plcall.c,v 1.1.1.1 2004/12/16 00:32:18 flaw
 * imp/src/plcall.c,v 1.6 2004/12/15 13:05:14 flaw
 *//*
 * Implements the Python interface to a procedural call
 */
#include <setjmp.h>
#include <postgres.h>
#include <miscadmin.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <commands/trigger.h>
#include <executor/executor.h>
#include <nodes/params.h>
#include <nodes/execnodes.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <compile.h>
#include <eval.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/error.h>
#include <pypg/utils.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/function.h>
#include <pypg/conv.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/trigger.h>

int
PyPgProceduralCall_InitializeParameters(PyObj self)
{
	FunctionCallInfo fcinfo = PyPgProceduralCall_FetchFCInfo(self);
	PyObj func = PyPgCall_FetchFunction(self);
	PyObj argtypes = PyPgFunction_FetchArgTypes(func);
	PyObj args = PyPgCall_FetchArguments(self);
	PyObj kw = PyPgCall_FetchKeywords(self);

	if (kw == NULL)
	{
		kw = PyDict_New();
		if (kw == NULL)
			return(-1);
	}

	if (args == NULL)
	{
		args = PyList_New(fcinfo->nargs);
		if (args == NULL) goto error;
	}

	args = PyPgFill_PySequence(args, argtypes, fcinfo);
	if (args == NULL) goto error;

	if (fcinfo->nargs > 0)
		PyPgCall_FixateKeywords(kw, args, PyPgFunction_FetchArgNames(func));

	PyPgCall_FixArguments(self, args);
	PyPgCall_FixKeywords(self, kw);

	return(0);
error:
	DECREF(kw);
	return(-1);
}

static PyObj
materialize(PyObj self, PyObj iter)
{
	ReturnSetInfo *rsi = PyPgProceduralCall_FetchRSI(self);
	MemoryContext former, mmc = rsi->econtext->ecxt_per_query_memory;

	former = MemoryContextSwitchTo(mmc);
	PG_TRY();
	{
		HT_FromTDAndPyObj_Con cf = NULL;
		Tuplestorestate *tss;
		TupleDesc td = NULL;
		HeapTuple ht;
		PyObj ob;

		tss = tuplestore_begin_heap(true, true, work_mem);
		rsi->setResult = tss;

		ob = PyIter_Next(iter);
 		if (ob == NULL)
		{
			/*
			 * This should fall through the rest of the function giving
			 * an empty tuple store, as desired. If not, it will crash by
			 * calling a NULL function pointer.
			 */
		}
		else if (PyPgHeapTuple_Check(ob))
		{
			PyObj ptd = PyPgHeapTuple_FetchPyPgTupleDesc(ob);

			if (ptd == PyPgCall_FetchExpected(self))
				td = rsi->expectedDesc;
			else
				td = PyPgTupleDesc_FetchTupleDesc(ptd);
			cf = HeapTuple_FromTupleDescAndIterable;
		}
		else
		{
			PyObj rt = PyPgCall_FetchReturnType(self);
			if (PyPgType_IsComposite(rt))
				cf = HeapTuple_FromTupleDescAndIterable;
			else
				cf = HeapTuple_FromTupleDescAndPyObject;
			td = PyPgType_FetchTupleDesc(rt);
		}

		if (td != rsi->expectedDesc)
			rsi->setDesc = CreateTupleDescCopyConstr(td);

		if (ob != NULL) do
		{
			ht = cf(td, ob);
			DECREF(ob);
			tuplestore_puttuple(tss, (void *) ht);
			heap_freetuple(ht);
		} while ((ob = PyIter_Next(iter)) != NULL);
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();
	MemoryContextSwitchTo(former);

	if (PyErr_Occurred())
	{
		rsi->setResult = NULL;
		return(NULL);
	}

	RETURN_NONE;
}

static PyObj
srf_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob = NULL;
	ReturnSetInfo *rsi = PyPgProceduralCall_FetchRSI(self);
	PyObj iter = PyPgCall_FetchReturned(self);

	if (iter == Py_None)
	{
		rob = self->ob_type->tp_base->tp_call(self, args, kw);
		if (rob == NULL)
			return(NULL);

		iter = PyObject_GetIter(rob);
		DECREF(rob);
		if (iter == NULL)
		{
			PyErr_Format(PyExc_TypeError, "expected iterable from SRF");
			return(NULL);
		}
		DECREF(PyPgCall_FetchReturned(self));
		PyPgCall_FixReturned(self, iter);
	}

	if (rsi->allowedModes & SFRM_Materialize)
	{
		rsi->returnMode = SFRM_Materialize;
		rsi->isDone = ExprSingleResult;
		rob = materialize(self, iter);
	}
	else
	{
		rsi->returnMode = SFRM_ValuePerCall;
		rob = PyIter_Next(iter);
		if (rob == NULL)
		{
			rsi->isDone = ExprEndResult;
			rob = Py_None;
			INCREF(rob);
		}
		else
			rsi->isDone = ExprMultipleResult;
	}

	return(rob);
}

static PyObj
plcall_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob = NULL;
	Node *ri;
	ReturnSetInfo *rsi;

	if (!PyArg_ParseTuple(args, "")) return(NULL);

	ri = PyPgProceduralCall_FetchResultInfo(self);
	rsi = ri && IsA(ri, ReturnSetInfo) ? (ReturnSetInfo *) ri : NULL;

	if (rsi)
		rob = srf_call(self, args, kw);
	else
	{
		rob = self->ob_type->tp_base->tp_call(self, args, kw);
		if (rob != NULL)
		{
			if (PyIter_Check(rob))
			{
				PyObj iter = rob;
				rob = PyIter_Next(iter);
				DECREF(iter);
				if (rob == NULL && !PyErr_Occurred())
				{
					rob = Py_None;
					Py_INCREF(Py_None);
				}
			}
		}
	}

	if (rob != NULL && rob != Py_None && !PyPgObject_Check(rob))
	{
		PyObj expect = PyPgCall_FetchExpected(self);
		PyObj ob = rob;
		if (PyPgType_IsComposite(expect) || PyPgType_IsArray(expect))
			rob = PyObject_Call(expect, ob, NULL);
		else
			rob = Py_Call(expect, ob);
		DECREF(ob);
	}

	return(rob);
}

const char PyPgProceduralCall_Doc[] =
"Python interface to a Postgres trigger fire";

PyTypeObject PyPgProceduralCall_Type = {
	PyObject_HEAD_INIT(NULL)
	0,												/* ob_size */
	"Postgres.ProceduralCall",				/* tp_name */
	sizeof(struct PyPgProceduralCall),	/* tp_basicsize */
	0,												/* tp_itemsize */
	NULL,											/* tp_dealloc */
	NULL,											/* tp_print */
	NULL,											/* tp_getattr */
	NULL,											/* tp_setattr */
	NULL,											/* tp_compare */
	NULL,											/* tp_repr */
	NULL,											/* tp_as_number */
	NULL,											/* tp_as_sequence */
	NULL,											/* tp_as_mapping */
	NULL,											/* tp_hash */
	plcall_call,								/* tp_call */
	NULL,											/* tp_str */
	NULL,											/* tp_getattro */
	NULL,											/* tp_setattro */
	NULL,											/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,						/* tp_flags */
	(char *) PyPgProceduralCall_Doc,		/* tp_doc */
	NULL,											/* tp_traverse */
	NULL,											/* tp_clear */
	NULL,											/* tp_richcompare */
	0,												/* tp_weaklistoffset */
	NULL,											/* tp_iter */
	NULL,											/* tp_iternext */
	NULL,											/* tp_methods */
	NULL,											/* tp_members */
	NULL,											/* tp_getset */
	&PyPgCall_Type,							/* tp_base */
	NULL,											/* tp_dict */
	NULL,											/* tp_descr_get */
	NULL,											/* tp_descr_set */
	0,												/* tp_dictoffset */
	NULL,											/* tp_init */
	NULL,											/* tp_alloc */
	NULL,											/* tp_new */
};

PyObj
PyPgProceduralCall_Initialize(PyObj self, FunctionCallInfo fcinfo)
{
	PyObj func, xtd;
	ReturnSetInfo *rsi;
	if (self == NULL) return(NULL);

	Assert(fcinfo != NULL);
	PyPgProceduralCall_FixFCInfo(self, fcinfo);
	rsi = PyPgProceduralCall_FetchRSI(self);

	func = PyPgFunction_FromOid(fcinfo->flinfo->fn_oid);
	if (func == NULL) goto free_self;
	PyPgCall_FixFunction(self, func);
	xtd = PyPgFunction_FetchReturnType(func);

	if (rsi)
	{
		if (!(rsi->allowedModes & (SFRM_ValuePerCall|SFRM_Materialize)))
		{
			PyErr_Format(PyExc_TypeError,
				"only VPC and materialization are known");
			goto DECREF_func;
		}

		if (rsi->expectedDesc)
		{
			xtd = PyPgTupleDesc_New(rsi->expectedDesc);
		}
	}
	else
		INCREF(xtd);
	PyPgCall_FixExpected(self, xtd);
	PyPgCall_FixReturned(self, Py_None);

	if (PyPgProceduralCall_InitializeParameters(self) < 0) goto DECREF_func;
	INCREF(Py_None);

	return(self);
DECREF_func:
	DECREF(func);
free_self:
	self->ob_type->tp_free(self);
	return(NULL);
}
/*
 * vim: ts=3:sw=3:noet:
 */
