/* $Id: transaction.c,v 1.7 2005/05/12 19:29:35 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * be/src/tb.c,v 1.2 2005/02/05 20:54:30 flaw
 * imp/src/tb.c,v 1.1 2004/12/15 13:10:01 flaw
 *//*
 * The Postgres Transaction interface
 */
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <executor/tstoreReceiver.h>
#include <nodes/parsenodes.h>
#include <utils/array.h>
#include <utils/guc.h>
#include <utils/palloc.h>
#include <utils/relcache.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/transaction.h>
#include <pypg/savepoint.h>
#include <pypg/error.h>

static char *
statestr(PgTBState tbs)
{
	switch (tbs)
	{
#define CASE(STATE) \
case PgTBS_##STATE: return(#STATE); break;
		CASE(INITIALIZING)
		CASE(RUNNING)
		CASE(COMMITTING)
		CASE(COMMITTED)
		CASE(ABORTING)
		CASE(ABORTED)
		CASE(INVALID)
#undef CASE
	}

	return("<unknown>");
}

static PyMemberDef PyPgTransaction_Members[] = {
	{"Savepoint", T_OBJECT, offsetof(struct PyPgTransaction, xa_sptp), RO,
	"Savepoint type for transaction"},
	{NULL}
};

PyObj
commit(PyObj self)
{
	if (PyPgTransaction_FetchState(self) != PgTBS_RUNNING)
		RETURN_NONE;

	PyPgTransaction_FixState(self, PgTBS_COMMITTING);
	PG_TRY();
	{
		if (EndTransactionBlock())
			PyPgTransaction_FixState(self, PgTBS_COMMITTED);
		else
			PyPgTransaction_FixState(self, PgTBS_ABORTED);

		CommitTransactionCommand();
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();

	RETURN_NONE_OR_NULL;
}

PyObj
rollback(PyObj self)
{
	if (PyPgTransaction_FetchState(self) != PgTBS_RUNNING)
		RETURN_NONE;

	PyPgTransaction_FixState(self, PgTBS_ABORTING);
	PG_TRY();
	{
		UserAbortTransactionBlock();
		PyPgTransaction_FixState(self, PgTBS_ABORTED);
		CommitTransactionCommand();
	}
	PG_CATCH();
	{
		PyErr_SetPgError();
	}
	PG_END_TRY();

	RETURN_NONE_OR_NULL;
}

static PyMethodDef PyPgTransaction_Methods[] = {
	{"Commit", (PyCFunction) commit, METH_NOARGS,
	"commit the transaction"},
	{"Abort", (PyCFunction) rollback, METH_NOARGS,
	"abort the transaction"},
	{NULL}
};

static void
tb_dealloc(PyObj self)
{
	if (PyPgTransaction_FetchState(self) == PgTBS_RUNNING)
	{
		Py_XDECREF(rollback(self));
	}
	Py_XDECREF(PyPgTransaction_FetchSavepoints(self));
	self->ob_type->tp_free(self);
}

static PyObj
tb_repr(PyObj self)
{
	PyObj rob;
	rob = PyString_FromFormat("<Postgres.Transaction '%s' at %p>",
		statestr(PyPgTransaction_FetchState(self)), self);
	return(rob);
}

static PyObj
tb_str(PyObj self)
{
	PyObj rob;
	rob = PyString_FromString(statestr(PyPgTransaction_FetchState(self)));
	return(rob);
}

static int
tb_init(PyObj self, PyObj args, PyObj kw)
{
	static char *kwlist[] = {
		"isolation", "readonly"
	};
	char *isolation = NULL;
	PyObj ro = NULL;
	PyPgTransaction_FixState(self, PgTBS_INITIALIZING);

	if (!PyArg_ParseTupleAndKeywords(args, kw, "|sO", kwlist, &isolation, &ro))
		return(-1);

	if (IsTransactionBlock())
	{
		if (IsAbortedTransactionBlockState())
			PyPgTransaction_FixState(self, PgTBS_ABORTED);
		else
			PyPgTransaction_FixState(self, PgTBS_RUNNING);
	}
	else
	{
		PG_TRY();
		{
			BeginTransactionBlock();
			if (isolation)
			{
				List *l;
				l = list_make1(isolation);
				SetPGVariable("transaction_isolation", l, false);
				list_free(l);
			}
			if (ro && ro == Py_True)
			{
				List *l;
				l = list_make1("read only");
				SetPGVariable("transaction_read_only", l, false);
				list_free(l);
			}
			CommitTransactionCommand();
		}
		PG_CATCH();
		{
			PyErr_SetPgError();
		}
		PG_END_TRY();

		if (PyErr_Occurred()) return(-1);

		PyPgTransaction_FixState(self, PgTBS_RUNNING);
	}

	{
		PyObj t, name, bases, d, tp;
		t = PyTuple_New(3);

		name = PyString_FromString("Transaction.Savepoint");
		bases = PyTuple_New(1);
		PyTuple_SET_ITEM(bases, 0, (PyObj) &PyPgSavepoint_Type);
		d = PyDict_New();
		PyDict_SetItemString(d, "__xact__", self);

		PyTuple_SET_ITEM(t, 0, name);
		PyTuple_SET_ITEM(t, 1, bases);
		PyTuple_SET_ITEM(t, 2, d);

		tp = PyObject_Call((PyObj) &PyType_Type, t, NULL);
		Py_DECREF(t);
		PyPgTransaction_FixSavepointType(self, tp);
	}

	return(0);
}

static const char doc[] =
"Python interface to the Postgres transaction";

PyTypeObject PyPgTransaction_Type = {
	PyObject_HEAD_INIT(NULL)
	0,													/* ob_size */
	"Postgres.Transaction",						/* tp_name */
	sizeof(struct PyPgTransaction),	/* tp_basicsize */
	0,													/* tp_itemsize */
	(destructor)tb_dealloc,						/* tp_dealloc */
	NULL,												/* tp_print */
	(getattrfunc)NULL,							/* tp_getattr */
	(setattrfunc)NULL,							/* tp_setattr */
	(cmpfunc)NULL,									/* tp_compare */
	(reprfunc)tb_repr,							/* tp_repr */
	NULL,												/* tp_as_number */
	NULL,												/* tp_as_sequence */
	NULL,												/* tp_as_mapping */
	(hashfunc)NULL,								/* tp_hash */
	(ternaryfunc)NULL,							/* tp_call */
	(reprfunc)tb_str,								/* tp_str */
	NULL,												/* tp_getattro */
	NULL,												/* tp_setattro */
	NULL,												/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT |
	Py_TPFLAGS_BASETYPE,							/* tp_flags */
	(char*)doc,										/* tp_doc */
	(traverseproc)NULL,							/* tp_traverse */
	(inquiry)NULL,									/* tp_clear */
	(richcmpfunc)NULL,							/* tp_richcompare */
	(long)0,											/* tp_weaklistoffset */
	(getiterfunc)NULL,							/* tp_iter */
	(iternextfunc)NULL,							/* tp_iternext */
	PyPgTransaction_Methods,					/* tp_methods */
	PyPgTransaction_Members,					/* tp_members */
	NULL,												/* tp_getset */
	NULL,												/* tp_base */
	NULL,												/* tp_dict */
	NULL,												/* tp_descr_get */
	NULL,												/* tp_descr_set */
	0,													/* tp_dictoffset */
	tb_init,											/* tp_init */
	NULL,												/* tp_alloc */
	PyType_GenericNew,							/* tp_new */
};
/*
 * vim: ts=3:sw=3:noet:
 */
