/* $Id: pl.c,v 1.21 2005/06/30 01:22:33 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/pl.c,v 1.11 2004/12/15 05:14:20 flaw
 * pl/src/pl.c,v 1.17 2004/08/06 08:27:14 flaw
 * pl/src/pl.h,v 1.11 2004/08/06 08:27:14 flaw
 * pl/src/plpy.c,v 1.37 2004/09/29 01:31:19 flaw
 *//*
 * Postgres procedural language interfaces
 */
#include <setjmp.h>

#include <Python.h>
#include <compile.h>
#include <marshal.h>
#include <structmember.h>
#include <cStringIO.h>

#include <postgres.h>
#include <fmgr.h>
#include <funcapi.h>
#include <miscadmin.h>
#include <access/htup.h>
#include <access/heapam.h>
#include <access/xact.h>
#include <catalog/pg_class.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_type.h>
#include <catalog/pg_language.h>
#include <catalog/indexing.h>
#include <commands/trigger.h>
#include <mb/pg_wchar.h>
#include <nodes/memnodes.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/builtins.h>
#include <utils/hsearch.h>
#include <utils/syscache.h>
#include <utils/relcache.h>

#include <pypg/python.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/function.h>
#include <pypg/error.h>
#include <pypg/call.h>
#include <pypg/call/pl.h>
#include <pypg/call/trigger.h>

#include <pypg/ci.h>
PyPgCI_Define();

#ifndef IF_PATH
#error IF_PATH must be defined to the module of the version specific \
package space
#endif

#ifndef PGV_TAG
#error PGV_TAG must be defined to the version of PostgreSQL that the PL \
is being built against, e.g. 80beta
#endif

#ifndef EX_IFM
#error EX_IFM must be defined to the specific module to be imported
#endif

#ifndef PL_STATE
#error PL_STATE must be defined
#endif

#define DEFAULT_POSTGRES_MODULE IF_PATH"."PL_STATE"-v"PGV_TAG"."EX_IFM
/*
 * indent
 *
 * Reallocate src with a volume of
 *		len(src) + fpad + lpad + (\n_count(src) * infact)
 * Start to copy src at msrc + fpad address
 * And set 'infact' number of characters following a \n to \t
 */
static char *
indent
(
	const char *const src,
	const unsigned short fpad,
	const unsigned short lpad,
	const unsigned short infact
)
{
	char *nsrc = NULL;
	register const char *frame = NULL;
	register char *msrc = NULL;
	register size_t sv, mv = 1 * infact;

	/*
	 * Existing Newline count, and source length.
	 */
	for (frame = src; *frame != '\0'; ++frame)
		if (*frame == '\n')
			mv += infact;

	mv += (sv = frame - src);
	nsrc = msrc = malloc(mv+fpad+lpad);

	msrc += fpad;

	/*
	 * Copy src to msrc with a new tab(s) following each newline.
	 */
	*msrc++ = '\t';
	for (frame = src; *frame != '\0'; ++frame)
	{
		*msrc = *frame;

		if (*(msrc++) == '\n')
			for (sv = infact; sv > 0; --sv, ++msrc)
				*msrc = '\t';
	}
	*msrc = '\0';

	return(nsrc);
}

/*
 * transform - make a code fragment a real live function
 */
#define FUNCENC "# -*- encoding: utf-8 -*-\n"
#define FUNCNAME "PGFunction"
#define FUNCDEF FUNCENC "def " FUNCNAME "(self, args, kw):"
#define FUNCDEFLEN sizeof(FUNCDEF)
static char *
transform(char *src)
{
	char *msrc;
	msrc = indent(src, FUNCDEFLEN, 0, 1);
	if (msrc == NULL) return(NULL);

	memcpy(msrc, FUNCDEF, FUNCDEFLEN);
	msrc[FUNCDEFLEN-1] = '\n';
	{
		register unsigned int msrcl = strlen(msrc)-1;
		if (msrc[msrcl] == '\t')
			msrc[msrcl] = '\n';
	}
	return(msrc);
}

/*
 * linecache_update - handle pg_proc oids
 */
static PyObj lc_updatecache;
static PyObj lc_cache;
static PyObj
linecache_update(PyObj self, PyObj args)
{
	PyObj rob = NULL;
	char *filename;
	Oid fn_oid;

	if (PyArg_ParseTuple(args, "s", &filename) < 0)
		return(NULL);

	fn_oid = DatumGetObjectId(strtol(filename, NULL, 10));
	if (fn_oid != InvalidOid)
	{
		PyObj src = NULL;

		if (PyMapping_HasKeyString(lc_cache, filename))
			PyDict_DelItemString(lc_cache, filename);

		PG_TRY();
		{
			Relation proc;
			HeapTuple ht;
			Datum srctext;
			char *srctxt, *msrc = NULL;
			bool isnull = false;

			proc = RelationIdGetRelation(ProcedureRelationId);
			if (!RelationIsValid(proc))
				ereport(ERROR, (
					errmsg("failed to get pg_proc Relation"),
					errdetail("RelationIdGetRelation(ProcedureRelationId = %d)",
						ProcedureRelationId)
				));

			ht = SearchSysCache(PROCOID, ObjectIdGetDatum(fn_oid), 0, 0, 0);
			if (ht == NULL)
			{
				ereport(ERROR, (
					errmsg("failed to get proc entry with oid \"%d\"", fn_oid)
				));
			}
			srctext = fastgetattr(ht, Anum_pg_proc_prosrc,
							RelationGetDescr(proc), &isnull
						);
			RelationClose(proc);

			srctxt = (char *) DirectFunctionCall1(textout, srctext);
			msrc = transform(srctxt);
			pfree(srctxt);
			if (msrc == NULL)
			{
				ereport(ERROR, (
					errmsg("failed to transform source for traceback")
				));
			}
			src = PyString_FromString(msrc);
			free(msrc);
			ReleaseSysCache(ht);
		}
		PG_CATCH();
		{
			PyErr_SetString(PyExc_IOError, "couldn't get proc source");
			return(NULL);
		}
		PG_END_TRY();

		if (src != NULL)
		{
			char *fn = "PostgreSQL Function";
			rob = PyObject_CallMethod(src, "split", "s", "\n");
			if (rob != NULL)
			{
				int size = PyString_GET_SIZE(src);
				PyObj cache;
				cache = Py_BuildValue("(iiOs)", size, 0, rob, fn);
				if (cache != NULL)
					PyDict_SetItemString(lc_cache, filename, cache);
			}
			Py_DECREF(src);
		}
		else
		{
			PyErr_Clear();
			return(PyList_New(0));
		}
	}
	else
	{
		rob = PyObject_Call(lc_updatecache, args, NULL);
	}

	return(rob);
}
static PyMethodDef linecache_open_def = {
	"pg_linecache_update",
	(PyCFunction) linecache_update,
	METH_VARARGS|METH_KEYWORDS,
	"custom updater allow linecache to see PostgreSQL pg_proc files"
};

/*
 * tbtostr - Python Traceback To String
 */
static char *
tbtostr(PyObj tb)
{
	static PyObj tb_print_tb = NULL;

	PyObj rob;
	/*
	 * WARNING: Must be kept in sync with python:Modules/cStringIO.c
	 */
	struct {
		PyObject_HEAD
		char *buf;
		int pos, string_size;
		int buf_size, softspace;
	} io = {0,};

	Assert(tb != NULL && PyTraceBack_Check(tb));

	if (tb_print_tb == NULL)
	{
		PyObj mod_traceback = PyImport_ImportModule("traceback");
		if (mod_traceback == NULL) return(NULL);

		tb_print_tb = PyObject_GetAttrString(mod_traceback, "print_tb");
		Py_DECREF(mod_traceback);
		if (!tb_print_tb) return(NULL);
	}

	io.ob_refcnt = 1;
	io.ob_type = PycStringIO->OutputType;
	io.buf = malloc(128);
	io.buf_size = 128;
	rob = Py_Call(tb_print_tb, tb, Py_None, &io);
	Py_XDECREF(rob);
	PycStringIO->cwrite((PyObj) &io, "\0", 1);

	if (PyErr_Occurred()) PyErr_Clear();
	return(io.buf);
}

/*
 * errorstr
 *
 *	This function constructs an error string that correlates with
 * the current python exception.
 */
static char *
errorstr(void)
{
	char *vstr = "NULL Exception Instance",
		  *estr = "NULL Exception",
		  *tbstr = NULL;
	PyObj e = NULL, v = NULL, tb = NULL;
	PyObj epystr = NULL, vpystr = NULL;

	char *errstr = NULL, *perrstr = NULL;

	Assert(PyErr_Occurred() != NULL);

	PyErr_Fetch(&e, &v, &tb);
	PyErr_NormalizeException(&e, &v, &tb);

	if (e != NULL)
	{
		epystr = PyObject_Str(e);
		Py_DECREF(e);
		if (epystr)
			estr = PyString_AS_STRING(epystr);
	}

	if (v != NULL)
	{
		vpystr = PyObject_Str(v);
		Py_DECREF(v);
		if (vpystr)
			vstr = PyString_AS_STRING(vpystr);
	}

	if (tb != NULL)
	{
		tbstr = tbtostr(tb);
		Py_DECREF(tb);
	}

	asprintf(&errstr, "%s%s%s: %s", tbstr?"(most recent call last):\n":"",
			tbstr?tbstr:"", estr, vstr);
	perrstr = pstrdup(errstr);
	if (tbstr) free(tbstr);

	Py_XDECREF(epystr);
	Py_XDECREF(vpystr);

	return(perrstr);
}
#define PythonExceptionDetail errdetail(errorstr())

unsigned int ist_count;
static PyObj
ist_begin(PyObj self)
{
	PG_TRY();
	{
		if (ist_count == -1)
		{
			ereport(ERROR,(
				errcode(ERRCODE_SAVEPOINT_EXCEPTION),
				errmsg("internal subtransactions prohibited"),
				errhint("Attempt to begin an IST in a prohibited area.")
			));
		}
		BeginInternalSubTransaction(NULL);
	}
	PYPG_CATCH_END(return(NULL));

	ist_count += 1;
	RETURN_NONE;
}
static PyMethodDef ist_begin_def = {
	"pg_ist_begin",
	(PyCFunction) ist_begin,
	METH_NOARGS,
	"begin an internal subtransaction"
};

static PyObj
ist_release(PyObj self)
{
	PG_TRY();
	{
		if (ist_count == 0 || ist_count == -1)
		{
			ereport(ERROR,(
				errcode(ERRCODE_SAVEPOINT_EXCEPTION),
				errmsg("no current internal subtransaction"),
				errhint("Attempt to release the current IST, when none running.")
			));
		}
		ReleaseCurrentSubTransaction();
	}
	PYPG_CATCH_END(return(NULL));

	ist_count -= 1;
	RETURN_NONE;
}
static PyMethodDef ist_release_def = {
	"pg_ist_release",
	(PyCFunction) ist_release,
	METH_NOARGS,
	"release the current internal subtransaction"
};

static PyObj
ist_rollback(PyObj self)
{
	PG_TRY();
	{
		if (ist_count == 0)
		{
			ereport(ERROR,(
				errcode(ERRCODE_SAVEPOINT_EXCEPTION),
				errmsg("no current internal subtransaction"),
				errhint("Attempt to rollback the current IST, when none running.")
			));
		}
		RollbackAndReleaseCurrentSubTransaction();
	}
	PYPG_CATCH_END(return(NULL));

	ist_count -= 1;
	RETURN_NONE;
}
static PyMethodDef ist_rollback_def = {
	"pg_ist_rollback",
	(PyCFunction) ist_rollback,
	METH_NOARGS,
	"rollback and release the current internal subtransaction"
};

static PyObj fnExtraCalls = NULL;
static void
srfeccb(Datum arg)
{
	int ind;
	FmgrInfo *fli = (FmgrInfo *) DatumGetPointer(arg);
	PyObj plc = (PyObj) fli->fn_extra;

	ind = PySequence_Index(fnExtraCalls, plc);
	PySequence_DelItem(fnExtraCalls, ind);

	fli->fn_extra = NULL;
}

PyObj XactDict;
static void
XactHook(XactEvent xev, void *arg)
{
	while (PyList_GET_SIZE(fnExtraCalls) > 0)
		PySequence_DelItem(fnExtraCalls, 0);
	ist_count = 0;

	PyDict_Clear(XactDict);
}

/*
 * Invoked by PyPgFunction through the PyPgCI.
 */
static int
CodeFixer(PyObj self)
{
	HeapTupleHeader hth = PyPgObject_FetchHeapTupleHeader(self);
	HeapTupleData htd = 
		{VARSIZE(hth)-VARHDRSZ, {{0,},0}, 0, TopMemoryContext, hth};
	TupleDesc td;
	bytea *data;
	PyObj code;
	bool isnull;

	td = PyPgObject_FetchTupleDesc(self);
	data = (bytea *) fastgetattr(&htd, Anum_pg_proc_probin, td, &isnull);
	if (isnull)
	{
		PyErr_SetString(PyExc_TypeError, "NULL probin for code");
		return(-1);
	}
	code = PyMarshal_ReadObjectFromString(VARDATA(data), VARSIZE(data)-VARHDRSZ);
	if (code == NULL) return(-1);

	PyPgFunction_FixCode(self, code);
	return(0);
}

/*
 * Use the library's on-load facility to initialize Python as soon as it's
 * loaded. Providing a warranty that Python is initialized from any point.
 * Allowing ignorance of the Python initialization state within the initializor
 * and the validator, the only entry points.
 */
static PyObj Module = NULL;
#ifdef __APPLE__
__attribute__((constructor))
#endif
void
_init(void)
{
	MemoryContext former;
	PyObj ob;

	START_CRIT_SECTION();

	Py_SetProgramName("Postgres");
	Py_Initialize();

	PycString_IMPORT;
	XactDict = PyDict_New();

	former = MemoryContextSwitchTo(TopMemoryContext);
	Module = PyImport_ImportModule(DEFAULT_POSTGRES_MODULE);
	if (Module == NULL)
		ereport(ERROR,(
			errmsg("failed to import Postgres interface module \""
				DEFAULT_POSTGRES_MODULE"\""),
			PythonExceptionDetail
		));
	MemoryContextSwitchTo(former);

	ob = PyObject_GetAttrString(Module, PyPgCI_CObjectName);
	if (ob == NULL)
		ereport(ERROR,(
			errmsg("failed to get C Interface from extension module"),
			PythonExceptionDetail
		));
	PyPgCI_Initialize(PyCObject_AsVoidPtr(ob));
	Py_DECREF(ob);
	PyPgCI.PL.Fixer = CodeFixer;

	if (PyUnicode_SetDefaultEncoding(
				PyEncoding_FromPgEncoding(GetDatabaseEncoding())) < 0)
		ereport(ERROR,(
			errmsg("failed to set default encoding"),
			PythonExceptionDetail
		));

	ob = PyImport_ImportModule("linecache");
	if (ob != NULL)
	{
		PyObj lc_open;
		lc_updatecache = PyObject_GetAttrString(ob, "updatecache");
		lc_cache = PyObject_GetAttrString(ob, "cache");
		lc_open = PyCFunction_New(&linecache_open_def, ob);
		PyObject_SetAttrString(ob, "updatecache", lc_open);
		Py_DECREF(ob);
		Py_DECREF(lc_open);
	}
	else
		PyErr_Clear();

	/*
	 * The Transaction and Savepoint types are a direct interface to PostgreSQL
	 * transaction management, therefore it is not appropriate in the context of
	 * a function call.
	 */
	PyObject_DelAttrString(Module, "Transaction");
	PyObject_DelAttrString(Module, "Savepoint");

	ob = PyImport_AddModule("__main__");
	if (ob == NULL)
	{
		ereport(ERROR, (
			errmsg("failed to get __main__ dictionary")
		));
	}
	else
	{
		PyObj item;
		ob = PyModule_GetDict(ob);
		PyDict_SetItemString(ob, "Postgres", Module);

		item = PyCFunction_New(&ist_begin_def, NULL);
		PyDict_SetItemString(ob, "BEGIN", item);

		item = PyCFunction_New(&ist_release_def, NULL);
		PyDict_SetItemString(ob, "RELEASE", item);

		item = PyCFunction_New(&ist_rollback_def, NULL);
		PyDict_SetItemString(ob, "ROLLBACK", item);

		Py_INCREF(XactDict);
		PyDict_SetItemString(ob, "XD", XactDict);
	}

	fnExtraCalls = PyList_New(0);
	if (fnExtraCalls == NULL)
	{
		ereport(ERROR, (
			errmsg("failed to create fnExtraCalls list")
		));
	}
	RegisterXactCallback(XactHook, NULL);

	END_CRIT_SECTION();
}
#if defined(_WIN32)
BOOL WINAPI
DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
	switch (fdwReason) {
		case DLL_PROCESS_ATTACH:
			_init();
		break;
	}
	return(TRUE);
}
#endif

static bytea *
mkbytea(const char *buf, Size len)
{
	bytea *ba;
	Size ba_len = 0;

	ba_len = len + VARHDRSZ;
	ba = (bytea *) palloc(ba_len);

	VARATT_SIZEP(ba) = ba_len;
	memcpy(VARDATA(ba), buf, len);

	return(ba);
}

static HeapTuple
pg_proc_probin(HeapTuple procTuple, Datum probin)
{
	Relation rel;
	HeapTuple tup;

	Datum	values[Natts_pg_proc] = {0};
	char nulls[Natts_pg_proc];
	char replace[Natts_pg_proc];

	memset(replace, ' ', Natts_pg_proc);
	memset(nulls, ' ', Natts_pg_proc);

	replace[Anum_pg_proc_probin-1] = 'r';
	values[Anum_pg_proc_probin-1] = probin;

	rel = relation_open(ProcedureRelationId, RowExclusiveLock);
	if (!RelationIsValid(rel))
		ereport(ERROR,(
			errmsg("failed to open pg_proc System relation")
		));
#if PGV_MM == 80
	tup = heap_modifytuple(procTuple, rel, values, nulls, replace);
#else
	tup = heap_modifytuple(procTuple,
				RelationGetDescr(rel), values, nulls, replace);
#endif
	if (tup == NULL)
		ereport(ERROR,(
			errmsg("failed to modify procedure's tuple")
		));

	simple_heap_update(rel, &tup->t_self, tup);
	CatalogUpdateIndexes(rel, tup);

	heap_close(rel, RowExclusiveLock);
	return(tup);
}

/*
 * compile - make a code object from prosrc
 */
static PyObj
compile(char *src)
{
	PyCompilerFlags flags = {CO_OPTIMIZED};
	PyObj locals = NULL;
	PyObj rob = NULL;
	char *msrc = NULL;

	msrc = transform(src);
	if (msrc == NULL) return(NULL);

	locals = PyDict_New();
	if (locals != NULL)
		PyRun_StringFlags(msrc, Py_file_input, locals, locals, &flags);

	free(msrc);

	if (locals && !PyErr_Occurred())
	{
		rob = PyDict_GetItemString(locals, FUNCNAME);
		rob = PyFunction_GET_CODE(rob);
		INCREF(rob);
		DECREF(locals);
	}

	return(rob);
}

static void
PostgresError_FromPythonException(void)
{
	PyObj ob, e, v, tb;

	PyErr_Fetch(&e, &v, &tb);
	PyErr_NormalizeException(&e, &v, &tb);
	errstart(ERROR, __FILE__, __LINE__, PG_FUNCNAME_MACRO);

	ob = PyObject_GetAttrString(v, "code");
	if (PyInt_Check(ob))
		errcode(PyInt_AS_LONG(ob));
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "message");
	if (PyString_Check(ob))
		errmsg(PyString_AS_STRING(ob));
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "detail");
	if (PyString_Check(ob) && tb == NULL)
		errdetail(PyString_AS_STRING(ob));
	else
	{
		char *tbstr = NULL;
		char *obstr = NULL;
		if (PyString_Check(ob))
			obstr = PyString_AS_STRING(ob);
		if (tb != NULL)
			tbstr = tbtostr(tb);

		if (tbstr != NULL)
		{
			if (obstr != NULL)
				errdetail("%s\n(most recent call last):\n%s", obstr, tbstr);
			else
				errdetail("(most recent call last):\n%s", tbstr);
			free(tbstr);
		}
		else if (obstr != NULL)
			errdetail(obstr);
	}
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "context");
	if (PyString_Check(ob))
		errcontext(PyString_AS_STRING(ob));
	Py_DECREF(ob);

	ob = PyObject_GetAttrString(v, "hint");
	if (PyString_Check(ob))
		errhint(PyString_AS_STRING(ob));
	Py_DECREF(ob);

	PyErr_Clear();
	errfinish(0);
}

/*
 * pl_validator
 *		fmgr interface to pl_compile()
 */
PG_FUNCTION_INFO_V1(pl_validator);
Datum
pl_validator(PG_FUNCTION_ARGS)
{
	Oid fn_oid = PG_GETARG_OID(0);
	volatile HeapTuple procTuple;

	Datum srctext;
	bytea *probin;
	char *src;
	bool isnull = false;

	PyObj code, mo;

	Assert(fn_oid != InvalidOid);
	Assert(Py_IsInitialized());
	
	procTuple = SearchSysCacheCopy(PROCOID, fn_oid, 0, 0, 0);
	if (procTuple == NULL)
		ereport(ERROR, (
			errmsg("failed to fetch procedure tuple"),
			errcontext("function validation"),
			errdetail("SearchSysCacheCopy(PROCOID = %d, fn_oid = %d, 0, 0, 0)",
				PROCOID, fn_oid)
		));

	PG_TRY();
	{
		Relation proc;
		proc = RelationIdGetRelation(ProcedureRelationId);
		if (!RelationIsValid(proc))
			ereport(ERROR, (
				errmsg("failed to get pg_proc Relation"),
				errdetail("RelationIdGetRelation(ProcedureRelationId = %d)",
					ProcedureRelationId)
			));
		srctext = fastgetattr(
						procTuple,
						Anum_pg_proc_prosrc,
						RelationGetDescr(proc),
						&isnull
					);
		RelationClose(proc);

		if (isnull == true || srctext == 0)
			ereport(ERROR,(
				errmsg("failed to get valid source from pg_proc tuple \"%d\"",
					HeapTupleGetOid(procTuple))
			));

		srctext = DirectFunctionCall1(textout, srctext);
		if (PG_UTF8 != GetDatabaseEncoding())
			src = pg_do_encoding_conversion(
				DatumGetPointer(srctext),
				strlen(DatumGetPointer(srctext)),
				GetDatabaseEncoding(), PG_UTF8
			);
		else
			src = DatumGetPointer(srctext);
		if (PointerGetDatum(src) != srctext)
			pfree(DatumGetPointer(srctext));
		code = compile(src);
		pfree(src);

		if (code == NULL)
			ereport(ERROR,(
				errmsg("procedure compilation failure"),
				PythonExceptionDetail
			));

		DECREF(((PyCodeObject *) code)->co_filename);
		((PyCodeObject *) code)->co_filename = PyString_FromFormat(
			"%d", HeapTupleGetOid(procTuple)
		);

		DECREF(((PyCodeObject *) code)->co_name);
		src = format_procedure(fn_oid);
		((PyCodeObject *) code)->co_name = PyString_FromString(src);
		pfree(src);

		mo = PyMarshal_WriteObjectToString(code, 0);
		DECREF(code);

		if (mo == NULL)
			ereport(ERROR,(
				errmsg("failed to marshal function"),
				PythonExceptionDetail
			));

		probin = mkbytea(PyString_AS_STRING(mo), PyString_Size(mo));
		DECREF(mo);

		if (probin == NULL)
			ereport(ERROR, (
				errmsg("failed to make bytea from a Python string")
			));

		pg_proc_probin(procTuple, PointerGetDatum(probin));
		pfree(probin);
	}
	PG_CATCH();
	{
		heap_freetuple(procTuple);
		PG_RE_THROW();
	}
	PG_END_TRY();

	heap_freetuple(procTuple);
	return(0);
}

static Datum
pull_trigger(PG_FUNCTION_ARGS)
{
	PyObj tp, rob;
	Datum rd = 0;

	tp = PyPgTriggerPull_New(fcinfo);
	if (tp == NULL)
		ereport(ERROR,(
			errmsg("failed to create a TriggerPull object "
				"from the FunctionCallInfo CObject"),
			PythonExceptionDetail
		));

	rob = Py_Call(tp);
	DECREF(tp);
	if (rob == NULL)
	{
		if (PyErr_ExceptionMatches(PyPgCI.Error.Type))
			PostgresError_FromPythonException();
		else
			ereport(ERROR,(
				errmsg("Python exception occurred"),
				PythonExceptionDetail
			));
	}

	if (rob == Py_None || rob == Py_False)
	{
		/* Skip */
		rd = 0;
	}
	else if (rob == Py_True)
	{
		/* Continue as planned */
		TriggerData *td = (TriggerData *) (fcinfo->context);
		rd = PointerGetDatum(
			td->tg_newtuple ? td->tg_newtuple : td->tg_trigtuple
		);
	}
	else
	{
		/* Replacement HeapTuple returned */
		MemoryContext former = CurrentMemoryContext;
		HeapTuple ht;
		Assert(PyPgHeapTuple_Check(rob));

		MemoryContextSwitchTo(fcinfo->flinfo->fn_mcxt);
		ht = heap_copytuple(PyPgHeapTuple_FetchHeapTuple(rob));
		MemoryContextSwitchTo(former);
		rd = PointerGetDatum(ht);
	}
	DECREF(rob);

	return(rd);
}

static Datum
call_function(PG_FUNCTION_ARGS)
{
	Node *ri = fcinfo->resultinfo;
	ReturnSetInfo *rsi =
		(ri && IsA(ri, ReturnSetInfo)) ? (ReturnSetInfo *) ri : NULL;
	PyObj plc, rob;
	Datum rd;

	plc = (PyObj) fcinfo->flinfo->fn_extra;
	if (plc == NULL)
	{
		plc = PyPgProceduralCall_New(fcinfo);
		if (plc == NULL)
			ereport(ERROR,(
				errmsg("failed to create a ProceduralCall object "
					"from the FunctionCallInfo CObject"),
				PythonExceptionDetail
			));
	}
	else
	{
		PyPgProceduralCall_FixFCInfo(plc, fcinfo);
		if (rsi == NULL)
			PyPgProceduralCall_InitializeParameters(plc);
	}

	rob = Py_Call(plc);

	if (rob == NULL)
	{
		if (PyErr_ExceptionMatches(PyPgCI.Error.Type))
			PostgresError_FromPythonException();
		else
			ereport(ERROR,(
				errmsg("Python exception occurred"),
				PythonExceptionDetail
			));
	}
	if (rsi)
	{
		if (rsi->returnMode == SFRM_ValuePerCall)
		{
			if (rsi->isDone == ExprEndResult)
			{
				if (fcinfo->flinfo->fn_extra != NULL)
				{
					int ind;
					UnregisterExprContextCallback(rsi->econtext,
							srfeccb, PointerGetDatum(fcinfo->flinfo));
					ind = PySequence_Index(fnExtraCalls, plc);
					PySequence_DelItem(fnExtraCalls, ind);
					fcinfo->flinfo->fn_extra = NULL;
				}
			}
			else
			{
				RegisterExprContextCallback(rsi->econtext,
						srfeccb, PointerGetDatum(fcinfo->flinfo));
				PyList_Append(fnExtraCalls, plc);
				fcinfo->flinfo->fn_extra = plc;
			}
			fcinfo->flinfo->fn_mcxt = rsi->econtext->ecxt_per_query_memory;
		}
	}
	else if (fcinfo->flinfo->fn_extra == NULL)
	{
		int err;
		err = PyList_Append(fnExtraCalls, plc);
		DECREF(plc);
		if (err < 0)
			ereport(ERROR,(
				errmsg("failed to store call in fnExtra"),
				PythonExceptionDetail
			));
		fcinfo->flinfo->fn_extra = plc;
	}

	if (rob == Py_None)
	{
		rd = 0;
		fcinfo->isnull = true;
	}
	else
	{
		if (PyPgObject_IsNULL(rob))
		{
			rd = 0;
			fcinfo->isnull = true;
		}
		else
		{
			Form_pg_type ts = PyPgObject_FetchTypeStruct(rob);
			MemoryContext former = CurrentMemoryContext;
			rd = PyPgObject_FetchDatum(rob);

			MemoryContextSwitchTo(fcinfo->flinfo->fn_mcxt);
			rd = datumCopy(rd, ts->typbyval, ts->typlen);
			MemoryContextSwitchTo(former);
		}
	}
	DECREF(rob);

	return(rd);
}

static PGFunction pl;

static Datum
pl_executor(PG_FUNCTION_ARGS)
{
	Datum rd = 0;

	if (CALLED_AS_TRIGGER(fcinfo))
		rd = pull_trigger(fcinfo);
	else
		rd = call_function(fcinfo);

	return(rd);
}

static Datum
pl_initializor(PG_FUNCTION_ARGS)
{
	PyObj code, func;

	/*
	 * This first lookup will not have the code initialized, as
	 * the function type does not know what is a Python language function
	 * and what is not.
	 */
	PyPgCI.PL.Handler = fcinfo->flinfo->fn_addr;
	func = PyPgFunction_FromOid(fcinfo->flinfo->fn_oid);
	code = PyPgFunction_FetchCode(func);
	if (CodeFixer(func) < 0)
		ereport(ERROR,(
			errmsg("failed to initialize procedural language")
		));
	Py_DECREF(code);
	PyPgCI.PL.Oid = PyPgFunction_FetchProcStruct(func)->prolang;
	Py_DECREF(func);

	/*
	 * Everything is initialized, and will be for the duration of this process.
	 * Set the handler to the executor, and call the executor.
	 */
	pl = pl_executor;
	return(pl_executor(fcinfo));
}
static PGFunction pl = pl_initializor;

/*
 * pl_handler
 *
 * This function just calls pl, which is set to a function which represents
 * its current demand for the function executor.
 *
 * Right now, it is used to avoid a conditional on post initialization.
 */
PG_FUNCTION_INFO_V1(pl_handler);
Datum pl_handler(PG_FUNCTION_ARGS)
{
	Datum rd;
	MemoryContext fn_mcxt = fcinfo->flinfo->fn_mcxt;
	MemoryContext former = MemoryContextSwitchTo(TopMemoryContext);

	Assert(Py_IsInitialized());

	rd = pl(fcinfo);
	fcinfo->flinfo->fn_mcxt = fn_mcxt;
	MemoryContextSwitchTo(former);
	return(rd);
}
/*
 * vim: ts=3:sw=3:noet:
 */
