#!/usr/bin/env python3
"""gitlab-release.py - this uploads a fetchmail release to Gitlab

Assumption: you have a python-gitlab configuration 
(usually in ~/.python-gitlab.cfg) with a "gitlab" section that has these fields
(replace your real gitlab API token on the private_token line):

[gitlab]
url = https://gitlab.com
api_version = 4
ssl_verify = true
timeout = 10
private_token = glpat-XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
"""

import gitlab
import argparse
import sys
from pathlib import Path

ap = argparse.ArgumentParser()
ap.add_argument("project", help="project ID (number from Gitlab)")
ap.add_argument("release", help="release name (should match tag and milestone)")
ap.add_argument("releasenotes", help="release notes (file)")
ap.add_argument("tarball", help="tarball to upload (file)")
args = ap.parse_args()

release = args.release

gl = gitlab.Gitlab.from_config('gitlab')
gl.auth()

project = gl.projects.get(args.project)

# check if we have a milestone
milestone = project.milestones.list(search=release)

# print(project.releases.list(get_all=True))

with open(args.releasenotes, 'rt', encoding='utf-8') as f:
    relnotes = f.read()

file1 = Path(args.tarball)
file2 = Path(args.tarball + ".asc")

with (open(file1, "rb") as f1,
      open(file2, "rb") as f2):
        file1cts = f1.read()
        file2cts = f2.read()

uploadfile1 = project.upload(file1.as_posix(), filedata=file1cts)
uploadfile2 = project.upload(file2.as_posix(), filedata=file2cts)

descpfx = \
F'''# Important                                                                  
                                                                              
Please do not use the auto-generated "Source code" `.zip` or `*.tar*` files from
the Assets above.

Instead, **use the .tar.xz official package and its signature from below:**
                                                                              
  - [{file1.name}]({uploadfile1["url"]}) - fetchmail distribution                                       
  - [{file2.name}]({uploadfile2["url"]}) - ASCII-form detached GnuPG signature                                               

--------

# '''

try:
      project.releases.delete(release)
except Exception as e:
      print("EXCEPTION {e!r} IGNORED")

releaseobj = project.releases.create({
    'name': release,
    'tag_name': release,
    'description': descpfx + relnotes
})

if milestone:
        r = project.releases.get(release)
        r.milestones = release
        r.save()

print(f"### SUCCESS -> {releaseobj} ###")