import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
import { VirtualElement } from '@lumino/virtualdom';
/**
 * An object which holds data related to an object's title.
 *
 * #### Notes
 * A title object is intended to hold the data necessary to display a
 * header for a particular object. A common example is the `TabPanel`,
 * which uses the widget title to populate the tab for a child widget.
 *
 * It is the responsibility of the owner to call the title disposal.
 */
export declare class Title<T> implements IDisposable {
    /**
     * Construct a new title.
     *
     * @param options - The options for initializing the title.
     */
    constructor(options: Title.IOptions<T>);
    /**
     * A signal emitted when the state of the title changes.
     */
    readonly changed: ISignal<this, void>;
    /**
     * The object which owns the title.
     */
    readonly owner: T;
    /**
     * Get the label for the title.
     *
     * #### Notes
     * The default value is an empty string.
     */
    /**
    * Set the label for the title.
    */
    label: string;
    /**
     * Get the mnemonic index for the title.
     *
     * #### Notes
     * The default value is `-1`.
     */
    /**
    * Set the mnemonic index for the title.
    */
    mnemonic: number;
    /**
     * Get the icon renderer for the title.
     *
     * #### Notes
     * The default value is undefined.
     *
     * DEPRECATED: if set to a string value, the .icon field will function as
     * an alias for the .iconClass field, for backwards compatibility
     */
    /**
    * Set the icon renderer for the title.
    *
    * #### Notes
    * A renderer is an object that supplies a render and unrender function.
    *
    * DEPRECATED: if set to a string value, the .icon field will function as
    * an alias for the .iconClass field, for backwards compatibility
    */
    icon: VirtualElement.IRenderer | undefined | string;
    /**
     * Get the icon class name for the title.
     *
     * #### Notes
     * The default value is an empty string.
     */
    /**
    * Set the icon class name for the title.
    *
    * #### Notes
    * Multiple class names can be separated with whitespace.
    */
    iconClass: string;
    /**
     * Get the icon label for the title.
     *
     * #### Notes
     * The default value is an empty string.
     */
    /**
    * Set the icon label for the title.
    *
    * #### Notes
    * Multiple class names can be separated with whitespace.
    */
    iconLabel: string;
    /**
     * @deprecated Use `icon` instead.
     */
    /**
    * @deprecated Use `icon` instead.
    */
    iconRenderer: VirtualElement.IRenderer | undefined;
    /**
     * Get the caption for the title.
     *
     * #### Notes
     * The default value is an empty string.
     */
    /**
    * Set the caption for the title.
    */
    caption: string;
    /**
     * Get the extra class name for the title.
     *
     * #### Notes
     * The default value is an empty string.
     */
    /**
    * Set the extra class name for the title.
    *
    * #### Notes
    * Multiple class names can be separated with whitespace.
    */
    className: string;
    /**
     * Get the closable state for the title.
     *
     * #### Notes
     * The default value is `false`.
     */
    /**
    * Set the closable state for the title.
    *
    * #### Notes
    * This controls the presence of a close icon when applicable.
    */
    closable: boolean;
    /**
     * Get the dataset for the title.
     *
     * #### Notes
     * The default value is an empty dataset.
     */
    /**
    * Set the dataset for the title.
    *
    * #### Notes
    * This controls the data attributes when applicable.
    */
    dataset: Title.Dataset;
    /**
     * Test whether the title has been disposed.
     */
    readonly isDisposed: boolean;
    /**
     * Dispose of the resources held by the title.
     *
     * #### Notes
     * It is the responsibility of the owner to call the title disposal.
     */
    dispose(): void;
    private _label;
    private _caption;
    private _mnemonic;
    private _icon;
    private _iconClass;
    private _iconLabel;
    private _className;
    private _closable;
    private _dataset;
    private _changed;
    private _isDisposed;
}
/**
 * The namespace for the `Title` class statics.
 */
export declare namespace Title {
    /**
     * A type alias for a simple immutable string dataset.
     */
    type Dataset = {
        readonly [key: string]: string;
    };
    /**
     * An options object for initializing a title.
     */
    interface IOptions<T> {
        /**
         * The object which owns the title.
         */
        owner: T;
        /**
         * The label for the title.
         */
        label?: string;
        /**
         * The mnemonic index for the title.
         */
        mnemonic?: number;
        /**
         * The icon renderer for the title.
         *
         * DEPRECATED: if set to a string value, the .icon field will function as
         * an alias for the .iconClass field, for backwards compatibility
         */
        icon?: VirtualElement.IRenderer | string;
        /**
         * The icon class name for the title.
         */
        iconClass?: string;
        /**
         * The icon label for the title.
         */
        iconLabel?: string;
        /**
         * @deprecated Use `icon` instead.
         */
        iconRenderer?: VirtualElement.IRenderer;
        /**
         * The caption for the title.
         */
        caption?: string;
        /**
         * The extra class name for the title.
         */
        className?: string;
        /**
         * The closable state for the title.
         */
        closable?: boolean;
        /**
         * The dataset for the title.
         */
        dataset?: Dataset;
    }
}
//# sourceMappingURL=title.d.ts.map