import { IDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
import { IObservable } from './modeldb';
/**
 * A string which can be observed for changes.
 */
export interface IObservableString extends IDisposable, IObservable {
    /**
     * The type of the Observable.
     */
    type: 'String';
    /**
     * A signal emitted when the string has changed.
     */
    readonly changed: ISignal<this, IObservableString.IChangedArgs>;
    /**
     * The value of the string.
     */
    text: string;
    /**
     * Insert a substring.
     *
     * @param index - The starting index.
     *
     * @param text - The substring to insert.
     */
    insert(index: number, text: string): void;
    /**
     * Remove a substring.
     *
     * @param start - The starting index.
     *
     * @param end - The ending index.
     */
    remove(start: number, end: number): void;
    /**
     * Set the ObservableString to an empty string.
     */
    clear(): void;
    /**
     * Dispose of the resources held by the string.
     */
    dispose(): void;
}
/**
 * The namespace for `IObservableString` associate interfaces.
 */
export declare namespace IObservableString {
    /**
     * The change types which occur on an observable string.
     */
    type ChangeType = 
    /**
     * Text was inserted.
     */
    'insert'
    /**
     * Text was removed.
     */
     | 'remove'
    /**
     * Text was set.
     */
     | 'set';
    /**
     * The changed args object which is emitted by an observable string.
     */
    interface IChangedArgs {
        /**
         * The type of change undergone by the list.
         */
        type: ChangeType;
        /**
         * The starting index of the change.
         */
        start: number;
        /**
         * The end index of the change.
         */
        end: number;
        /**
         * The value of the change.
         *
         * ### Notes
         * If `ChangeType` is `set`, then
         * this is the new value of the string.
         *
         * If `ChangeType` is `insert` this is
         * the value of the inserted string.
         *
         * If `ChangeType` is remove this is the
         * value of the removed substring.
         */
        value: string;
    }
}
/**
 * A concrete implementation of [[IObservableString]]
 */
export declare class ObservableString implements IObservableString {
    /**
     * Construct a new observable string.
     */
    constructor(initialText?: string);
    /**
     * The type of the Observable.
     */
    get type(): 'String';
    /**
     * A signal emitted when the string has changed.
     */
    get changed(): ISignal<this, IObservableString.IChangedArgs>;
    /**
     * Set the value of the string.
     */
    set text(value: string);
    /**
     * Get the value of the string.
     */
    get text(): string;
    /**
     * Insert a substring.
     *
     * @param index - The starting index.
     *
     * @param text - The substring to insert.
     */
    insert(index: number, text: string): void;
    /**
     * Remove a substring.
     *
     * @param start - The starting index.
     *
     * @param end - The ending index.
     */
    remove(start: number, end: number): void;
    /**
     * Set the ObservableString to an empty string.
     */
    clear(): void;
    /**
     * Test whether the string has been disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by the string.
     */
    dispose(): void;
    private _text;
    private _isDisposed;
    private _changed;
}
