## Code of Conduct

When interacting with the project, the
[GNOME Code of Conduct](https://conduct.gnome.org/) applies.

## Use of Generative AI

This project does not allow contributions generated by large languages models
(LLMs) and chatbots. This ban includes tools like ChatGPT, Claude, Copilot,
DeepSeek, and Devin AI. We are taking these steps as precaution due to the
potential negative influence of AI generated content on quality, as well as
likely copyright violations.

This ban of AI generated content applies to all parts of the projects,
including, but not limited to, code, documentation, issues, and artworks. An
exception applies for purely translating texts for issues and comments to
English.

AI tools can be used to answer questions and find information. However, we
encourage contributors to avoid them in favor of using
[existing documentation](https://developer.gnome.org) and our
[chats and forums](https://welcome.gnome.org). Since AI generated information is
frequently misleading or false, we cannot supply support on anything referencing
AI output.

## Building

For build instructions see the [README.md](./README.md)

## Pull Requests

Before filing a pull request run the tests:

```sh
ninja -C _build test
```

Use descriptive commit messages, see

   https://wiki.gnome.org/Git/CommitMessages

and check

   https://wiki.openstack.org/wiki/GitCommitMessages

for good examples.

## Coding Style

We mostly use kernel style but

* Use spaces, never tabs
* Use 2 spaces for indentation

### GTK Style Function Argument Indentation

Use GTK style function argument indentation. It's harder for renames but it's
what GNOME upstream projects do.

*Good*:

```c
static gboolean
key_press_event_cb (GtkWidget *widget,
                    GdkEvent  *event,
                    gpointer   data)
```

*Bad*:

```c
static gboolean
key_press_event_cb (GtkWidget *widget, GdkEvent *event, gpointer data)
```

### Function Prototypes

Function prototypes should be grouped together in logical groups, e.g. all
constructors, or the getter and setter of a given property.
There should be no empty line in a group, and groups should be separated from
each other by an empty line.

The function's attributes and returned type should be on the same line as the
function's name.
Have only one parameter per line.
If a function takes no parameters, explicit it by using `void` as the parameter.

In a group, align function names, the opening parentheses, and parameter names
together.
Use the least possible amount of spaces while preserving alignment.
Asterisks from the returned type should be stuck to the function's name.
Similarly, asterisks from a parameter's type should be stuck to the parameter's
name.

Getters should come before setters.

In public headers, set the function's availability or deprecation state in a
line preceding its prototype.

*Good*:

```c
ADW_AVAILABLE_IN_ALL
AdwFoo *adw_foo_new (void) G_GNUC_WARN_UNUSED_RESULT;

ADW_AVAILABLE_IN_ALL
AdwBar *adw_foo_get_bar (AdwFoo *self);
ADW_AVAILABLE_IN_ALL
void    adw_foo_set_bar (AdwFoo *self,
                         AdwBar *bar);

ADW_AVAILABLE_IN_ALL
gboolean adw_foo_get_bit (AdwFoo   *self);
ADW_AVAILABLE_IN_ALL
void     adw_foo_set_bit (AdwFoo   *self,
                          gboolean  bit);

ADW_AVAILABLE_IN_ALL
void adw_foo_add_baz    (AdwFoo *self,
                         AdwBaz *baz);
ADW_AVAILABLE_IN_ALL
void adw_foo_remove_baz (AdwFoo *self,
                         AdwBaz *baz);

ADW_AVAILABLE_IN_ALL
void adw_foo_frobnicate (AdwFoo *self);
```

If the function transfers a new handle to a resource, like a reference, a
floating reference, a file handle, or any other kind of handle that would result
into a resource leak if ignored, add `G_GNUC_WARN_UNUSED_RESULT` after the
closing parenthesis.
No need to add `G_GNUC_WARN_UNUSED_RESULT` when the caller is guaranteed to have
a handle to the resource, e.g. in methods incrementing a reference counter.

*Good*:

```c
ADW_AVAILABLE_IN_ALL
AdwFoo *adw_foo_new (void) G_GNUC_WARN_UNUSED_RESULT;
ADW_AVAILABLE_IN_ALL
AdwFoo *adw_foo_ref (AdwFoo *self);

ADW_AVAILABLE_IN_ALL
char *adw_foo_to_string (AdwFoo *self) G_GNUC_WARN_UNUSED_RESULT;
```

*Bad*:

```c
ADW_AVAILABLE_IN_ALL
AdwFoo *adw_foo_new (void);
ADW_AVAILABLE_IN_ALL
AdwFoo *adw_foo_ref (AdwFoo *self) G_GNUC_WARN_UNUSED_RESULT;

ADW_AVAILABLE_IN_ALL
char *adw_foo_to_string (AdwFoo *self);
```

### Braces

Everything besides functions and structs have the opening curly brace on the same line.

*Good*:

```c
if (i < 0) {
    ...
}
```

*Bad*:

```c
if (i < 0)
{
    ...
}
```

Single line `if` or `else` statements don't need braces but if either `if` or
`else` have braces both get them:

*Good*:

```c
if (i < 0)
  i++;
else
  i--;
```

```c
if (i < 0) {
  i++;
  j++;
} else {
  i--;
}
```

```c
if (i < 0) {
  i++;
} else {
  i--;
  j--;
}
```

*Bad*:

```c
if (i < 0) {
  i++;
} else {
  i--;
}
```

```c
if (i < 0) {
  i++;
  j++;
} else
  i--;
```

```c
if (i < 0)
  i++;
else {
  i--;
  j--;
}
```

Function calls have a space between function name and invocation:

*Good*:

```c
visible_child_name = gtk_stack_get_visible_child_name (GTK_STACK (self->stack));
```

*Bad*:

```c
visible_child_name = gtk_stack_get_visible_child_name(GTK_STACK(self->stack));
```

### Header Inclusion Guards

Guard header inclusion with `#pragma once` rather than the traditional
`#ifndef`-`#define`-`#endif` trio.

Internal headers (for consistency, whether they need to be installed or not)
should contain the following guard to prevent users from directly including
them:
```c
#if !defined(_ADWAITA_INSIDE) && !defined(ADWAITA_COMPILATION)
#error "Only <adwaita.h> can be included directly."
#endif
```

Only after these should you include headers.

### Signals

Prefix signal enum names with *SIGNAL_*.

*Good*:

```c
enum {
  SIGNAL_SUBMITTED = 0,
  SIGNAL_DELETED,
  SIGNAL_SYMBOL_CLICKED,
  SIGNAL_LAST_SIGNAL,
};
```

Also note that the last element ends with a comma to reduce diff noise when
adding further signals.

### Properties

Prefix property enum names with *PROP_*.

*Good*:

```c
enum {
  PROP_0 = 0,
  PROP_NUMBER,
  PROP_SHOW_ACTION_BUTTONS,
  PROP_COLUMN_SPACING,
  PROP_ROW_SPACING,
  PROP_RELIEF,
  PROP_LAST_PROP,
};
```

Also note that the last element ends with a comma to reduce diff noise when
adding further properties.

### Comment Style

In comments use full sentences with proper capitalization and punctuation.

*Good*:

```c
/* Make sure we don't overflow. */
```

*Bad:*

```c
/* overflow check */
```

### Callbacks

Signal callbacks have a *_cb* suffix.

*Good*:

```c
g_signal_connect(self, "clicked", G_CALLBACK (button_clicked_cb), NULL);
```

*Bad*:

```c
g_signal_connect(self, "clicked", G_CALLBACK (handle_button_clicked), NULL);
```

### Static Functions

Static functions don't need the class prefix.  E.g. with a type foo_bar:

*Good*:

```c
static void
selection_changed_cb (AdwViewSwitcher   *self,
                      guint              position,
                      guint              n_items)
```

*Bad*:

```c
static void
adw_view_switcher_selection_changed_cb (AdwViewSwitcher   *self,
                                        guint              position,
                                        guint              n_items)
```

Note however that virtual methods like
*<class_name>_{init,constructed,finalize,dispose}* do use the class prefix.
These functions are usually never called directly but only assigned once in
*<class_name>_constructed* so the longer name is kind of acceptable. This also
helps to distinguish virtual methods from regular private methods.

### Self Argument

The first argument is usually the object itself so call it *self*. E.g. for a
non public function:

*Good*:

```c
static gboolean
expire_cb (FooButton *self)
{
  g_return_val_if_fail (BAR_IS_FOO_BUTTON (self), FALSE);
  ...
  return FALSE;
}
```

And for a public function:

*Good*:

```c
int
foo_button_get_state (FooButton *self)
{
  FooButtonPrivate *priv = bar_foo_get_instance_private(self);

  g_return_val_if_fail (BAR_IS_FOO_BUTTON (self), -1);
  return priv->state;
}
```

### User Interface Files

User interface files should end in *.ui*. If there are multiple ui
files put them in a ui/ subdirectory below the sources
(e.g. *src/ui/main-window.ui*).

#### Properties

Use minus signs instead of underscores in property names:

*Good*:

```xml
<property name="margin-start">12</property>
```

*Bad*:

```xml
<property name="margin_start">12</property>
```

## Docs

Libadwaita documentation uses gi-docgen and should follow its conventions, as
described in [gi-docgen docs](https://gnome.pages.gitlab.gnome.org/gi-docgen/tutorial.html).

### Summaries

gi-docgen uses the first paragraph of most doc comments as a summary that can be
displayed in a list of symbols. This paragraph must be kept short and to the
point. As a rule of thumb, it should not exceed a single line. It doesn't need
to describe every detail, add more paragraphs for that.

*Good*:

```md
 * Sets whether @self can be closed.
 *
 * If set to `FALSE`, the close button, shortcuts and
 * [method@Dialog.close] will result in [signal@Dialog::close-attempt] being
 * emitted instead, and bottom sheet close swipe will be disabled.
 * [method@Dialog.force_close] still works.
```

*Bad*:

```md
 * Sets whether @self can be closed. If set to `FALSE`, the close button,
 * shortcuts and [method@Dialog.close] will result in
 * [signal@Dialog::close-attempt] being emitted instead, and bottom sheet close
 * swipe will be disabled. [method@Dialog.force_close] still works.
```

### GTK-doc syntax

Even though gi-docgen supports legacy gtk-doc syntax within doc comments, avoid
using it and stick to gi-docgen syntax instead.

*Good*:

```md
`TRUE`
[class@Dialog]
[method@Dialog.present]
```

*Bad*:

```md
%TRUE
%AdwDialog
adw_dialog_present()
```

### Parameters and Return Values

gi-docgen already mentions parameter and return value types. Don't mention them
explicitly, instead use more specific descriptions when possible. Even in
obvious cases, such as the `@self` parameter, avoid repetition and omit the
type.

*Good*:

```md
 * @self: a window
 * @breakpoint: (transfer full): the breakpoint to add
```

*Bad*:

```md
 * @self: an `AdwWindow`
 * @breakpoint: (transfer full): an `AdwBreakpoint`
```

### Linking to Symbols

When linking to other libadwaita symbols, don't include the namespace when
possible.

*Good*:

```md
[class@NavigationView]
[property@Banner:title]
[method@Dialog.present]
[signal@AlertDialog::response]
[enum@AnimationState]
```

Linking to enum members needs a namespace, see
https://gitlab.gnome.org/GNOME/gi-docgen/-/issues/186

```md
[enum@Adw.AnimationState.PLAYING]
```

*Bad*:

```xml
[class@Adw.NavigationView]
[property@Adw.Banner:title]
[method@Adw.Dialog.present]
[signal@Adw.AlertDialog::response]
[enum@Adw.AnimationState]
```

### Headings

Avoid level 1 headings in doc comments, since gi-docgen will use those for the
symbol itself. Start from level 2.

*Good*:

```md
## CSS Nodes
```

*Bad*:

```md
# CSS Nodes
```

For standalone pages, use level 1 headings anyway. Otherwise, the table of
contents will look wrong.

### Class Descriptions

Try to include an overview of all the functionality the class has, as well as
an example when appropriate.

#### Widgets

Widget documentation should include a screenshot after the summary when
appropriate.

Widgets should also usually include the following sections:

- CSS nodes: describe the widget's CSS name, style classes and internal
  structure

- Style classes: if the widget has style classes available for it, list them
  here in addition to [doc/style-classes.md](./doc/style-classes.md)

- Accessibility: describe the widget's accessible role here

### Getter, and Setter and Property Descriptions

Getter, setter and property documentation should use the same wording when
possible.

Most of the time, the detailed description is irrelevant for getters, so omit it
and only include the summary.

### Copying From GTK

GTK documentation is less strict about style, so generally speaking docs copied
from GTK need to be reworded and expanded.

### Standalone Pages

Libadwaita has a number of standalone doc pages. Keep them up to date when
making changes.

- [adaptive-layouts.md](./doc/adaptive-layouts.md) lists adaptive patterns with
  screenshots and detailed examples. Update it when adding widgets that can be
  used for adaptive UIs.

  Note: this page lists patterns, not widgets. Use real-world examples. For
  example, split views are typically used for sidebars, so even though they don't
  provide sidebar contents on their own, include sidebar contents into the
  screenshots, rather than leaving the panes empty.

  Patterns can also span multiple widgets. For example, `AdwTabView`, `AdwTabBar`,
  `AdwTabOverview` and `AdwTabButton` comprise a single pattern and are meant to
  be used together, so they are all mentioned in a single section.

- [boxed-lists.md](./doc/boxed-lists.md) lists widgets implementing boxed lists:
  list rows, preferences groups etc. Update it when adding new row widgets.

- [css-variables.md](./doc/css-variables.md) lists CSS variables from the
  stylesheet, as well as their conventions. Update it whenever you add or change
  the variables.

- [style-classes.md](./doc/style-classes.md) lists the available style classes
  along with their screenshots, both unique to libadwaita and inherited from
  GTK. It should be updated whenever adding a new style class, making an
  existing one work for more widgets, or adding a new widget that can use
  existing classes.

- [styles-and-appearance.md](./doc/styles-and-appearance.md) lists general
  recommendations and guidelines for style handling. It should only be updated
  when adding significant new features, such as accent colors.

- [widget-gallery.md](./doc/widget-gallery.md) lists widgets along with their
  screenshots. Update it whenever adding a new widget.

### Screenshots

Libadwaita docs use screenshots generated from UI files.

#### Adding a screenshot

1. Create an `IMAGE.ui` file in the `doc/tools/data/` directory.
2. Put the widget to screenshot inside with the `widget` id. For example:

```xml
<?xml version="1.0" encoding="UTF-8"?>
<interface>
  <requires lib="gtk" version="4.0"/>
  <requires lib="libadwaita" version="1.0"/>
  <object class="GtkButton" id="widget">
    <property name="label">Example</property>
  </object>
</interface>
```

If a widget needs to be hovered - for example, a list item - put the `hover` id
onto it.

If the widget needs special treatment - for example, it's a `GtkPopover` - it
should be special-cased in `screenshot.c` based on its type.

If you need special styles, add them to [`doc/tools/style.css`](./doc/tools/style.css)
and/or [`doc/tools/style-dark.css`](./doc/tools/style-dark.css).

When demoing widgets that don't have a background and may look confusing in the
middle of a doc page, use the `.docs-background` CSS class. It will add a faint
grey background under the screenshot, like in the `AdwStatusPage` screenshot.

3. From the build directory, run:

```
./doc/tools/screenshot ../doc/tools/data/ ../doc/images/ -i IMAGE
```

4. The generator will create `IMAGE.png` and `IMAGE-dark.png` images. Add them
to [`doc/libadwaita.toml.in`](./doc/libadwaita.toml.in).
5. Use them in the docs as follows:

```html
<picture>
  <source srcset="IMAGE-dark.png" media="(prefers-color-scheme: dark)">
  <img src="IMAGE.png" alt="IMAGE">
</picture>
```

#### Regenerating Screenshots

Make sure your system has Adwaita Sans and Adwaita Mono fonts installed on your
system, otherwise the screenshots may have wrong fonts.

Make sure your system document and monospace fonts are set to:

- `org.gnome.desktop.interface` `document-font-name`: `Adwaita Sans 11`
- `org.gnome.desktop.interface` `monospace-font-name`: `Adwaita Mono 11`

To regenerate all screenshots, run:

```c
./doc/tools/screenshot ../doc/tools/data/ ../doc/images/
```

from the build directory.
