/* $Id: object.c,v 1.9 2005/06/19 19:33:31 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * be/src/obj.c,v 1.2 2005/02/05 19:36:02 flaw
 * imp/src/obj.c,v 1.3 2004/11/24 20:27:54 flaw
 * if/src/obj.c,v 1.34 2004/09/28 16:01:16 flaw
 *//*
 * Postgres object interface
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/heapam.h>
#include <access/hash.h>
#include <catalog/pg_type.h>
#include <catalog/pg_proc.h>
#include <catalog/pg_operator.h>
#include <nodes/params.h>
#include <parser/parse_func.h>
#include <parser/parse_type.h>
#include <parser/parse_oper.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <utils/array.h>
#include <utils/catcache.h>
#include <utils/datum.h>
#include <utils/geo_decls.h>
#include <utils/palloc.h>
#include <utils/builtins.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>

#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/utils.h>
#include <pypg/conv.h>
#include <pypg/tif.h>
#include <pypg/error.h>

PyObj
obj_keys(PyObj self)
{
	PyObj rob = NULL;

	if (PyPgObject_IsComposite(self))
	{
		TupleDesc td = PyPgObject_FetchTupleDesc(self);
		rob = TupleDesc_Keys(td);
	}
	else
	{
		rob = PyList_New(0);
	}

	return(rob);
}

PyObj
obj_haskey(PyObj self, PyObj name)
{
	PyObj rob = NULL;

	if (PyPgObject_IsComposite(self))
	{
		TupleDesc td = PyPgObject_FetchTupleDesc(self);
		PyObj str;
		char *cstr;

		str = PyObject_Str(name);
		if (str == NULL) return(NULL);

 		cstr = PyString_AS_STRING(str);

		rob = TupleDesc_HasKey(td, cstr) ? Py_True : Py_False;
		Py_DECREF(str);
	}
	else
		rob = Py_False;

	Py_INCREF(rob);
	return(rob);
}

static PyMethodDef PyPgObject_Methods[] = {
	{"keys", (PyCFunction) obj_keys, METH_NOARGS,
	"get the keys of the object or an empty list if none"},
	{"has_key", (PyCFunction) obj_haskey, METH_O,
	"check if the object has the given key"},
	{NULL}
};

static PyMemberDef PyPgObject_Members[] = {
	{"type", T_OBJECT, offsetof(struct PyPgObject, pgo_type), RO,
	"the object's Postgres type"},
	{NULL}
};

#define fetch_opOid(oprname, oprleft, oprright) \
LookupOperName(stringToQualifiedNameList(oprname, NULL), oprleft, oprright, 1)

static PyObj
obj_operate(char *op, PyObj self, PyObj with)
{
	Oid opOid = InvalidOid, opCode = InvalidOid, opRes = InvalidOid;
	HeapTuple opTup = NULL;

	volatile PyObj rob = NULL;
	volatile Datum rd = 0;

	Oid self_typoid = PyPgObject_FetchTypeOid(self);
	Oid with_typoid = InvalidOid;

	if (with)
	{
		with_typoid = PyPgObject_FetchTypeOid(with);
		opOid = fetch_opOid(op, self_typoid, with_typoid);
	}
	else
		opOid = fetch_opOid(op, InvalidOid, self_typoid);

	if ((opTup = SearchSysCache(OPEROID, opOid, 0, 0, 0)))
	{
		Form_pg_operator opStruct = OPERSTRUCT(opTup);
		opCode = opStruct->oprcode;
		opRes = opStruct->oprresult;
		ReleaseSysCache(opTup);
	}
	else
	{
		PyErr_Format(PyExc_LookupError,
			"no operator %s", op);
		return(NULL);
	}

	{
		struct FmgrInfo flinfo = {0,};
		struct FunctionCallInfoData fcid = {0,};
		fmgr_info(opCode, &flinfo);
		fcid.flinfo = &flinfo;
		fcid.nargs = 1;

		fcid.arg[0] = PyPgObject_FetchDatum(self);
		fcid.argnull[0] = PyPgObject_IsNULL(self);
		if (with)
		{
			++fcid.nargs;
			fcid.arg[1] = PyPgObject_FetchDatum(with);
			fcid.argnull[1] = PyPgObject_IsNULL(with);
		}
		PgError_TRAP(rd = FunctionCallInvoke(&fcid));
	}

	if (!PyErr_Occurred())
		rob = PyPgObject_FromTypeOidAndDatum(opRes, rd);

	return(rob);
}


static int
obj_length(PyObj self)
{
	int len = 0;
	Form_pg_type typs = PyPgObject_FetchTypeStruct(self);
	
	if (typs->typrelid)
	{
		len = PyPgObject_FetchTupleDesc(self)->natts;
	}
	else if (typs->typelem != 0 && typs->typlen == -1)
	{
		len = ARR_DIMS(PyPgObject_FetchDatum(self))[0];
	}
	else switch(PyPgObject_FetchTypeOid(self))
	{
		case BYTEAOID:
			len = PyPgObject_FetchVARSIZE(self) - VARHDRSZ;
		break;

		case BOXOID:
		case POINTOID:
		case LSEGOID:
		case CIRCLEOID:
			len = 2;
		break;

		case LINEOID:
			len = 3;
		break;

		case POLYGONOID:
			len = DatumGetPolygonP(PyPgObject_FetchDatum(self))->npts;
		break;

		case PATHOID:
			len = DatumGetPathP(PyPgObject_FetchDatum(self))->npts;
		break;
	}
	
	return(len);
}

static PyObj
obj_item(PyObj self, int item)
{
	Datum d = PyPgObject_FetchDatum(self);
	Oid typoid = PyPgObject_FetchTypeOid(self);
	Form_pg_type typs = PyPgObject_FetchTypeStruct(self);
	PyObj rob = NULL;

	if (PyPgObject_IsComposite(self))
	{
		HeapTupleHeader hth = (HeapTupleHeader) d;
		HeapTupleData htd;
		TupleDesc td = PyPgObject_FetchTupleDesc(self);
		HeapTupleData_FromHeader(htd, hth);

		rob = PyPgTIF_FetchItem(td, &htd, item);	
	}
	else if (typs->typelem != 0 && typs->typlen == -1)
	{
		ArrayType *array = DatumGetArrayTypeP(d);
		int eles, ndims;
		bool isnull = false;
		HeapTuple ett;
		Oid elmtypoid;
		Form_pg_type etyps;
		Datum rd;
		
		eles = ARR_DIMS(array)[0];
		if (item >= eles)
		{
			PyErr_Format(PyExc_IndexError,
				"index(%d) out of range for a '%s' object with %d elements",
				item, NameStr(typs->typname), eles
			);
			return(NULL);
		}
		
		ndims = ARR_NDIM(array);
		if (ndims > 1)
		{
			/* TODO: implement MDA support */
			PyErr_SetString(PyExc_NotImplementedError,
				"subscripting multi-dimensional arrays is unimplemented");
			return(NULL);
		}
		else
		{
			ett = SearchSysCache(TYPEOID, ObjectIdGetDatum(typs->typelem),
						0, 0, 0);
			elmtypoid = HeapTuple_FetchOid(ett);
			etyps = TYPESTRUCT(ett);
		}

		++item; /* lower bounds is 1 */
		rd = array_ref(array, 1, &item,
				typs->typlen,
				etyps->typlen,
				etyps->typbyval,
				etyps->typalign,
				&isnull);
		rd = datumCopy(rd, etyps->typbyval, etyps->typlen);
		if (ett)
			ReleaseSysCache(ett);

		rob = PyPgObject_FromTypeOidAndDatum(elmtypoid, rd);
	}
	else
	{
		switch(typoid)
		{
			case BYTEAOID:
			{
				Size s = VARSIZE(d) - VARHDRSZ;
				char *ds = ((char *) d) + VARHDRSZ + item;
				if (item > s)
				{
					PyErr_Format(PyExc_IndexError,
						"index(%d) is out of range for bytea(%d)", item, s);
					return(NULL);
				}
				rob = PyString_FromStringAndSize(ds, 1);
			}
			break;

			case CIRCLEOID:
				d = PointPGetDatum(&(DatumGetCircleP(d)->center));
			case POINTOID:
			{
				Point *p;
				if (item >= 2)
				{
					PyErr_Format(PyExc_IndexError,
						"index(%d) is out of range for Point objects", item);
					return(NULL);
				}
				p = DatumGetPointP(d);
				rob = PyFloat_FromDouble(item == 0 ? p->x : p->y);
			}
			break;

			case LSEGOID:
			{
				LSEG *ls;
				if (item >= 2)
				{
					PyErr_Format(PyExc_IndexError,
						"index(%d) is out of range for Line Segment objects", item);
					return(NULL);
				
				}
				ls = DatumGetLsegP(d);
				rob = PyPgObject_FromTypeOidAndDatum(POINTOID,
										PointPGetDatum(&(ls->p[item])));
			}
			break;

			case PATHOID:
			{
				PATH *path = DatumGetPathP(d);
				if (item >= path->npts)
				{
					PyErr_Format(PyExc_IndexError,
						"index(%d) is out of range for this Path(%d)",
						item, path->npts);
					return(NULL);
				}
				rob = PyPgObject_FromTypeOidAndDatum(POINTOID,
										PointPGetDatum(&(path->p[item])));
			}
			break;

			case LINEOID:
			{
				LINE *l = DatumGetLineP(d);
				double rd;
				switch (item)
				{
					case 0:
						rd = l->A;
					break;

					case 1:
						rd = l->B;
					break;

					case 2:
						rd = l->C;
					break;

					default:
					{
						PyErr_Format(PyExc_IndexError,
							"index(%d) is out of range for Line objects", item);
						return(NULL);
					}
					break;
				}
				rob = PyFloat_FromDouble(rd);
			}
			break;

			case BOXOID:
			{
				BOX *b = DatumGetBoxP(d);
				Point *rp = NULL;
				switch (item)
				{
					case 0:
						rp = &(b->high);
					break;

					case 1:
						rp = &(b->low);
					break;

					default:
					{
						PyErr_Format(PyExc_IndexError,
							"index(%d) is out of range for Box objects", item);
						return(NULL);
					}
				}
				rob = PyPgObject_FromTypeOidAndDatum(POINTOID, PointPGetDatum(rp));
			}
			break;

			case POLYGONOID:
			{
				POLYGON *poly = DatumGetPolygonP(d);
				Point *p;
				if (item >= poly->npts)
				{
					PyErr_Format(PyExc_IndexError,
						"index(%d) is out of range for this Polygon(%d)",
						item, poly->npts);
					return(NULL);
				}
				p = &(poly->p[item]);
				rob = PyPgObject_FromTypeOidAndDatum(POINTOID, PointPGetDatum(p));
			}
			break;

			default:
				PyErr_Format(PyExc_TypeError,
					"%s objects are not subscriptable",
					NameStr(typs->typname)
				);
			break;
		}
	}

	return(rob);
}

static PyObj
obj_slice(PyObj self, int fitem, int titem)
{
	RETURN_NONE;
}

/*
 * sq_concat and sq_repeat are NULL'd because they conflict with
 * nb_add and nb_multiply, which is a true representation of what
 * occurs when PyPgObject * ob or PyPgObject + ob happens. As the object
 * will be obj_operate'd with '+' or '*', instead of catenated
 * or repeated, regardless of ob_pgtype.
 */
static PySequenceMethods PyPgObjectAsSequence = {
	obj_length,			/* sq_length */
	NULL,					/* sq_concat */
	NULL,					/* sq_repeat */
	obj_item,			/* sq_item */
	obj_slice,			/* sq_slice */
	NULL,					/* sq_ass_item */
	NULL,					/* sq_ass_slice */
	NULL,					/* sq_contains */
	NULL,					/* sq_inplace_concat */
	NULL,					/* sq_inplace_repeat */
};

PyObj
obj_subscript(PyObj self, PyObj sub)
{
	PyObj rob = NULL;

	if (PyPgObject_IsComposite(self))
	{
		TupleDesc td;
		HeapTupleHeader hth;
		HeapTupleData htd;
		td = PyPgObject_FetchTupleDesc(self);
		hth = PyPgObject_FetchHeapTupleHeader(self);
		HeapTupleData_FromHeader(htd, hth);

		rob = PyPgTIF_FetchSubscript(td, &htd, sub);
	}
	else
	{
		long i = 0;
		i = PyInt_AsLong(sub);
		if (PyErr_Occurred())
			return(NULL);
		rob = PySequence_GetItem(self, i);
	}
	
	return(rob);
}

static PyMappingMethods PyPgObjectAsMapping = {
	obj_length,				/* mp_length */
	obj_subscript,			/* mp_subscript */
	NULL,						/* mp_ass_subscript */
};


static PyObj
obj_add(PyObj self, PyObj with)
{
	return(obj_operate("+", self, with));
}

static PyObj
obj_subtract(PyObj self, PyObj with)
{
	return(obj_operate("-", self, with));
}

static PyObj
obj_multiply(PyObj self, PyObj with)
{
	return(obj_operate("*", self, with));
}

static PyObj
obj_divide(PyObj self, PyObj with)
{
	return(obj_operate("/", self, with));
}

static PyObj
obj_remainder(PyObj self, PyObj with)
{
	return(obj_operate("%", self, with));
}

static PyObj
obj_divmod(PyObj self, PyObj with)
{
	PyObj div, mod, rob;
	rob = PyTuple_New(2);
	if (rob == NULL) return(NULL);

	div = obj_operate("/", self, with);
	if (div == NULL) goto error;
	PyTuple_SET_ITEM(rob, 0, div);
	mod = obj_operate("%", self, with);
	if (mod == NULL) goto error;
	PyTuple_SET_ITEM(rob, 1, mod);

	return(rob);
error:
	Py_DECREF(rob);
	return(NULL);
}

static PyObj
obj_power(PyObj self, PyObj with, PyObj andthis)
{
	return(obj_operate("^", self, with));
}

static PyObj
obj_negative(PyObj self)
{
	return(obj_operate("-", self, NULL));
}

static PyObj
obj_positive(PyObj self)
{
	return(obj_operate("+", self, NULL));
}

static PyObj
obj_absolute(PyObj self)
{
	Oid typoid = PyPgObject_FetchTypeOid(self);
	volatile Oid procoid, roid;
	volatile HeapTuple procTuple;
	volatile Datum rd = 0;
	volatile PyObj rob = NULL;

	procoid = LookupFuncName(stringToQualifiedNameList("abs", NULL),
			1, &(typoid), true);
	if (procoid == InvalidOid)
	{
		PyErr_Format(PyExc_LookupError,
				"no such function named 'abs' for type %u", typoid);
		return(NULL);
	}
	procTuple = SearchSysCache(PROCOID, procoid, 0, 0, 0);
	if (procTuple == NULL)
	{
		PyErr_Format(PyExc_LookupError,
				"no procedure with Oid %u", procoid);
		return(NULL);
	}
	
	roid = PROCSTRUCT(procTuple)->prorettype;
	PgError_TRAP(rd = OidFunctionCall1(procoid, PyPgObject_FetchDatum(self)));
	ReleaseSysCache(procTuple);

	if (!PyErr_Occurred())
	{
		rob = PyPgObject_FromTypeOidAndDatum(roid, rd);
		if (!(PyPgObject_FetchTypeStruct(rob)->typbyval))
			pfree(DatumGetPointer(rd));
	}

	return(rob);
}

static int
obj_nonzero(PyObj self)
{
	if (PyPgObject_IsNULL(self) || !PyPgObject_FetchDatum(self))
		return(0);

	/* FIXME: do a better postgres check on the object to see if its non-zero */
	return(1);
}

static PyObj
obj_invert(PyObj self)
{
	return(obj_operate("~", self, NULL));
}

static PyObj
obj_lshift(PyObj self, PyObj with)
{
	return(obj_operate("<<", self, with));
}

static PyObj
obj_rshift(PyObj self, PyObj with)
{
	return(obj_operate(">>", self, with));
}

static PyObj
obj_and(PyObj self, PyObj with)
{
	return(obj_operate("&", self, with));
}

static PyObj
obj_xor(PyObj self, PyObj with)
{
	return(obj_operate("#", self, with));
}

static PyObj
obj_or(PyObj self, PyObj with)
{
	return(obj_operate("|", self, with));
}

static PyObj obj_str(PyObj);
static PyObj
obj_int(PyObj self)
{
	Datum d = PyPgObject_FetchDatum(self);
	long r = 0;
	PyObj rob;

	switch(PyPgObject_FetchTypeOid(self))
	{
		case INT2OID:
			r = (long) DatumGetInt16(d);
		break;

		case INT4OID:
			r = (long) DatumGetInt32(d);
		break;

		case INT8OID:
		{
			long long ll;
			ll = DatumGetInt64(d);

			if (ll > LONG_MAX)
			{
				PyErr_SetString(PyExc_OverflowError,
						"Postgres int8 object overflows Python int");
				return(NULL);
			}
			r = (long) ll;
		}
		break;

		case REGTYPEOID:
		case REGPROCOID:
		case OIDOID:
			r = (long) DatumGetObjectId(d);
		break;

		case FLOAT4OID:
		{
			float f = floorf(DatumGetFloat4(d));
			
			if (f > LONG_MAX)
			{
				PyErr_SetString(PyExc_OverflowError,
						"Postgres float4 object overflows Python int");
				return(NULL);
			}
			r = (long) f;
		}
		break;
		
		case FLOAT8OID:
		{
			double f = floor(DatumGetFloat8(d));
			
			if (f > LONG_MAX)
			{
				PyErr_SetString(PyExc_OverflowError,
						"Postgres float8 object overflows Python int");
				return(NULL);
			}
			r = (long) f;
		}
		break;

		default:
		{
			PyObj obstr;
			obstr = PyObject_Str(self);
			if (obstr == NULL)
				return(NULL);
			rob = PyInt_FromString(PyString_AS_STRING(obstr), NULL, 0);
			Py_DECREF(obstr);
			return(rob);
		}
		break;
	}

	rob = PyInt_FromLong(r);
	return(rob);
}

static PyObj
obj_long(PyObj self)
{
	Datum d = PyPgObject_FetchDatum(self);
	long long r = 0;
	PyObj rob;

	switch(PyPgObject_FetchTypeOid(self))
	{
		case INT2OID:
			r = (long long) DatumGetInt16(d);
		break;

		case INT4OID:
			r = (long long) DatumGetInt32(d);
		break;

		case INT8OID:
			r = (long long) DatumGetInt64(d);
		break;

		case REGTYPEOID:
		case REGPROCOID:
		case OIDOID:
			r = (long long) DatumGetObjectId(d);
		break;

		case FLOAT4OID:
			r = (long long) floorf(DatumGetFloat4(d));
		break;
		
		case FLOAT8OID:
			r = (long long) floor(DatumGetFloat8(d));
		break;

		default:
		{
			PyObj obstr;
			obstr = obj_str(self);
			rob = PyLong_FromString(PyString_AS_STRING(obstr), NULL, 0);
			DECREF(obstr);
			return(rob);
		}
		break;
	}

	rob = PyLong_FromLongLong(r);
	return(rob);
}

static PyObj
obj_float(PyObj self)
{
	Datum d = PyPgObject_FetchDatum(self);
	double r = 0;
	PyObj rob;

	switch(PyPgObject_FetchTypeOid(self))
	{
		case INT2OID:
			r = (double) DatumGetInt16(d);
		break;

		case INT4OID:
			r = (double) DatumGetInt32(d);
		break;

		case INT8OID:
			r = (double) DatumGetInt64(d);
		break;

		case REGTYPEOID:
		case REGPROCOID:
		case OIDOID:
			r = (double) DatumGetObjectId(d);
		break;

		case FLOAT4OID:
			r = (double) DatumGetFloat4(d);
		break;
		
		case FLOAT8OID:
			r = (double) DatumGetFloat8(d);
		break;

		default:
		{
			PyObj obstr;
			obstr = obj_str(self);
			rob = PyFloat_FromString(obstr, NULL);
			DECREF(obstr);
			return(rob);
		}
		break;
	}

	rob = PyFloat_FromDouble(r);
	return(rob);

}

static PyObj
obj_oct(PyObj self)
{
	PyObj lo, rob = NULL;
	lo = obj_long(self);
	if (lo)
	{
		rob = lo->ob_type->tp_as_number->nb_oct(lo);
		DECREF(lo);
	}
	return(rob);
}

static PyObj
obj_hex(PyObj self)
{
	PyObj lo, rob = NULL;
	lo = obj_long(self);
	if (lo)
	{
		rob = lo->ob_type->tp_as_number->nb_hex(lo);
		DECREF(lo);
	}
	return(rob);
}

static PyNumberMethods PyPgObjectAsNumber = {
	(binaryfunc)obj_add,							/* nb_add */
	(binaryfunc)obj_subtract,					/* nb_subtract */
	(binaryfunc)obj_multiply,					/* nb_multiply */
	(binaryfunc)obj_divide,						/* nb_divide */
	(binaryfunc)obj_remainder,					/* nb_remainder */
	(binaryfunc)obj_divmod,						/* nb_divmod */
	(ternaryfunc)obj_power,						/* nb_power */
	(unaryfunc)obj_negative,					/* nb_negative */
	(unaryfunc)obj_positive,					/* nb_positive */
	(unaryfunc)obj_absolute,					/* nb_absolute */
	(inquiry)obj_nonzero,						/* nb_nonzero */
	(unaryfunc)obj_invert,						/* nb_invert */
	(binaryfunc)obj_lshift,						/* nb_lshift */
	(binaryfunc)obj_rshift,						/* nb_rshift */
	(binaryfunc)obj_and,							/* nb_and */
	(binaryfunc)obj_xor,							/* nb_xor */
	(binaryfunc)obj_or,							/* nb_or */
	(coercion)NULL,								/* nb_coerce */
	(unaryfunc)obj_int,							/* nb_int */
	(unaryfunc)obj_long,							/* nb_long */
	(unaryfunc)obj_float,						/* nb_float */
	(unaryfunc)obj_oct,							/* nb_oct */
	(unaryfunc)obj_hex,							/* nb_hex */

	(binaryfunc)NULL,								/* nb_inplace_add */
	(binaryfunc)NULL,								/* nb_inplace_subtract */
	(binaryfunc)NULL,								/* nb_inplace_multiply */
	(binaryfunc)NULL,								/* nb_inplace_divide */
	(binaryfunc)NULL,								/* nb_inplace_remainder */
	(ternaryfunc)NULL,							/* nb_inplace_power */
	(binaryfunc)NULL,								/* nb_inplace_lshift */
	(binaryfunc)NULL,								/* nb_inplace_rshift */
	(binaryfunc)NULL,								/* nb_inplace_and */
	(binaryfunc)NULL,								/* nb_inplace_xor */
	(binaryfunc)NULL,								/* nb_inplace_or */

	(binaryfunc)NULL,								/* nb_floor_divide */
	(binaryfunc)NULL,								/* nb_true_divide */
	(binaryfunc)NULL,								/* nb_inplace_floor_divide */
	(binaryfunc)NULL,								/* nb_inplace_true_divide */
};

static int
obj_compare(PyObj self, PyObj with)
{
	int ret = -1;

	if (PyPgObject_IsNULL(self) || PyPgObject_IsNULL(with) || with == Py_None)
		ret = -1;
	else if (self == with)
		ret = 0;
	else
	{
		PyObj withd;

		if (!PyPgObject_Check(with))
		{
			withd = PyPgObject_FromPyPgTypeAndPyObject(
							PyPgObject_FetchType(self), with);
			if (withd == NULL)
				return(-1);
		}
		else
		{
			withd = with;
			INCREF(withd);
		}

		if (PyPgObject_FetchDatum(self) == PyPgObject_FetchDatum(withd) &&
			 PyPgObject_FetchTypeOid(self) == PyPgObject_FetchTypeOid(withd))
			ret = 0;
		else
		{
			PyObj et;
			et = obj_operate("=", self, withd);
			if (et == NULL)
			{
				DECREF(withd);
				return(-1);
			}
			if (PyPgObject_FetchDatum(et)) ret = 0;
			DECREF(et);
			
			if (ret == -1)
			{
				PyObj lt;
				lt = obj_operate("<", self, withd);
				if (lt == NULL)
				{
					DECREF(withd);
					return(-1);
				}
				if (!PyPgObject_FetchDatum(lt)) ret = 1;

				DECREF(lt);
			}
		}

		DECREF(withd);
	}

	return(ret);
}

static long
obj_hash(PyObj self)
{
	Datum ob_datum = PyPgObject_FetchDatum(self);
	Form_pg_type ts = PyPgObject_FetchTypeStruct(self);
	int2 typlen = ts->typlen;
	long rv = 0;

	if (ts->typbyval)
	{
		rv = ((long) ob_datum);
	}
	else if (typlen > -1 && typlen <= sizeof(long))
	{
		rv = (*((long *) DatumGetPointer(ob_datum)));
	}
	else
	{
		int len;
		switch(typlen)
		{
			case -2:
				len = strlen((char *) DatumGetPointer(ob_datum));
			break;

			case -1:
				len = VARSIZE(ob_datum) - VARHDRSZ;
				ob_datum = PointerGetDatum(VARDATA(ob_datum));
			break;

			default:
				len = (int) typlen;
			break;
		}
		rv = hash_any(DatumGetPointer(ob_datum), len);
	}

	return(rv);
}

static PyObj
obj_repr(PyObj self)
{
	Form_pg_type ts = PyPgObject_FetchTypeStruct(self);
	char *typname = NameStr(ts->typname);
	PyObj rob;

	if (PyPgObject_IsNULL(self))
	{
		return(PyString_FromFormat("NULL::%s", typname));
	}
	else
	{
		char *str = NULL;

		PG_TRY();
		{
			Datum d = PyPgObject_FetchDatum(self);
			Datum typeoid = PyPgObject_FetchTypeOid(self);
			str = (char *) OidFunctionCall3(ts->typoutput, d, typeoid, -1);
		}
		PYPG_CATCH_END(return(NULL));
		rob = PyString_FromFormat("'%s'::%s", str, typname);
		pfree(str);
	}

	return(rob);
}

static PyObj
obj_str(PyObj self)
{
	Datum d = PyPgObject_FetchDatum(self);
	Form_pg_type ts = PyPgObject_FetchTypeStruct(self);
	volatile PyObj rob = NULL;
	char * volatile str = NULL;
	int size;

	if (PyPgObject_IsNULL(self))
	{
		str = "NULL";
		size = 4;
	}
	else switch (PyPgObject_FetchTypeOid(self))
	{
		case BYTEAOID:
		{
			size = VARSIZE(d) - VARHDRSZ;
			str = VARDATA(d);
		}
		break;

		default:
		{
			PgError_TRAP(str = (char *) OidFunctionCall1(ts->typoutput, d));
			if (PyErr_Occurred()) return(NULL);
			size = strlen(str);
			rob = PyString_FromStringAndSize(str, size);
			pfree(str);
			return(rob);
		}
		break;
	}

	rob = PyString_FromStringAndSize(str, size);
	return(rob);
}

static PyObj
obj_richcompare(PyObj self, PyObj with, int comparison)
{
	char *op = NULL;
	PyObj rob;

	if (!PyPgObject_Check(with) && !PyPgObject_Check(self))
	{
		Py_INCREF(Py_False);
		return(Py_False);
	}

	if (PyPgObject_IsNULL(self) || PyPgObject_IsNULL(with))
	{
		switch (comparison)
		{
			case Py_NE:
				rob = Py_True;
			break;
			default:
				rob = Py_False;
			break;
		}
		Py_INCREF(rob);
		return(rob);
	}

	switch (comparison)
	{
		case Py_LT:
			op = "<";
		break;

		case Py_LE:
			op = "<=";
		break;

		case Py_EQ:
			op = "=";
		break;

		case Py_NE:
			op = "!=";
		break;

		case Py_GT:
			op = ">";
		break;

		case Py_GE:
			op = ">=";
		break;

		default:
			PyErr_Format(PyExc_TypeError,
					"unknown comparison operation(%d)", comparison);
			return(NULL);
		break;
	}

	rob = obj_operate(op, self, with);
	return(rob);
}

static int
obj_traverse(PyObj self, visitproc visit, void *arg)
{
	int err = 0;
	PyObj ob;

	ob = PyPgObject_FetchType(self);
	if (ob != NULL)
	{
		err = visit(ob, arg);
	}

	return(err);
}

static int
obj_clear(PyObj self)
{
	Datum d = PyPgObject_FetchDatum(self);
	PyObj tp = PyPgObject_FetchType(self);

	/*
	 * If the type is NULL, there is no way to tell if the Datum is to be freed.
	 */
	if (tp != NULL)
	{
		if (PyPgObject_ShouldFree(self) && d != 0)
		{
			PyPgObject_FixDatum(self, 0);
			pfree(DatumGetPointer(d));
		}

		PyPgObject_FixType(self, NULL);
		DECREF(tp);
	}

	return(0);
}

static void
obj_dealloc(PyObj self)
{
	self->ob_type->tp_clear(self);
	self->ob_type->tp_free(self);
}

static PyObj
obj_getattro(PyObj self, PyObj name)
{
	Form_pg_type typs = PyPgObject_FetchTypeStruct(self);
	Datum ob_datum = PyPgObject_FetchDatum(self);
	char *attr = PyString_AS_STRING(name);
	PyObj rob = NULL;
	Oid to;

	if (!strcmp(attr, "size"))
	{
		int len;

		switch (typs->typlen)
		{
			/* Null terminated */
			case -2:
				len = strlen((char *) DatumGetPointer(ob_datum));
			break;

			/* Varlena */
			case -1:
				len = VARSIZE(ob_datum) - VARHDRSZ;
			break;

			default:
				len = typs->typlen;
			break;
		}

		return(PyInt_FromLong(len));
	}
	else if (!strcmp(attr, "data"))
	{
		char *str;
		int size;

		if (typs->typbyval)
		{
			str = (char *) &ob_datum;
			size = typs->typlen;
		}
		else
		{
			int2 typlen = typs->typlen;

			switch (typlen)
			{
				case -1:
					str = (char *) VARDATA(ob_datum);
					size = (Size) VARSIZE(ob_datum) - VARHDRSZ;
				break;

				case -2:
					str = DatumGetPointer(ob_datum);
					size = strlen(str);
				break;

				default:
					str = DatumGetPointer(ob_datum);
					size = typlen;
				break;
			}
		}
		rob = PyString_FromStringAndSize(str, size);
		return(rob);
	}

	if (PyPgObject_IsComposite(self))
	{
		HeapTupleHeader hth = (HeapTupleHeader) ob_datum;
		HeapTupleData htd;
		HeapTupleData_FromHeader(htd, hth);
		rob = PyPgTIF_FetchAttribute(&htd, attr);
	}
	else
	{
		to = PyPgObject_FetchTypeOid(self);
		switch (to)
		{
			case LSEGOID:
			{
				LSEG *ls = DatumGetLsegP(ob_datum);
				if (!strcmp(attr, "m"))
					rob = PyFloat_FromDouble(ls->m);
			}
			break;

			case PATHOID:
			{
				PATH *path = DatumGetPathP(ob_datum);
				if (!strcmp(attr, "closed"))
					rob = PyInt_FromLong(path->closed);
			}
			break;

			case LINEOID:
			{
				LINE *line = DatumGetLineP(ob_datum);
				if (!strcmp(attr, "m"))
					rob = PyFloat_FromDouble(line->m);
				else if (!strcmp(attr, "A"))
					rob = PyFloat_FromDouble(line->A);
				else if (!strcmp(attr, "B"))
					rob = PyFloat_FromDouble(line->B);
				else if (!strcmp(attr, "C"))
					rob = PyFloat_FromDouble(line->C);
			}
			break;

			case POLYGONOID:
			{
				POLYGON *poly = DatumGetPolygonP(ob_datum);
				if (!strcmp(attr, "boundbox"))
					rob = PyPgObject_FromTypeOidAndDatum(BOXOID,
								BoxPGetDatum(&(poly->boundbox)));
			}
			break;

			case CIRCLEOID:
			{
				CIRCLE *circ = DatumGetCircleP(ob_datum);
				if (!strcmp(attr, "radius"))
				{
					rob = PyFloat_FromDouble(circ->radius);
					break;
				}

				ob_datum = PointPGetDatum(&(circ->center));
			}
			case POINTOID:
			{
				char az = attr[0];
				if ((az == 'x' || az == 'y') && attr[1] == '\0')
				{
					Point *p = DatumGetPointP(ob_datum);
					double pd = 0;
					switch (attr[0])
					{
						case 'x':
							pd = p->x;
						break;
						case 'y':
							pd = p->y;
						break;
					}
					rob = PyFloat_FromDouble(pd);
				}
			}
			break;

			case TIDOID:
			{
				ItemPointer ip = (ItemPointer) ob_datum;
				if (!strcmp(attr, "block"))
				{
					rob = PyLong_FromUnsignedLong(ItemPointerGetBlockNumber(ip));
				}
				else if (!strcmp(attr, "offset"))
				{
					rob = PyInt_FromLong(ItemPointerGetOffsetNumber(ip));
				}
			}
			break;

			case BOXOID:
			{
				BOX *b = DatumGetBoxP(ob_datum);
				Point *rp = NULL;

				if (!strcmp(attr, "high"))
					rp = &(b->high);
				else if (!strcmp(attr, "low"))
					rp = &(b->low);

				if (rp != NULL)
				{
					ob_datum = PointPGetDatum(rp);
					rob = PyPgObject_FromTypeOidAndDatum(POINTOID, ob_datum);
				}
			}
		}
	}

	if (rob == NULL && !PyErr_Occurred())
		rob = PyObject_GenericGetAttr(self, name);
	return(rob);
}

/*
 * Initialize a Postgres.Object based on a PyPgType and a Datum source
 */
static PyObj
obj_new(PyTypeObject *subtype, PyObj args, PyObj kw)
{
	PyObj tob, dob, rob; /* Postgres.Type, Datum Source Object */
	Size sz;
	char *str;
	Form_pg_type typs;
	Datum dat;

	if (PyCFunctionErr_NoKeywordsAllowed(kw))
		return(NULL);

	if (!PyArg_ParseTuple(args, "OO", &tob, &dob))
		return(NULL);
	
	if (!PyPgType_Check(tob))
	{
		PyErr_Format(PyExc_TypeError,
				"first argument must be a Postgres.Type, not a '%s'", 
				tob->ob_type->tp_name
		);
		return(NULL);
	}

	if (!PyString_Check(dob))
	{
		PyErr_Format(PyExc_TypeError,
				"second argument must be a string, not a '%s'", 
				dob->ob_type->tp_name
		);
		return(NULL);
	}

	sz = PyObject_Length(dob);
	str = PyString_AS_STRING(dob);
	typs = PyPgType_FetchTypeStruct(tob);

	if (typs->typbyval)
	{
		dat = *((Datum *) str);
	}
	else
	{
		int2 typlen = typs->typlen;

		switch (typlen)
		{
			case -1:
			{
				struct varattrib *varl;
				Size tsz = sz + VARHDRSZ;

				varl = palloc(tsz);
				VARATT_SIZEP(varl) = tsz;
				memcpy(VARDATA(varl), str, sz);

				dat = PointerGetDatum(varl);
			}
			break;

			case -2:
				dat = PointerGetDatum(pstrdup(str));
			break;

			default:
			{
				char *dst;
				if (sz != typlen)
				{
					PyErr_Format(PyExc_OverflowError,
						"the type's length is %d bytes, "
						"but the given data is %d bytes", typlen, sz
					);
					return(NULL);
				}

				dst = palloc(typlen);
				memcpy(dst, str, typlen);
				dat = PointerGetDatum(dst);
			}
			break;
		}
	}

	rob = subtype->tp_alloc(subtype, 0);
	PyPgObject_Initialize(rob, tob, dat);
	return(rob);
}

const char PyPgObject_Doc[] =
"Python interface to Postgres objects";

PyTypeObject PyPgObject_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Object",				/* tp_name */
	sizeof(struct PyPgObject),		/* tp_basicsize */
	0,										/* tp_itemsize */
	obj_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	obj_compare,						/* tp_compare */
	obj_repr,							/* tp_repr */
	&PyPgObjectAsNumber,				/* tp_as_number */
	&PyPgObjectAsSequence,			/* tp_as_sequence */
	&PyPgObjectAsMapping,			/* tp_as_mapping */
	obj_hash,							/* tp_hash */
	NULL,									/* tp_call */
	obj_str,								/* tp_str */
	obj_getattro,						/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT|
	Py_TPFLAGS_HAVE_GC,			   /* tp_flags */
	(char *) PyPgObject_Doc,		/* tp_doc */
	obj_traverse,						/* tp_traverse */
	obj_clear,							/* tp_clear */
	obj_richcompare,					/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	PySeqIter_New,						/* tp_iter */
	NULL,									/* tp_iternext */
	PyPgObject_Methods,				/* tp_methods */
	PyPgObject_Members,				/* tp_members */
	NULL,									/* tp_getset */
	NULL,									/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	obj_new,								/* tp_new */
	NULL,									/* tp_free */
};

PyObj
PyPgObject_Initialize(PyObj self, PyObj typo, Datum d)
{
	if (self == NULL)
		return(NULL);

	INCREF(typo);
	PyPgObject_FixType(self, typo);
	PyPgObject_FixDatum(self, d);
	
	return(self);
}

PyObj
PyPgObject_Data(PyObj self)
{
	Datum ob_datum;
	int2 typlen;
	bool typbyval;
	char *str;
	int size;

	ob_datum = PyPgObject_FetchDatum(self);
	typlen = PyPgObject_FetchTypeStruct(self)->typlen;
	typbyval = PyPgObject_FetchTypeStruct(self)->typbyval;

	if (typbyval)
	{
		str = (char *) &ob_datum;
		size = typlen;
	}
	else
	{
		switch (typlen)
		{
			case -1:
				str = (char *) VARDATA(ob_datum);
				size = (Size) VARSIZE(ob_datum) - VARHDRSZ;
			break;

			case -2:
				str = DatumGetPointer(ob_datum);
				size = strlen(str);
			break;

			default:
				str = DatumGetPointer(ob_datum);
				size = typlen;
			break;
		}
	}

	return(PyString_FromStringAndSize(str, size));
}

PyObj
PyPgObject_FromPyPgTypeAndDatum(PyObj to, Datum dat)
{
	Form_pg_type ts = PyPgType_FetchTypeStruct(to);
	Datum rv = 0;
	PyObj rob;

	if (ts->typbyval == false)
	{
		if (ts->typlen == -1)
			rv = PointerGetDatum(PG_DETOAST_DATUM_COPY(dat));
		else
			rv = datumCopy(dat, false, ts->typlen);
	}
	else
		rv = dat;

	rob = PyPgObject_New(to, rv);
	return(rob);
}

PyObj
PyPgObject_FromTypeOidAndDatum(Oid typeoid, Datum d)
{
	PyObj typ, rob;

	typ = PyPgType_FromOid(typeoid);
	rob = PyPgObject_FromPyPgTypeAndDatum(typ, d);
	DECREF(typ);

	return(rob);
}

PyObj
PyPgObject_FromPyPgTypeAndHeapTuple(PyObj to, HeapTuple ht)
{
	HeapTupleHeader hth;
	PyObj rob;

	if (!PyPgType_IsComposite(to))
	{
		PyErr_SetString(PyExc_TypeError,
			"Postgres.Type is not a composite type");
		return(NULL);
	}

	hth = HeapTupleHeader_FromHeapTuple(ht);
	HeapTupleHeaderSetTypeId(hth, PyPgObject_FetchOid(to));
	rob = PyPgObject_New(to, PointerGetDatum(hth));

	return(rob);
}
/*
 * vim: ts=3:sw=3:noet:
 */
