[comment {-*- tcl -*- doctools manpage}]
[manpage_begin "Clipboard" n ""]
[copyright "2004-2009 Ashok P. Nadkarni"]
[moddesc "Clipboard access"]
[titledesc "Commands related to accessing the clipboard"]
[require twapi_clipboard]
[description]

[para]
This module provides procedures to read and write the clipboard as well
as to monitor changes. Tcl/Tk has a [cmd clipboard]
command that provides similar functionality. The [cmd TWAPI] commands
are useful in two circumstances:
[list_begin bullet]
[bullet]The Tcl [cmd clipboard] command is not available
in console-based Tcl programs that do not load [cmd Tk]. The [cmd TWAPI]
command may be used to access the clipboard in this case.
[bullet]More control is desired over the specific format of the data stored
into the clipboard.
[list_end]
In other cases, the Tcl/Tk [cmd clipboard] command should be preferred.

[section "Clipboard ownership"]

Access to the clipboard requires it to be first opened with
[uri #open_clipboard [cmd open_clipboard]] and closed afterwards with
[uri #close_clipboard [cmd close_clipboard]]. However, several commands will
do this on behalf of the caller if the clipboard was not already opened.

[para]
The [uri #get_clipboard_owner [cmd get_clipboard_owner]] and
[uri #get_open_clipboard_window [cmd get_open_clipboard_window]] return
information about the clipboard owner and the window
that has the clipboard open.

[section "Clipboard formats"]

Data may be stored in the clipboard in any number for formats
identified by integer values. When reading clipboard data, applications
must check in what formats the current clipboard data is currently
available and specify one of these formats when reading the
data. Similarly, when writing data to the clipboard, applications need
to specify the format in which data is to stored in the clipboard.
In some cases, Windows will automatically make multiple formats
available. For example, storing text in format type [const 1] (text)
will also make format type [const 13] (Unicode) available in the clipboard.

[para]

In addition to the standard set of formats defined by Windows
applications may register their own formats with the system. The format
values associated with these are created the first time the format name
is registered through
[uri \#register_clipboard_format [cmd register_clipboard_format]] and can be
retrieved later with the same call. Conversely, the name associated
with a format can be obtained through the
[uri \#get_registered_clipboard_format_name [cmd get_registered_clipboard_format_name]]
command. An example of a
registered application format name is "HTML Format" which is used
by Internet Explorer to store HTML fragments on the clipboard.

[para]
Multiple formats may be simultaneously stored into the clipboard. The formats
currently stored in the clipboard may be obtained through the
[uri \#get_clipboard_formats [cmd get_clipboard_formats]] command.

[para]
When reading or writing data
[uri \#read_clipboard [cmd read_clipboard]],
[uri \#write_clipboard [cmd write_clipboard]],
the application is responsible for formatting
data appropriately. These commands support the following clipboard formats:

[list_begin opt]
[opt_def 1]
ANSI text (CF_TEXT).
[opt_def 6]
TIFF image (CF_TIFF).
[opt_def 7]
OEM text (CF_OEMTEXT).
[opt_def 8]
Device independent bitmap (CF_DIB).
[opt_def 13]
Unicode text (CF_UNICODE).
[opt_def 14]
Enhanced meta file (CF_ENHMETAFILE). Only reads supported.
[opt_def 15]
File path list (CF_HDROP).
[opt_def 16]
Locale for the clipboard data (CF_LOCALE).
[opt_def 17]
Device independent bitmap V5 (CF_DIBV5).
[opt_def "HTML Format"]
HTML format. The numeric code should be obtained with [cmd register_clipboard_format].
[opt_def "PNG"]
PNG image. The numeric code should be obtained with [cmd register_clipboard_format].
[opt_def "GIF"]
GIF image. The numeric code should be obtained with [cmd register_clipboard_format].
[list_end]

Additionally, applications may call the
[uri \#clipboard_format_uses_global_heap [cmd clipboard_format_uses_global_heap]]
command to register formats that are known to use the global memory heap. These
can then be read and written with the [cmd read_clipboard] and [cmd write_clipboard]
commands.

The
[uri \#read_clipboard_text [cmd read_clipboard_text]] and
[uri \#write_clipboard_text [cmd write_clipboard_text]] commands are more
convenient to use for any text format. Likewise the
[uri \#read_clipboard_paths [cmd read_clipboard_paths]] and
[uri \#write_clipboard_paths [cmd write_clipboard_paths]]
are an easier alternative to use of the generic clipboard functions for
CF_HDROP format.

[section "Writing Data"]

To store data in a single format, the
[uri \#write_clipboard [cmd write_clipboard]],
[uri \#write_clipboard_text [cmd write_clipboard_text]] or
[uri \#write_clipboard_paths [cmd write_clipboard_paths]] may
be directly called.
[para]
To store multiple formats, the clipboard must first be
opened with the [uri \#open_clipboard [cmd open_clipboard]]
call followed by a call to [uri #empty_clipboard [cmd empty_clipboard]].
Data may then be stored in the clipboard in different formats
by calling 
[uri \#write_clipboard [cmd write_clipboard]],
[uri \#write_clipboard_text [cmd write_clipboard_text]] or
[uri \#write_clipboard_paths [cmd write_clipboard_paths]]
multiple times. Finally the clipboard must be
closed with [uri \#close_clipboard [cmd close_clipboard]].

[section "Reading Data"]

To read data from the clipboard, call
[uri \#read_clipboard [cmd read_clipboard]],
[uri \#read_clipboard_text [cmd read_clipboard_text]] or
[uri \#read_clipboard_paths [cmd read_clipboard_paths]].

[section "Monitoring clipboard changes"]
An application can monitor the clipboard by calling
[uri #start_clipboard_monitor [cmd start_clipboard_monitor]] to set up a notification
callback that is invoked when the contents of the clipboard change.
The clipboard also maintains a sequence number that is incremented on every
change. This sequence number can be retrieved with
[uri #get_clipboard_sequence [cmd get_clipboard_sequence]].


[section Commands]
[list_begin definitions]

[call [cmd clipboard_format_available] [arg FORMAT]]
Returns 1 if the clipboard contains data in the specified format, and
0 otherwise. This command does not require the clipboard to have been
opened.

[call [cmd clipboard_format_uses_global_heap] [arg FORMAT] [arg FORMAT] ...]

Call to register one or more clipboard formats that use the global memory heap
for transferring data. This allows clipboard content in those formats to be
read and written with [uri \#read_clipboard [cmd read_clipboard]] and
[uri \#write_clipboard [cmd write_clipboard]]. [arg FORMAT] may be specified
as an integer value or as the format name for custom formats. In the latter
case, the name will be registered with [cmd register_clipboard_format].

[call [cmd close_clipboard]]

Closes the clipboard which must have been
previously opened with [uri \#open_clipboard [cmd open_clipboard]].

[call [cmd empty_clipboard]]

Empties the clipboard and claims ownership for further writes.
The clipboard must have been
previously opened with [uri \#open_clipboard [cmd open_clipboard]].

[call [cmd get_clipboard_formats]]
Returns a list of the formats currently available in the clipboard. The
clipboard must have been opened before this function is called.

[call [cmd get_clipboard_owner]]
Returns the window handle of the window that owns the clipboard. This is
the window associated with the last call to [cmd EmptyClipboard] and is
generally the window that last wrote to the clipboard. The returned window
handle may be NULL if the clipboard is not currently owned.

[call [cmd get_clipboard_sequence]]
Returns the current clipboard sequence number. This number is incremented
every time the clipboard changes and can therefore be used by applications
that track the clipboard to figure out if the contents have changed.

[call [cmd get_open_clipboard_window]]
Returns the window handle of the window that has the clipboard open.
This may be NULL if no window has the clipboard open or if the current
[cmd OpenClipboard] call specified a NULL window handle.

[call [cmd get_registered_clipboard_format_name] [arg FORMAT]]
Returns the name associated with a registered clipboard format. [arg FORMAT]
identifies the format and must correspond to a registered format. An
exception will be raised if [arg FORMAT] is a standard Windows format or
a unregistered private format.

[call [cmd open_clipboard]]

Opens the clipboard for reading or writing. Most clipboard operations
require the clipboard to have been previously opened. The clipboard must
be closed once the operations are done by calling
[uri \#close_clipboard [cmd close_clipboard]].

[call [cmd read_clipboard] [arg FORMAT]]

Reads the contents of the clipboard in the given format and returns it.
If the clipboard was already opened through
[uri \#open_clipboard [cmd open_clipboard]],
it remains open when the command returns except in the case of any Tcl
exceptions. If the clipboard was not open, it will opened and closed
before the command returns.
In all cases where an exception is raised, the clipboard is closed before
the command returns.
[nl]
[arg FORMAT] must be one of the clipboard formats as defined in
[sectref "Clipboard formats"]. The content is an exact copy of the
contents of the clipboard in binary format. Callers will need to
use Tcl commands such as [cmd binary] and [cmd encoding] to parse the data.
The command will raise an exception for unsupported clipboard formats.

[call [cmd read_clipboard_paths]]

Returns a list of file paths stored in the clipboard.
If the clipboard was already opened through
[uri \#open_clipboard [cmd open_clipboard]],
it remains open when the command returns except in the case of any Tcl
exceptions. If the clipboard was not open, it will be opened and closed
before the command returns.
In all cases where an exception is raised, the clipboard is closed before
the command returns.

[call [cmd read_clipboard_text] [opt "[cmd -raw] [arg BOOLEAN]"]]

Retrieves the content of the clipboard as text.
If the clipboard was already opened through
[uri \#open_clipboard [cmd open_clipboard]],
it remains open when the command returns except in the case of any Tcl
exceptions. If the clipboard was not open, it will be opened and closed
before the command returns.
In all cases where an exception is raised, the clipboard is closed before
the command returns.
[nl]
If [cmd -raw] is specified as false (default),
the command converts CR-LF line terminators in clipboard content
to LF line terminators. If [cmd -raw] specified as true, the command
does no conversion of the clipboard data.

[call [cmd register_clipboard_format] [arg FORMATNAME]]

Registers an application defined format with the system
(see [sectref "Clipboard Formats"]). [arg FORMATNAME] is the name
associated with the format. The command returns an integer format identifier
that may be used in calls to read and write data. If the format name
is already registered, the same integer format value is returned.
This command does not require the clipboard to be open.

[call [cmd start_clipboard_monitor] [arg SCRIPT]]
Begins monitoring of clipboard contents.
[arg SCRIPT] will be invoked when the clipboard
is written to by any application.
[nl]
The command returns a handle.
When no longer required, this handle must be passed to the command
[uri #stop_clipboard_monitor [cmd stop_clipboard_monitor]].
[nl]
Multiple clipboard monitoring scripts may be active at the same
time. However, all returned handles must be passed to
[uri #stop_clipboard_monitor [cmd stop_clipboard_monitor]].
before the application exits.

[call [cmd stop_clipboard_monitor] [arg MONITOR_ID]]
Stops a previously registered script that monitors clipboard contents.
[arg MONITOR_ID] is monitoring handle previously returned by
[uri #start_clipboard_monitor [cmd start_clipboard_monitor]].

[call [cmd write_clipboard] [arg FORMAT] [arg DATA]]

Writes [arg DATA] to the clipboard in the given format.
If the clipboard is already open, caller must have also claimed ownership
through the [uri #empty_clipboard [cmd empty_clipboard]] command.  In
this case, the clipboard remains open when the command returns
after writing the data except in the case of any Tcl exceptions.
[nl]
If the clipboard was not open when this command is called,
the command opens the clipboard, clears it of all content
by calling [uri #empty_clipboard [cmd empty_clipboard]],
writes the data to the clipboard and closes it.
[nl]
In all cases, the clipboard is
closed before returning if an exception is thrown. This
ensures other applications
are not locked out of the clipboard on errors.

[nl]
[arg FORMAT] must be one of the clipboard formats as defined in
[sectref "Clipboard formats"]. The content is an exact copy of the
contents of the clipboard in binary format. Callers will need to
use Tcl commands such as [cmd binary] and [cmd encoding] to parse the data.
The command will raise an exception for unsupported clipboard formats.
[nl]
Multiple [cmd write_clipboard] commands may be executed between
a [uri \#open_clipboard [cmd open_clipboard]]
and [uri \#close_clipboard [cmd close_clipboard]] pair as long as the
specified formats are different for each. Writing a format previously
specified in the sequence overwrites the previous contents of that format.

[call [cmd write_clipboard_paths] [arg PATHS]]

Stores the specified list of file paths in the clipboard
in the [const CF_HDROP] ([const 15]) clipboard format.
Each path is normalized and converted to native format.
[nl]
If the clipboard is already open, caller must have also claimed ownership
through the [uri #empty_clipboard [cmd empty_clipboard]] command.  In
this case, the clipboard remains open when the command returns
after writing the data except in the case of any Tcl exceptions.
[nl]
If the clipboard was not open when this command is called,
the command opens the clipboard, clears it of all content
by calling [uri #empty_clipboard [cmd empty_clipboard]],
writes the data to the clipboard and closes it.
[nl]
In all cases, the clipboard is
closed before returning if an exception is thrown. This
ensures other applications
are not locked out of the clipboard on errors.

[call [cmd write_clipboard_text] [arg TEXT] [opt "[cmd -raw] [arg BOOLEAN]"]]

Writes the given text string to the clipboard.
If the clipboard is already open, caller must have also claimed ownership
through the [uri #empty_clipboard [cmd empty_clipboard]] command.  In
this case, the clipboard remains open when the command returns
after writing the data except in the case of any Tcl exceptions.
[nl]
If the clipboard was not open when this command is called,
the command opens the clipboard, clears it of all content
by calling [uri #empty_clipboard [cmd empty_clipboard]],
writes the data to the clipboard and closes it.
[nl]
In all cases, the clipboard is
closed before returning if an exception is thrown. This
ensures other applications
are not locked out of the clipboard on errors.
[nl]
If [cmd -raw] is specified as false (default),
the command converts LF line terminators without a preceding CR
to CRLF line terminators. If [cmd -raw] specified as true, the command
does no conversion.

[list_end]

[keywords clipboard "format, clipboard" "read clipboard" "clipboard formats" "write clipboard"]

[manpage_end]
