﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lexv2-models/LexModelsV2Request.h>
#include <aws/lexv2-models/LexModelsV2_EXPORTS.h>
#include <aws/lexv2-models/model/ImportResourceSpecification.h>
#include <aws/lexv2-models/model/MergeStrategy.h>

#include <utility>

namespace Aws {
namespace LexModelsV2 {
namespace Model {

/**
 */
class StartImportRequest : public LexModelsV2Request {
 public:
  AWS_LEXMODELSV2_API StartImportRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartImport"; }

  AWS_LEXMODELSV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier for the import. It is included in the response from the
   * <a
   * href="https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateUploadUrl.html">CreateUploadUrl</a>
   * operation.</p>
   */
  inline const Aws::String& GetImportId() const { return m_importId; }
  inline bool ImportIdHasBeenSet() const { return m_importIdHasBeenSet; }
  template <typename ImportIdT = Aws::String>
  void SetImportId(ImportIdT&& value) {
    m_importIdHasBeenSet = true;
    m_importId = std::forward<ImportIdT>(value);
  }
  template <typename ImportIdT = Aws::String>
  StartImportRequest& WithImportId(ImportIdT&& value) {
    SetImportId(std::forward<ImportIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Parameters for creating the bot, bot locale or custom vocabulary.</p>
   */
  inline const ImportResourceSpecification& GetResourceSpecification() const { return m_resourceSpecification; }
  inline bool ResourceSpecificationHasBeenSet() const { return m_resourceSpecificationHasBeenSet; }
  template <typename ResourceSpecificationT = ImportResourceSpecification>
  void SetResourceSpecification(ResourceSpecificationT&& value) {
    m_resourceSpecificationHasBeenSet = true;
    m_resourceSpecification = std::forward<ResourceSpecificationT>(value);
  }
  template <typename ResourceSpecificationT = ImportResourceSpecification>
  StartImportRequest& WithResourceSpecification(ResourceSpecificationT&& value) {
    SetResourceSpecification(std::forward<ResourceSpecificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The strategy to use when there is a name conflict between the imported
   * resource and an existing resource. When the merge strategy is
   * <code>FailOnConflict</code> existing resources are not overwritten and the
   * import fails.</p>
   */
  inline MergeStrategy GetMergeStrategy() const { return m_mergeStrategy; }
  inline bool MergeStrategyHasBeenSet() const { return m_mergeStrategyHasBeenSet; }
  inline void SetMergeStrategy(MergeStrategy value) {
    m_mergeStrategyHasBeenSet = true;
    m_mergeStrategy = value;
  }
  inline StartImportRequest& WithMergeStrategy(MergeStrategy value) {
    SetMergeStrategy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The password used to encrypt the zip archive that contains the resource
   * definition. You should always encrypt the zip archive to protect it during
   * transit between your site and Amazon Lex.</p>
   */
  inline const Aws::String& GetFilePassword() const { return m_filePassword; }
  inline bool FilePasswordHasBeenSet() const { return m_filePasswordHasBeenSet; }
  template <typename FilePasswordT = Aws::String>
  void SetFilePassword(FilePasswordT&& value) {
    m_filePasswordHasBeenSet = true;
    m_filePassword = std::forward<FilePasswordT>(value);
  }
  template <typename FilePasswordT = Aws::String>
  StartImportRequest& WithFilePassword(FilePasswordT&& value) {
    SetFilePassword(std::forward<FilePasswordT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_importId;

  ImportResourceSpecification m_resourceSpecification;

  MergeStrategy m_mergeStrategy{MergeStrategy::NOT_SET};

  Aws::String m_filePassword;
  bool m_importIdHasBeenSet = false;
  bool m_resourceSpecificationHasBeenSet = false;
  bool m_mergeStrategyHasBeenSet = false;
  bool m_filePasswordHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexModelsV2
}  // namespace Aws
