﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datazone/DataZoneRequest.h>
#include <aws/datazone/DataZone_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace DataZone {
namespace Model {

/**
 */
class ListEnvironmentActionsRequest : public DataZoneRequest {
 public:
  AWS_DATAZONE_API ListEnvironmentActionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListEnvironmentActions"; }

  AWS_DATAZONE_API Aws::String SerializePayload() const override;

  AWS_DATAZONE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the Amazon DataZone domain in which the environment actions are
   * listed.</p>
   */
  inline const Aws::String& GetDomainIdentifier() const { return m_domainIdentifier; }
  inline bool DomainIdentifierHasBeenSet() const { return m_domainIdentifierHasBeenSet; }
  template <typename DomainIdentifierT = Aws::String>
  void SetDomainIdentifier(DomainIdentifierT&& value) {
    m_domainIdentifierHasBeenSet = true;
    m_domainIdentifier = std::forward<DomainIdentifierT>(value);
  }
  template <typename DomainIdentifierT = Aws::String>
  ListEnvironmentActionsRequest& WithDomainIdentifier(DomainIdentifierT&& value) {
    SetDomainIdentifier(std::forward<DomainIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the envrironment whose environment actions are listed.</p>
   */
  inline const Aws::String& GetEnvironmentIdentifier() const { return m_environmentIdentifier; }
  inline bool EnvironmentIdentifierHasBeenSet() const { return m_environmentIdentifierHasBeenSet; }
  template <typename EnvironmentIdentifierT = Aws::String>
  void SetEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    m_environmentIdentifierHasBeenSet = true;
    m_environmentIdentifier = std::forward<EnvironmentIdentifierT>(value);
  }
  template <typename EnvironmentIdentifierT = Aws::String>
  ListEnvironmentActionsRequest& WithEnvironmentIdentifier(EnvironmentIdentifierT&& value) {
    SetEnvironmentIdentifier(std::forward<EnvironmentIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When the number of environment actions is greater than the default value for
   * the <code>MaxResults</code> parameter, or if you explicitly specify a value for
   * <code>MaxResults</code> that is less than the number of environment actions, the
   * response includes a pagination token named <code>NextToken</code>. You can
   * specify this <code>NextToken</code> value in a subsequent call to
   * <code>ListEnvironmentActions</code> to list the next set of environment
   * actions.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListEnvironmentActionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of environment actions to return in a single call to
   * <code>ListEnvironmentActions</code>. When the number of environment actions to
   * be listed is greater than the value of <code>MaxResults</code>, the response
   * contains a <code>NextToken</code> value that you can use in a subsequent call to
   * <code>ListEnvironmentActions</code> to list the next set of environment
   * actions.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListEnvironmentActionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainIdentifier;

  Aws::String m_environmentIdentifier;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_domainIdentifierHasBeenSet = false;
  bool m_environmentIdentifierHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
