/* $Id: extF80_sincos.h $ */
/** @file
 * SoftFloat - VBox Extension - constants for sine & cosine calculation.
 */

/*
 * Copyright (C) 2022-2025 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * SPDX-License-Identifier: GPL-3.0-only
 */

#ifndef VBOX_INCLUDED_SRC_source_extF80_sincos_h
#define VBOX_INCLUDED_SRC_source_extF80_sincos_h
#ifndef RT_WITHOUT_PRAGMA_ONCE
# pragma once
#endif

/** The pi constant as 128-bit floating point value.
 * base-10: 3.14159265358979323846264338327950279
 * base-16: 3.243f6a8885a308d313198a2e0370
 * base-2 : 1.1001001000011111101101010100010001000010110100011000010001101001100010011000110011000101000101110000000110111000e1 */
const RTFLOAT128U g_r128pi = RTFLOAT128U_INIT_C(0, 0x921fb54442d1, 0x8469898cc51701b8, 0x4000);

/** The pi/2 constant as 128-bit floating point value.
 * base-10: 1.57079632679489661923132169163975139
 * base-16: 1.921fb54442d18469898cc51701b8
 * base-2 : 1.1001001000011111101101010100010001000010110100011000010001101001100010011000110011000101000101110000000110111000 */
const RTFLOAT128U g_r128pi2 = RTFLOAT128U_INIT_C(0, 0x921fb54442d1, 0x8469898cc51701b8, 0x3fff);

/** CORDIC constants for fsin and fcos, defined by c(i)=atan(2^(-i)) */
const RTFLOAT128U g_ar128FsincosCORDICConsts[] =
{
    /* c0
     * base-10: 7.85398163397448309615660845819875699e-1
     * base-16: c.90fdaa22168c234c4c6628b80dc0@-1
     * base-2 : 1.1001001000011111101101010100010001000010110100011000010001101001100010011000110011000101000101110000000110111000e-1 */
    RTFLOAT128U_INIT_C(0, 0x921fb54442d1, 0x8469898cc51701b8, 0x3ffe),
    /* c1
     * base-10: 4.63647609000806116214256231461214397e-1
     * base-16: 7.6b19c1586ed3da2b7f222f65e1d4@-1
     * base-2 : 1.1101101011000110011100000101011000011011101101001111011010001010110111111100100010001011110110010111100001110101e-2 */
    RTFLOAT128U_INIT_C(0, 0xdac670561bb4, 0xf68adfc88bd97875, 0x3ffd),
    /* c2
     * base-10: 2.44978663126864154172082481211275806e-1
     * base-16: 3.eb6ebf25901bac55b71e7bd7de88@-1
     * base-2 : 1.1111010110110111010111111001001011001000000011011101011000101010110110111000111100111101111010111110111101000100e-3 */
    RTFLOAT128U_INIT_C(0, 0xf5b75f92c80d, 0xd62adb8f3debef44, 0x3ffc),
    /* c3
     * base-10: 1.24354994546761435031354849163871024e-1
     * base-16: 1.fd5ba9aac2f6dc65912f313e7d11@-1
     * base-2 : 1.1111110101011011101010011010101011000010111101101101110001100101100100010010111100110001001111100111110100010001e-4 */
    RTFLOAT128U_INIT_C(0, 0xfd5ba9aac2f6, 0xdc65912f313e7d11, 0x3ffb),
    /* c4
     * base-10: 6.24188099959573484739791129855051129e-2
     * base-16: f.faaddb967ef4e36cb2792dc0e2e0@-2
     * base-2 : 1.1111111101010101101110110111001011001111110111101001110001101101100101100100111100100101101110000001110001011100e-5 */
    RTFLOAT128U_INIT_C(0, 0xff55bb72cfde, 0x9c6d964f25b81c5c, 0x3ffa),
    /* c5
     * base-10: 3.12398334302682762537117448924909747e-2
     * base-16: 7.ff556eea5d892a13bcebbb6ed460@-2
     * base-2 : 1.1111111111010101010110111011101010010111011000100100101010000100111011110011101011101110110110111011010100011000e-6 */
    RTFLOAT128U_INIT_C(0, 0xffd55bba9762, 0x4a84ef3aeedbb518, 0x3ff9),
    /* c6
     * base-10: 1.56237286204768308028015212565703177e-2
     * base-16: 3.ffeaab776e5356ef9e31590057dc@-2
     * base-2 : 1.1111111111110101010101011011101110110111001010011010101101110111110011110001100010101100100000000010101111101110e-7 */
    RTFLOAT128U_INIT_C(0, 0xfff555bbb729, 0xab77cf18ac802bee, 0x3ff8),
    /* c7
     * base-10: 7.81234106010111129646339184219928134e-3
     * base-16: 1.fffd555bbba972d00c46a3f77cc1@-2
     * base-2 : 1.1111111111111101010101010101101110111011101010010111001011010000000011000100011010100011111101110111110011000001e-8 */
    RTFLOAT128U_INIT_C(0, 0xfffd555bbba9, 0x72d00c46a3f77cc1, 0x3ff7),
    /* c8
     * base-10: 3.90623013196697182762866531142438676e-3
     * base-16: f.fffaaaaddddb94bb12afb6b6d4f0@-3
     * base-2 : 1.1111111111111111010101010101010110111011101110110111001010010111011000100101010111110110110101101101101010011110e-9 */
    RTFLOAT128U_INIT_C(0, 0xffff5555bbbb, 0x72976255f6d6da9e, 0x3ff6),
    /* c9
     * base-10: 1.95312251647881868512148262507671384e-3
     * base-16: 7.ffff55556eeeea5ca6adeab02250@-3
     * base-2 : 1.1111111111111111110101010101010101011011101110111011101010010111001010011010101101111010101011000000100010010100e-10 */
    RTFLOAT128U_INIT_C(0, 0xffffd5555bbb, 0xba9729ab7aac0894, 0x3ff5),
    /* c10
     * base-10: 9.76562189559319430403430199717290821e-4
     * base-16: 3.ffffeaaaab77776e52e5a019fbce@-3
     * base-2 : 1.1111111111111111111101010101010101010101101110111011101110110111001010010111001011010000000011001111110111100111e-11 */
    RTFLOAT128U_INIT_C(0, 0xfffff55555bb, 0xbbb72972d00cfde7, 0x3ff4),
    /* c11
     * base-10: 4.88281211194898275469239625644848635e-4
     * base-16: 1.fffffd55555bbbbba97297625624@-3
     * base-2 : 1.1111111111111111111111010101010101010101010110111011101110111011101010010111001010010111011000100101011000100100e-12 */
    RTFLOAT128U_INIT_C(0, 0xfffffd55555b, 0xbbbba97297625624, 0x3ff3),
    /* c12
     * base-10: 2.44140620149361764016722943259659969e-4
     * base-16: f.fffffaaaaaaddddddb94b94d5bd0@-4
     * base-2 : 1.1111111111111111111111110101010101010101010101011011101110111011101110110111001010010111001010011010101101111010e-13 */
    RTFLOAT128U_INIT_C(0, 0xffffff555555, 0xbbbbbb729729ab7a, 0x3ff2),
    /* c13
     * base-10: 1.22070311893670204239058646117956300e-4
     * base-16: 7.ffffff5555556eeeeeea5ca5cb40@-4
     * base-2 : 1.1111111111111111111111111101010101010101010101010101101110111011101110111011101010010111001010010111001011010000e-14 */
    RTFLOAT128U_INIT_C(0, 0xffffffd55555, 0x5bbbbbba972972d0, 0x3ff1),
    /* c14
     * base-10: 6.10351561742087750216625691738291515e-5
     * base-16: 3.ffffffeaaaaaab7777776e52e52e@-4
     * base-2 : 1.1111111111111111111111111111010101010101010101010101010110111011101110111011101110110111001010010111001010010111e-15 */
    RTFLOAT128U_INIT_C(0, 0xfffffff55555, 0x55bbbbbbb7297297, 0x3ff0),
    /* c15
     * base-10: 3.05175781155260968618259534385360192e-5
     * base-16: 1.fffffffd5555555bbbbbbba97297@-4
     * base-2 : 1.1111111111111111111111111111110101010101010101010101010101011011101110111011101110111011101010010111001010010111e-16 */
    RTFLOAT128U_INIT_C(0, 0xfffffffd5555, 0x555bbbbbbba97297, 0x3fef),
    /* c16
     * base-10: 1.52587890613157621072319358126978848e-5
     * base-16: f.fffffffaaaaaaaaddddddddb94b8@-5
     * base-2 : 1.1111111111111111111111111111111101010101010101010101010101010101101110111011101110111011101110110111001010010111e-17 */
    RTFLOAT128U_INIT_C(0, 0xffffffff5555, 0x5555bbbbbbbb7297, 0x3fee),
    /* c17
     * base-10: 7.62939453110197026338848234010509046e-6
     * base-16: 7.ffffffff555555556eeeeeeeea5c@-5
     * base-2 : 1.1111111111111111111111111111111111010101010101010101010101010101010110111011101110111011101110111011101010010111e-18 */
    RTFLOAT128U_INIT_C(0, 0xffffffffd555, 0x55555bbbbbbbba97, 0x3fed),
    /* c18
     * base-10: 3.81469726560649628292307561637299366e-6
     * base-16: 3.ffffffffeaaaaaaaab777777776e@-5
     * base-2 : 1.1111111111111111111111111111111111110101010101010101010101010101010101011011101110111011101110111011101110110111e-19 */
    RTFLOAT128U_INIT_C(0, 0xfffffffff555, 0x555555bbbbbbbbb7, 0x3fec),
    /* c19
     * base-10: 1.90734863281018703536536930591724404e-6
     * base-16: 1.fffffffffd555555555bbbbbbbbb@-5
     * base-2 : 1.1111111111111111111111111111111111111101010101010101010101010101010101010101101110111011101110111011101110111011e-20 */
    RTFLOAT128U_INIT_C(0, 0xfffffffffd55, 0x5555555bbbbbbbbb, 0x3feb),
    /* c20
     * base-10: 9.53674316405960879420670689923112322e-7
     * base-16: f.fffffffffaaaaaaaaaadddddddd8@-6
     * base-2 : 1.1111111111111111111111111111111111111111010101010101010101010101010101010101010110111011101110111011101110111011e-21 */
    RTFLOAT128U_INIT_C(0, 0xffffffffff55, 0x55555555bbbbbbbb, 0x3fea),
    /* c21
     * base-10: 4.76837158203088859927583821449247042e-7
     * base-16: 7.ffffffffff55555555556eeeeeec@-6
     * base-2 : 1.1111111111111111111111111111111111111111110101010101010101010101010101010101010101011011101110111011101110111011e-22 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffd5, 0x555555555bbbbbbb, 0x3fe9),
    /* c22
     * base-10: 2.38418579101557982490947977218932680e-7
     * base-16: 3.ffffffffffeaaaaaaaaaab777776@-6
     * base-2 : 1.1111111111111111111111111111111111111111111101010101010101010101010101010101010101010101101110111011101110111011e-23 */
    RTFLOAT128U_INIT_C(0, 0xfffffffffff5, 0x5555555555bbbbbb, 0x3fe8),
    /* c23
     * base-10: 1.19209289550780685311368497137922104e-7
     * base-16: 1.fffffffffffd55555555555bbbbb@-6
     * base-2 : 1.1111111111111111111111111111111111111111111111010101010101010101010101010101010101010101010110111011101110111011e-24 */
    RTFLOAT128U_INIT_C(0, 0xfffffffffffd, 0x55555555555bbbbb, 0x3fe7),
    /* c24
     * base-10: 5.96046447753905544139210621417888700e-8
     * base-16: f.fffffffffffaaaaaaaaaaaadddd8@-7
     * base-2 : 1.1111111111111111111111111111111111111111111111110101010101010101010101010101010101010101010101011011101110111011e-25 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0x555555555555bbbb, 0x3fe6),
    /* c25
     * base-10: 2.98023223876953036767401327677095012e-8
     * base-16: 7.ffffffffffff5555555555556eec@-7
     * base-2 : 1.1111111111111111111111111111111111111111111111111101010101010101010101010101010101010101010101010101101110111011e-26 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xd555555555555bbb, 0x3fe5),
    /* c26
     * base-10: 1.49011611938476551470925165959632460e-8
     * base-16: 3.ffffffffffffeaaaaaaaaaaaab76@-7
     * base-2 : 1.1111111111111111111111111111111111111111111111111111010101010101010101010101010101010101010101010101010110111011e-27 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xf5555555555555bb, 0x3fe4),
    /* c27
     * base-10: 7.45058059692382798713656457449539158e-9
     * base-16: 1.fffffffffffffd5555555555555b@-7
     * base-2 : 1.1111111111111111111111111111111111111111111111111111110101010101010101010101010101010101010101010101010101011011e-28 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfd5555555555555b, 0x3fe3),
    /* c28
     * base-10: 3.72529029846191404526707057181192332e-9
     * base-16: f.fffffffffffffaaaaaaaaaaaaaa8@-8
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111101010101010101010101010101010101010101010101010101010101e-29 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xff55555555555555, 0x3fe2),
    /* c29
     * base-10: 1.86264514923095702909588382147649037e-9
     * base-16: 7.ffffffffffffff55555555555554@-8
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111010101010101010101010101010101010101010101010101010101e-30 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffd5555555555555, 0x3fe1),
    /* c30
     * base-10: 9.31322574615478515355735477684561273e-10
     * base-16: 3.ffffffffffffffeaaaaaaaaaaaaa@-8
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111110101010101010101010101010101010101010101010101010101e-31 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfff5555555555555, 0x3fe0),
    /* c31
     * base-10: 4.65661287307739257778841934710570148e-10
     * base-16: 1.fffffffffffffffd555555555555@-8
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111101010101010101010101010101010101010101010101010101e-32 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffd555555555555, 0x3fdf),
    /* c32
     * base-10: 2.32830643653869628902042741838821262e-10
     * base-16: f.fffffffffffffffaaaaaaaaaaaa8@-9
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111010101010101010101010101010101010101010101010101e-33 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffff555555555555, 0x3fde),
    /* c33
     * base-10: 1.16415321826934814452599092729852655e-10
     * base-16: 7.ffffffffffffffff555555555554@-9
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111110101010101010101010101010101010101010101010101e-34 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffd55555555555, 0x3fdd),
    /* c34
     * base-10: 5.82076609134674072264967615912315804e-11
     * base-16: 3.ffffffffffffffffeaaaaaaaaaaa@-9
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111101010101010101010101010101010101010101010101e-35 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffff55555555555, 0x3fdc),
    /* c35
     * base-10: 2.91038304567337036132730326989039468e-11
     * base-16: 1.fffffffffffffffffd5555555555@-9
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111010101010101010101010101010101010101010101e-36 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffd5555555555, 0x3fdb),
    /* c36
     * base-10: 1.45519152283668518066395978373629930e-11
     * base-16: f.fffffffffffffffffaaaaaaaaaa8@-10
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111110101010101010101010101010101010101010101e-37 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffff5555555555, 0x3fda),
    /* c37
     * base-10: 7.27595761418342590332018410467037395e-12
     * base-16: 7.ffffffffffffffffff5555555554@-10
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111101010101010101010101010101010101010101e-38 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffd555555555, 0x3fd9),
    /* c38
     * base-10: 3.63797880709171295166014020058379665e-12
     * base-16: 3.ffffffffffffffffffeaaaaaaaaa@-10
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111010101010101010101010101010101010101e-39 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffff555555555, 0x3fd8),
    /* c39
     * base-10: 1.81898940354585647583007611882297453e-12
     * base-16: 1.fffffffffffffffffffd55555555@-10
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111110101010101010101010101010101010101e-40 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffd55555555, 0x3fd7),
    /* c40
     * base-10: 9.09494701772928237915038811727871795e-13
     * base-16: f.fffffffffffffffffffaaaaaaaa8@-11
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111101010101010101010101010101010101e-41 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffff55555555, 0x3fd6),
    /* c41
     * base-10: 4.54747350886464118957519499903483963e-13
     * base-16: 7.ffffffffffffffffffff55555554@-11
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111010101010101010101010101010101e-42 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffd5555555, 0x3fd5),
    /* c42
     * base-10: 2.27373675443232059478759761706685489e-13
     * base-16: 3.ffffffffffffffffffffeaaaaaaa@-11
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111110101010101010101010101010101e-43 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffff5555555, 0x3fd4),
    /* c43
     * base-10: 1.13686837721616029739379882322710683e-13
     * base-16: 1.fffffffffffffffffffffd555555@-11
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111101010101010101010101010101e-44 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffd555555, 0x3fd3),
    /* c44
     * base-10: 5.68434188608080148696899413450263340e-14
     * base-16: f.fffffffffffffffffffffaaaaaa8@-12
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111010101010101010101010101e-45 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffff555555, 0x3fd2),
    /* c45
     * base-10: 2.84217094304040074348449706954720410e-14
     * base-16: 7.ffffffffffffffffffffff555554@-12
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111110101010101010101010101e-46 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffd55555, 0x3fd1),
    /* c46
     * base-10: 1.42108547152020037174224853506058797e-14
     * base-16: 3.ffffffffffffffffffffffeaaaaa@-12
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111101010101010101010101e-47 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffff55555, 0x3fd0),
    /* c47
     * base-10: 7.10542735760100185871124267566167230e-15
     * base-16: 1.fffffffffffffffffffffffd5555@-12
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111010101010101010101e-48 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffffd5555, 0x3fcf),
    /* c48
     * base-10: 3.55271367880050092935562133787567770e-15
     * base-16: f.fffffffffffffffffffffffaaaa8@-13
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111110101010101010101e-49 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffff5555, 0x3fce),
    /* c49
     * base-10: 1.77635683940025046467781066894344404e-15
     * base-16: 7.ffffffffffffffffffffffff5554@-13
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111101010101010101e-50 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffd555, 0x3fcd),
    /* c50
     * base-10: 8.88178419700125232338905334472422671e-16
     * base-16: 3.ffffffffffffffffffffffffeaaa@-13
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111010101010101e-51 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffffff555, 0x3fcc),
    /* c51
     * base-10: 4.44089209850062616169452667236298917e-16
     * base-16: 1.fffffffffffffffffffffffffd55@-13
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111110101010101e-52 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffffffd55, 0x3fcb),
    /* c52
     * base-10: 2.22044604925031308084726333618160406e-16
     * base-16: f.fffffffffffffffffffffffffaa8@-14
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111101010101e-53 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffff55, 0x3fca),
    /* c53
     * base-10: 1.11022302462515654042363166809081571e-16
     * base-16: 7.ffffffffffffffffffffffffff54@-14
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111010101e-54 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffd5, 0x3fc9),
    /* c54
     * base-10: 5.55111512312578270211815834045409568e-17
     * base-16: 3.ffffffffffffffffffffffffffea@-14
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111110101e-55 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffffffff5, 0x3fc8),
    /* c55
     * base-10: 2.77555756156289135105907917022704997e-17
     * base-16: 1.fffffffffffffffffffffffffffd@-14
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111101e-56 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xfffffffffffffffd, 0x3fc7),
    /* c56
     * base-10: 1.38777878078144567552953958511352525e-17
     * base-16: f.fffffffffffffffffffffffffff8@-15
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-57 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc6),
    /* c57
     * base-10: 6.93889390390722837764769792556762628e-18
     * base-16: 7.fffffffffffffffffffffffffffc@-15
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-58 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc5),
    /* c58
     * base-10: 3.46944695195361418882384896278381314e-18
     * base-16: 3.fffffffffffffffffffffffffffe@-15
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-59 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc4),
    /* c59
     * base-10: 1.73472347597680709441192448139190657e-18
     * base-16: 1.ffffffffffffffffffffffffffff@-15
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-60 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc3),
    /* c60
     * base-10: 8.67361737988403547205962240695953285e-19
     * base-16: f.fffffffffffffffffffffffffff8@-16
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-61 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc2),
    /* c61
     * base-10: 4.33680868994201773602981120347976642e-19
     * base-16: 7.fffffffffffffffffffffffffffc@-16
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-62 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc1),
    /* c62
     * base-10: 2.16840434497100886801490560173988321e-19
     * base-16: 3.fffffffffffffffffffffffffffe@-16
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-63 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fc0),
    /* c63
     * base-10: 1.08420217248550443400745280086994160e-19
     * base-16: 1.ffffffffffffffffffffffffffff@-16
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-64 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fbf),
    /* c64
     * base-10: 5.42101086242752217003726400434970803e-20
     * base-16: f.fffffffffffffffffffffffffff8@-17
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-65 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fbe),
    /* c65
     * base-10: 2.71050543121376108501863200217485401e-20
     * base-16: 7.fffffffffffffffffffffffffffc@-17
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-66 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fbd),
    /* c66
     * base-10: 1.35525271560688054250931600108742700e-20
     * base-16: 3.fffffffffffffffffffffffffffe@-17
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-67 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fbc),
    /* c67
     * base-10: 6.77626357803440271254658000543713504e-21
     * base-16: 1.ffffffffffffffffffffffffffff@-17
     * base-2 : 1.1111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111111e-68 */
    RTFLOAT128U_INIT_C(0, 0xffffffffffff, 0xffffffffffffffff, 0x3fbb),
};

/** CORDIC multipliers for fsin and fcos, defined by K(i)=1/sqrt(1+2^(-2i)) */
const RTFLOAT128U g_ar128FsincosCORDICConsts2[] =
{
    /* K_0
     * base-10: 7.07106781186547524400844362104848992e-1
     * base-16: b.504f333f9de6484597d89b3754a8@-1
     * base-2 : 1.0110101000001001111001100110011111110011101111001100100100001000101100101111101100010011011001101110101010010101e-1 */
    RTFLOAT128U_INIT_C(0, 0x6a09e667f3bc, 0xc908b2fb1366ea95, 0x3ffe),
    /* K_1
     * base-10: 6.32455532033675866399778708886543707e-1
     * base-16: a.1e89b12424876d9b744b679ebd80@-1
     * base-2 : 1.0100001111010001001101100010010010000100100100001110110110110011011011101000100101101100111100111101011110110000e-1 */
    RTFLOAT128U_INIT_C(0, 0x43d136248490, 0xedb36e896cf3d7b0, 0x3ffe),
    /* K_2
     * base-10: 6.13571991077896349607809087758040841e-1
     * base-16: 9.d130dd36bd1b4be3ce38c2fa55e8@-1
     * base-2 : 1.0011101000100110000110111010011011010111101000110110100101111100011110011100011100011000010111110100101010111101e-1 */
    RTFLOAT128U_INIT_C(0, 0x3a261ba6d7a3, 0x697c79c7185f4abd, 0x3ffe),
    /* K_3
     * base-10: 6.08833912517752421022113507547389826e-1
     * base-16: 9.bdc8a0ef59fef6a460db793be8a8@-1
     * base-2 : 1.0011011110111001000101000001110111101011001111111101111011010100100011000001101101101111001001110111110100010101e-1 */
    RTFLOAT128U_INIT_C(0, 0x37b9141deb3f, 0xded48c1b6f277d15, 0x3ffe),
    /* K_4
     * base-10: 6.07648256256168200929321660309522991e-1
     * base-16: 9.b8ed60c1777ac645ec45ba994918@-1
     * base-2 : 1.0011011100011101101011000001100000101110111011110101100011001000101111011000100010110111010100110010100100100011e-1 */
    RTFLOAT128U_INIT_C(0, 0x371dac182eef, 0x58c8bd88b7532923, 0x3ffe),
    /* K_5
     * base-10: 6.07351770141295959053512390387764175e-1
     * base-16: 9.b7b67d5ecb0f9eb3185c60b4f4d8@-1
     * base-2 : 1.0011011011110110110011111010101111011001011000011111001111010110011000110000101110001100000101101001111010011011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36f6cfabd961, 0xf3d6630b8c169e9b, 0x3ffe),
    /* K_6
     * base-10: 6.07277644093525999046915367337588874e-1
     * base-16: 9.b768c34f93f4616513c2419d8f20@-1
     * base-2 : 1.0011011011101101000110000110100111110010011111101000110000101100101000100111100001001000001100111011000111100100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36ed1869f27e, 0x8c2ca2784833b1e4, 0x3ffe),
    /* K_7
     * base-10: 6.07259112298892730060294541822503540e-1
     * base-16: 9.b75554b859077bd2a38fd31394f0@-1
     * base-2 : 1.0011011011101010101010101001011100001011001000001110111101111010010101000111000111111010011000100111001010011110e-1 */
    RTFLOAT128U_INIT_C(0, 0x36eaaa970b20, 0xef7a5471fa62729e, 0x3ffe),
    /* K_8
     * base-10: 6.07254479332562329717398086325156546e-1
     * base-16: 9.b7507911536845cc04aea4f6eba0@-1
     * base-2 : 1.0011011011101010000011110010001000101010011011010000100010111001100000001001010111010100100111101101110101110100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36ea0f222a6d, 0x08b98095d49edd74, 0x3ffe),
    /* K_9
     * base-10: 6.07253321089875163343435198563766406e-1
     * base-16: 9.b74f42277e91f21041fa5fc3b078@-1
     * base-2 : 1.0011011011101001111010000100010011101111110100100011111001000010000010000011111101001011111110000111011000001111e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9e844efd2, 0x3e42083f4bf8760f, 0x3ffe),
    /* K_10
     * base-10: 6.07253031529134335402284654661528154e-1
     * base-16: 9.b74ef46d082573a3ecafb1c8b090@-1
     * base-2 : 1.0011011011101001110111101000110110100001000001001010111001110100011111011001010111110110001110010001011000010010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9de8da104, 0xae747d95f6391612, 0x3ffe),
    /* K_11
     * base-10: 6.07252959138944813630351797637571092e-1
     * base-16: 9.b74ee0fe6a76e56c9a04725fa1e8@-1
     * base-2 : 1.0011011011101001110111000001111111001101010011101101110010101101100100110100000010001110010010111111010000111101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9dc1fcd4e, 0xdcad93408e4bf43d, 0x3ffe),
    /* K_12
     * base-10: 6.07252941041397163512970186424103417e-1
     * base-16: 9.b74edc22c30a0af4efe7db5b8e50@-1
     * base-2 : 1.0011011011101001110110111000010001011000011000010100000101011110100111011111110011111011011010110111000111001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db845861, 0x415e9dfcfb6b71ca, 0x3ffe),
    /* K_13
     * base-10: 6.07252936517010234128971242079738721e-1
     * base-16: 9.b74edaebd92ec0e867c3282d3d10@-1
     * base-2 : 1.0011011011101001110110110101110101111011001001011101100000011101000011001111100001100101000001011010011110100010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5d7b25, 0xd81d0cf86505a7a2, 0x3ffe),
    /* K_14
     * base-10: 6.07252935385913500729555602745276675e-1
     * base-16: 9.b74eda9e1eb7ed2e5bdf08c6e640@-1
     * base-2 : 1.0011011011101001110110110101001111000011110101101111110110100101110010110111101111100001000110001101110011001000e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db53c3d6, 0xfda5cb7be118dcc8, 0x3ffe),
    /* K_15
     * base-10: 6.07252935103139317313863198069543355e-1
     * base-16: 9.b74eda8ab01a382c6a484d5c94b0@-1
     * base-2 : 1.0011011011101001110110110101000101010110000000110100011100000101100011010100100100001001101010111001001010010110e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db515603, 0x47058d4909ab9296, 0x3ffe),
    /* K_16
     * base-10: 6.07252935032445771455825190959085565e-1
     * base-16: 9.b74eda85d472caeab6f8c3375850@-1
     * base-2 : 1.0011011011101001110110110101000010111010100011100101100101011101010101101101111100011000011001101110101100001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db50ba8e, 0x595d56df1866eb0a, 0x3ffe),
    /* K_17
     * base-10: 6.07252935014772384991058507559908879e-1
     * base-16: 9.b74eda849d88ef9a36b642f91850@-1
     * base-2 : 1.0011011011101001110110110101000010010011101100010001110111110011010001101101011011001000010111110010001100001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5093b1, 0x1df346d6c85f230a, 0x3ffe),
    /* K_18
     * base-10: 6.07252935010354038374850762858763680e-1
     * base-16: 9.b74eda844fce78c6156eb90e3828@-1
     * base-2 : 1.0011011011101001110110110101000010001001111110011100111100011000110000101010110111010111001000011100011100000101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5089f9, 0xcf18c2add721c705, 0x3ffe),
    /* K_19
     * base-10: 6.07252935009249451720797822067767802e-1
     * base-16: 9.b74eda843c5fdb110d0967f5cb10@-1
     * base-2 : 1.0011011011101001110110110101000010000111100010111111101101100010001000011010000100101100111111101011100101100010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db50878b, 0xfb6221a12cfeb962, 0x3ffe),
    /* K_20
     * base-10: 6.07252935008973305057284524081536971e-1
     * base-16: 9.b74eda84378433a3caeedcc5d478@-1
     * base-2 : 1.0011011011101001110110110101000010000110111100001000011001110100011110010101110111011011100110001011101010001111e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086f0, 0x8674795ddb98ba8f, 0x3ffe),
    /* K_21
     * base-10: 6.07252935008904268391406195660699088e-1
     * base-16: 9.b74eda84364d49c87a68268b3918@-1
     * base-2 : 1.0011011011101001110110110101000010000110110010011010100100111001000011110100110100000100110100010110011100100011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086c9, 0xa9390f4d04d16723, 0x3ffe),
    /* K_22
     * base-10: 6.07252935008887009224936613310222053e-1
     * base-16: 9.b74eda8435ff8f51a64677c5a860@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111111111000111101010001101001100100011001110111110001011010100001100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bf, 0xf1ea34c8cef8b50c, 0x3ffe),
    /* K_23
     * base-10: 6.07252935008882694433319217707273524e-1
     * base-16: 9.b74eda8435ec20b3f13e0c00d590@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111011000010000010110011111100010011111000001100000000001101010110010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bd, 0x84167e27c1801ab2, 0x3ffe),
    /* K_24
     * base-10: 6.07252935008881615735414868805578290e-1
     * base-16: 9.b74eda8435e7450c83fbf10e69f0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001110100010100001100100000111111101111110001000011100110100111110e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xe8a1907f7e21cd3e, 0x3ffe),
    /* K_25
     * base-10: 6.07252935008881346060938781580094585e-1
     * base-16: 9.b74eda8435e60e22a8ab6a51bb98@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001100000111000100010101010001010101101101010010100011011101110011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xc1c455156d4a3773, 0x3ffe),
    /* K_26
     * base-10: 6.07252935008881278642319759773719879e-1
     * base-16: 9.b74eda8435e5c06831d748a28ec8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011100000001101000001100011101011101001000101000101000111011001e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb80d063ae91451d9, 0x3ffe),
    /* K_27
     * base-10: 6.07252935008881261787665004322125961e-1
     * base-16: 9.b74eda8435e5acf994224036c380@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010110011111001100101000010001001000000001101101100001110000e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb59f32844806d870, 0x3ffe),
    /* K_28
     * base-10: 6.07252935008881257574001315459227506e-1
     * base-16: 9.b74eda8435e5a81decb4fe1bd0b0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010100000011101111011001011010011111110000110111101000010110e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb503bd969fc37a16, 0x3ffe),
    /* K_29
     * base-10: 6.07252935008881256520585393243502941e-1
     * base-16: 9.b74eda8435e5a6e702d9ad951400@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011011100111000000101101100110101101100101010001010000000e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4dce05b35b2a280, 0x3ffe),
    /* K_30
     * base-10: 6.07252935008881256257231412689571847e-1
     * base-16: 9.b74eda8435e5a6994862d97364d8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011010011001010010000110001011011001011100110110010011011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4d3290c5b2e6c9b, 0x3ffe),
    /* K_31
     * base-10: 6.07252935008881256191392917551089098e-1
     * base-16: 9.b74eda8435e5a685d9c5246af910@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011010000101110110011100010100100100011010101111100100010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4d0bb38a48d5f22, 0x3ffe),
    /* K_32
     * base-10: 6.07252935008881256174933293766468435e-1
     * base-16: 9.b74eda8435e5a680fe1db728de20@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011010000000111111100001110110110111001010001101111000100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4d01fc3b6e51bc4, 0x3ffe),
    /* K_33
     * base-10: 6.07252935008881256170818387820313317e-1
     * base-16: 9.b74eda8435e5a67fc733dbd85768@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111110001110011001111011011110110000101011101101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cff8e67b7b0aed, 0x3ffe),
    /* K_34
     * base-10: 6.07252935008881256169789661333774610e-1
     * base-16: 9.b74eda8435e5a67f7979650435c0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111011110010111100101100101000001000011010111000e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfef2f2ca086b8, 0x3ffe),
    /* K_35
     * base-10: 6.07252935008881256169532479712139957e-1
     * base-16: 9.b74eda8435e5a67f660ac74f2d58@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111011001100000101011000111010011110010110101011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfecc158e9e5ab, 0x3ffe),
    /* K_36
     * base-10: 6.07252935008881256169468184306731318e-1
     * base-16: 9.b74eda8435e5a67f612f1fe1eb40@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111011000010010111100011111111000011110101101000e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfec25e3fc3d68, 0x3ffe),
    /* K_37
     * base-10: 6.07252935008881256169452110455379231e-1
     * base-16: 9.b74eda8435e5a67f5ff836069ac0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111111100000110110000001101001101011000e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebff06c0d358, 0x3ffe),
    /* K_38
     * base-10: 6.07252935008881256169448091992541305e-1
     * base-16: 9.b74eda8435e5a67f5faa7b8fc6a8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111010101001111011100011111100011010101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf54f71f8d5, 0x3ffe),
    /* K_39
     * base-10: 6.07252935008881256169447087376831896e-1
     * base-16: 9.b74eda8435e5a67f5f970cf211a8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001011100001100111100100001000110101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf2e19e4235, 0x3ffe),
    /* K_40
     * base-10: 6.07252935008881256169446836222904544e-1
     * base-16: 9.b74eda8435e5a67f5f92314aa468@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001001000110001010010101010010001101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf24629548d, 0x3ffe),
    /* K_41
     * base-10: 6.07252935008881256169446773434422802e-1
     * base-16: 9.b74eda8435e5a67f5f90fa60c920@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000011111010011000001100100100100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21f4c1924, 0x3ffe),
    /* K_42
     * base-10: 6.07252935008881256169446757737302390e-1
     * base-16: 9.b74eda8435e5a67f5f90aca65250@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010101100101001100101001001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21594ca4a, 0x3ffe),
    /* K_43
     * base-10: 6.07252935008881256169446753813022336e-1
     * base-16: 9.b74eda8435e5a67f5f909937b4a0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010011001001101111011010010100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21326f694, 0x3ffe),
    /* K_44
     * base-10: 6.07252935008881256169446752831952370e-1
     * base-16: 9.b74eda8435e5a67f5f90945c0d38@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010100010111000000110100111e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf2128b81a7, 0x3ffe),
    /* K_45
     * base-10: 6.07252935008881256169446752586684903e-1
     * base-16: 9.b74eda8435e5a67f5f9093252360@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010011001001010010001101100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21264a46c, 0x3ffe),
    /* K_46
     * base-10: 6.07252935008881256169446752525368108e-1
     * base-16: 9.b74eda8435e5a67f5f9092d768f0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010110101110110100011110e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf2125aed1e, 0x3ffe),
    /* K_47
     * base-10: 6.07252935008881256169446752510038958e-1
     * base-16: 9.b74eda8435e5a67f5f9092c3fa58@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010110000111111101001011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf212587f4b, 0x3ffe),
    /* K_48
     * base-10: 6.07252935008881256169446752506206742e-1
     * base-16: 9.b74eda8435e5a67f5f9092bf1eb8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111110001111010111e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257e3d7, 0x3ffe),
    /* K_49
     * base-10: 6.07252935008881256169446752505248689e-1
     * base-16: 9.b74eda8435e5a67f5f9092bde7d0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011110011111010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257bcfa, 0x3ffe),
    /* K_50
     * base-10: 6.07252935008881256169446752505009199e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd9a18@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011001101000011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b343, 0x3ffe),
    /* K_51
     * base-10: 6.07252935008881256169446752504949399e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd86b0@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000011010110e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b0d6, 0x3ffe),
    /* K_52
     * base-10: 6.07252935008881256169446752504934473e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd81d8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000111011e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b03b, 0x3ffe),
    /* K_53
     * base-10: 6.07252935008881256169446752504930814e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd80a8@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000010101e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b015, 0x3ffe),
    /* K_54
     * base-10: 6.07252935008881256169446752504929947e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8060@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001100e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00c, 0x3ffe),
    /* K_55
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_56
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_57
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_58
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_59
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_60
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_61
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_62
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_63
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_64
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_65
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_66
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
    /* K_67
     * base-10: 6.07252935008881256169446752504929755e-1
     * base-16: 9.b74eda8435e5a67f5f9092bd8050@-1
     * base-2 : 1.0011011011101001110110110101000010000110101111001011010011001111111010111111001000010010010101111011000000001010e-1 */
    RTFLOAT128U_INIT_C(0, 0x36e9db5086bc, 0xb4cfebf21257b00a, 0x3ffe),
};

#endif /* !VBOX_INCLUDED_SRC_source_extF80_sincos_h */
