/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWL2Ecore.java,v 1.3 2007/04/03 09:50:42 lzhang Exp $
 */
package org.eclipse.eodm.owl.transformer;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.EcoreResourceFactoryImpl;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.owlbase.EnumeratedClass;
import org.eclipse.eodm.owl.owlbase.OWLClass;
import org.eclipse.eodm.owl.owlbase.OWLDataRange;
import org.eclipse.eodm.owl.owlbase.OWLGraph;
import org.eclipse.eodm.owl.owlbase.OWLObjectProperty;
import org.eclipse.eodm.owl.owlbase.Property;
import org.eclipse.eodm.owl.reasoner.OWLReasonerException;
import org.eclipse.eodm.owl.reasoner.OWLTaxonomyReasoner;
import org.eclipse.eodm.owl.reasoner.structural.OWLStructuralSubsumptionReasoner;
import org.eclipse.eodm.owl.resource.parser.OWLDocument;
import org.eclipse.eodm.owl.resource.parser.OWLParser;
import org.eclipse.eodm.owl.resource.parser.impl.OWLDocumentImpl;
import org.eclipse.eodm.owl.resource.parser.impl.OWLOntologyGraph;
import org.eclipse.eodm.owl.resource.parser.impl.OWLParserImpl;
import org.eclipse.eodm.owl.transformer.EODMOWLTransformerException;
import org.eclipse.eodm.rdf.rdfbase.PlainLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfbase.URIReference;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;
import org.eclipse.eodm.rdf.rdfs.RDFSDatatype;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;

public class OWL2Ecore {
	
	public static void owl2ecore(String owlfilePath, String ecoreFilePath, Map options) throws EODMOWLTransformerException  {
		EPackage ep = owl2ecore(owlfilePath,options);
		
		try {
			saveEcore(ep,ecoreFilePath);
		} catch (IOException e) {
			throw new EODMOWLTransformerException("Fail to save ecore file:" + e.getMessage());
		}
	}
	
	public static EPackage owl2ecore(String owlFilePath, Map options) throws EODMOWLTransformerException {
		
		OWLGraph graph = null;
		
		try {
			graph = loadOWLGraph(owlFilePath);
		} catch (IOException e) {
			throw new EODMOWLTransformerException("Fail to load owl file:" + e.getMessage());
		}
		
		return owl2ecore(graph,options);
	}
	
	
	public static EPackage owl2ecore(OWLGraph graph, Map options) throws EODMOWLTransformerException {
		OWLTaxonomyReasoner reasoner = new OWLStructuralSubsumptionReasoner();
	
		try {
			reasoner.initialize(graph);
		} catch (OWLReasonerException e) {
			throw new EODMOWLTransformerException("Fail to inintialize OWL Reasoner: " + e.getMessage());
		}
		
		
	    EPackage epackage = EcoreFactory.eINSTANCE.createEPackage();

		//set info for ePackage
		URIReference graphUri = graph.getGraphName();
		
		if(graphUri != null) {
			if(graphUri.getNamespace() != null ) {
				//@TODO
				//ePackage.setNsPrefix( graphUri.getNamespace().getNamespaceURIRef());
			}
			epackage.setName(graphUri.getUri().getName());
		}
	    
	    Map owl2ecoreMap = new HashMap(); 
	    
	    try {
	    	
			// OWLClass -> EClass
			List classList = graph.getTypeResources(RDFS.C_CLASS_STR);
			
			for(Iterator ri = classList.iterator();ri.hasNext(); ) {
				RDFSClass rclass = (RDFSClass) ri.next();
	
				// filter out Restriction anonymous owlclass
				if (!isNamedObject(rclass)) {
					continue;
				}
	
	            // fitler out owl:thing
	            if(rclass.getURI().equals(OWL_THING) && rclass.getSubClass().isEmpty() ) {
	                continue;
	            }
	            
	            EClassifier eclass = owl2eclassifier(rclass, owl2ecoreMap);
	            
	            if(eclass==null) 
	            	continue;
	            
	            if(rclass.canAsType(OWLClass.class) ) { // instanceof OWLClass) {
	            	OWLClass oclass = (OWLClass)rclass.asType(OWLClass.class);
		            // RDFSSubClassOf -> EClass.eSuperType
					for (Iterator itSuper = reasoner.getAncestorClasses(oclass).iterator();itSuper.hasNext();) {
						OWLClass superClass = (OWLClass) itSuper.next();
						
						if( superClass != oclass) {
						    EClass eSuperClass = (EClass)owl2eclassifier(superClass,owl2ecoreMap);
						    
						    // check whether eclass if father of eSuperClass
						    if ( eSuperClass.getESuperTypes().contains(eclass) ) {
						        eSuperClass.getESuperTypes().remove(eclass);
						        
						        EAnnotation annotation1 = EcoreFactory.eINSTANCE.createEAnnotation();
						        annotation1.setSource("The " + eclass.getName() + " class is equivalent with " + eSuperClass.getName() + ".");
						        eclass.getEAnnotations().add(annotation1);
						        
						        EAnnotation annotation2 = EcoreFactory.eINSTANCE.createEAnnotation();
						        annotation2.setSource("The " + eSuperClass.getName() + " class is equivalent with " + eclass.getName() + ".");
						        eSuperClass.getEAnnotations().add(annotation2);
						    } else {
						        ((EClass)eclass).getESuperTypes().add(eSuperClass);
						    }
						}
					}
		            			
					
					// property -> EReference, Attribute
					for(Iterator itP= oclass.getPropertyForDomain().iterator();itP.hasNext();){
						Property p = (Property) ((RDFSResource)itP.next()).asType(Property.class);
						
						EStructuralFeature ep = prop2EStructuralFeature(p,owl2ecoreMap);

						if(p instanceof OWLObjectProperty) {
							List lst = ((OWLObjectProperty)p).getOWLinverseOf();
							for(Iterator itInv = lst.iterator();itInv.hasNext();) {
								OWLObjectProperty pinv = (OWLObjectProperty)itInv.next();
								if(pinv!=null && !pinv.getRDFSdomain().isEmpty() ) {
									EReference epinv = (EReference)prop2EStructuralFeature(pinv,owl2ecoreMap);
									epinv.setEOpposite((EReference)ep);
									((EReference)ep).setEOpposite(epinv);
								}
							}
						}
						
						// set domain
						((EClass)eclass).getEStructuralFeatures().add(ep);
						
						// set range
						List rlst = p.getRDFSrange();
						if(!rlst.isEmpty()) {
							EClassifier erange = owl2eclassifier( (RDFSResource)p.getRDFSrange().get(0), owl2ecoreMap);
							ep.setEType(erange);
						}
						
					}
	            }
	            
				epackage.getEClassifiers().add(eclass);
				
			}		


	    } catch (UnsupportedViewTypeException e) {
	    	throw new EODMOWLTransformerException("Cannot load resources from graph: " + e.getMessage());
	    }
	    
		return epackage;

	}

	private static EStructuralFeature prop2EStructuralFeature(Property p, Map owl2ecoreMap) throws UnsupportedViewTypeException {
		EStructuralFeature ep = (EStructuralFeature)owl2ecoreMap.get(p);

		if(ep == null) {
			if(p instanceof OWLObjectProperty) {
				ep = EcoreFactory.eINSTANCE.createEReference();
				OWLClass range = p.getRDFSrange().isEmpty() ? null : 
					(OWLClass) ((RDFSResource)p.getRDFSrange().get(0)).asType(OWLClass.class);
				if(range != null) {
					// if property without range, leave it as null
					EClass erange = (EClass)owl2ecoreMap.get(range);
					ep.setEType(erange);
				}
				
				// property attribute to annotation
				if( ((OWLObjectProperty)p).isIsTransitive() ){
					EAnnotation ea = EcoreFactory.eINSTANCE.createEAnnotation();
					ea.setSource("The property is Transitive.");
					ep.getEAnnotations().add(ea);
				}
				
				if( ((OWLObjectProperty)p).isIsSymmetric() ) {
					EAnnotation ea = EcoreFactory.eINSTANCE.createEAnnotation();
					ea.setSource("The property is Functional.");
					ep.getEAnnotations().add(ea);
				}
				
				if( ((OWLObjectProperty)p).isIsInverseFunctional() ) {
					EAnnotation ea = EcoreFactory.eINSTANCE.createEAnnotation();
					ea.setSource("The property is Inverse Functional.");
					ep.getEAnnotations().add(ea);
				}

			} else {
				// OWLDatatype Property
				ep = EcoreFactory.eINSTANCE.createEAttribute();
			}
			
			ep.setName( getName(p) );
			ep.getEAnnotations().addAll( createEAnnotation(p));
						
			// subPropertyOf will be in eannoations
			for(Iterator itSuper = p.getRDFSsubPropertyOf().iterator();itSuper.hasNext();) {
			    Property superP = (Property) ((RDFSResource)itSuper.next()).asType(Property.class);
			    if (superP != null) {
			        EAnnotation ea = EcoreFactory.eINSTANCE.createEAnnotation();
			        ea.setSource("The property is subPropertyOf " + getName(superP) + ".");
			        ep.getEAnnotations().add(ea);
			    }
			}
			//	property attribute to annotation
			if( p.isIsFunctional() ) {
				EAnnotation ea = EcoreFactory.eINSTANCE.createEAnnotation();
				ea.setSource("The property is Functional.");
				ep.getEAnnotations().add(ea);
			}
			
			owl2ecoreMap.put(p,ep);
		}

		return ep;
	}
	
	// test whether it is a blanknode
    private static boolean isNamedObject(RDFSResource c){
    	String nid = c.getNodeID();
    	return (nid==null || nid.length()==0);
    }
	
    private static EClassifier owl2eclassifier(RDFSResource oclass, Map owc2ecMap) {
    	EClassifier eclass = (EClassifier)owc2ecMap.get(oclass);
    	
    	if(eclass == null) {
    		if(oclass instanceof OWLDataRange || oclass instanceof EnumeratedClass) {
    			// enumerate, datarange -> eenum
    			eclass = oneof2EEnum((RDFSClass)oclass, owc2ecMap);   			
    		}
    		else if(oclass instanceof RDFSDatatype) {
	            RDFSDatatype dt = (RDFSDatatype) oclass;
	            String dtURI = dt.getURI();
	            if( dtURI != null ) {
	            	if(dtURI.equals(XSD_BOOLEAN) )
	            		eclass = EcorePackage.eINSTANCE.getEBoolean();
	            	else if(dtURI.equals(XSD_FLOAT) )
	            		eclass = EcorePackage.eINSTANCE.getEFloat();
	            	else if(dtURI.equals(XSD_INT))
	            		eclass = EcorePackage.eINSTANCE.getEInt();
	            	else if(dtURI.equals(XSD_BYTE))
	            		eclass = EcorePackage.eINSTANCE.getEByte();
	            	else if(dtURI.equals(XSD_LONG))
	            		eclass = EcorePackage.eINSTANCE.getELong();
	            	else if(dtURI.equals(XSD_DOUBLE))
	            		eclass = EcorePackage.eINSTANCE.getEDouble();
	            	else if(dtURI.equals(XSD_SHORT))
	            		eclass = EcorePackage.eINSTANCE.getEShort();
	            	else if(dtURI.equals(XSD_STRING)|| dtURI.equals(RDFS.C_LITERAL_STR) 
							|| dtURI.equals(RDF.C_XMLLITERAL_STR))
	            		eclass = EcorePackage.eINSTANCE.getEString();
//	            	else if( dt.getRDFSisDefinedBy().size()>0 && OWLDataRange.class.isInstance(
//	            			dt.getRDFSisDefinedBy().get(0)) ) {
//	            		eclass = oneof2EEnum((RDFSClass) dt.getRDFSisDefinedBy().get(0), owc2ecMap);
//	            	}
	            	else {
		                // Create a user-defined EDatatType
		                eclass = EcoreFactory.eINSTANCE.createEDataType();
		                eclass.setName( getName(dt) );
		                eclass.getEAnnotations().addAll( createEAnnotation(dt) );
	            	}
	            }
			} 
			else {
				// owlclass
		    	eclass = EcoreFactory.eINSTANCE.createEClass();
		    	
		    	// create name for eclass    	
		    	eclass.setName(getName(oclass));
		    	eclass.getEAnnotations().addAll( createEAnnotation(oclass) );
			}
	    	
			if(eclass!=null)
				owc2ecMap.put(oclass,eclass);
    	}
    	
    	return eclass;
    }
	
    private static String getName(RDFSResource r) {
    	// create name for eclass
    	String name = null;
    	
    	if( !r.getUriRef().isEmpty() ) {
    		URIReference uriRef = (URIReference)r.getUriRef().get(0);
    		name = (uriRef.getFragmentIdentifier()==null) ? 
    				uriRef.getUri().getName() : uriRef.getFragmentIdentifier().getName();
    	} else {
    		name = r.getURI();
    	}
    	
    	return name;
    }
    
    
    private static List createEAnnotation(RDFSResource rs){
    	List lst = new ArrayList(); 
    	
    	// create comments as annotations
    	for(Iterator it = rs.getRDFScomment().iterator();it.hasNext();) {
    		EAnnotation annotation = EcoreFactory.eINSTANCE.createEAnnotation();
    		annotation.setSource("comment - " + ((PlainLiteral) it.next()).getLexicalForm());
    		lst.add(annotation);
    	}

    	// create isDefinedBy as annotations
    	for (Iterator it = rs.getRDFSisDefinedBy().iterator();it.hasNext();) {
    		EAnnotation annotation = EcoreFactory.eINSTANCE.createEAnnotation();
    		annotation.setSource("isDefinedBy - " + ((RDFSResource) it.next()).getURI());
    		lst.add(annotation);
    	}

    	// create Label as annotations
    	for (Iterator it = rs.getRDFSlabel().iterator();it.hasNext();) {
    		EAnnotation annotation = EcoreFactory.eINSTANCE.createEAnnotation();
    		annotation.setSource("label - " + ((PlainLiteral) it.next()).getLexicalForm());
    		lst.add(annotation);
    	}

    	if(rs instanceof OWLClass) {
    		OWLClass c = (OWLClass)rs;
    		
			// record disjointWith
			for (Iterator it = c.getOWLdisjointWith().iterator();it.hasNext();) {
			    OWLClass owlc = (OWLClass) it.next();
			    if (owlc != null) {
			        EAnnotation annotation = EcoreFactory.eINSTANCE.createEAnnotation();
			        annotation.setSource("The " + c.getURI() + " class is disjoint with " + owlc.getURI() + ".");
			        lst.add(annotation);
			    }
			}
    	}
    	
    	return lst;
    }
    
    
	private static EEnum oneof2EEnum(RDFSClass enumclass, Map owl2eclassifer) {
		
		EEnum eenum = (EEnum)owl2eclassifer.get(enumclass);

		if(eenum==null){
			List memberList = null;
			
			if(enumclass instanceof OWLDataRange) {
				memberList = ((OWLDataRange)enumclass).getOWLDataRangeOneOf();
			} else {
				memberList = ((EnumeratedClass)enumclass).getOWLoneOf();
			}
			
			// if memberList.size > 0, create EEnum
			if (!memberList.isEmpty()) {
				eenum = EcoreFactory.eINSTANCE.createEEnum();
				eenum.setName( getName(enumclass));
				eenum.getEAnnotations().addAll( createEAnnotation(enumclass) );

				// create enumliterals from enumclass members
				int intValue = 0;
				for (Iterator it = memberList.iterator();it.hasNext();) {
					RDFSResource r = (RDFSResource) (it.next());
					
					// test whether contains label "EEnum"
					String enumeration = (r instanceof RDFSLiteral ? 
							((RDFSLiteral) r).getLexicalForm() : r.getURI());
		
					// create EEnumLiteral
					EEnumLiteral eliteral = EcoreFactory.eINSTANCE.createEEnumLiteral();
					eliteral.getEAnnotations().addAll( createEAnnotation(r) );

					eliteral.setName(enumeration);
					eliteral.setValue(intValue);
					eenum.getELiterals().add(eliteral);
					intValue++;
				}
			}
		}

		return eenum;
	}
    
    static OWLGraph loadOWLGraph(String owlFile) throws IOException {
    	OWLParser parser = new OWLParserImpl();
    	OWLDocument document = new OWLDocumentImpl(owlFile, null, true);
    	parser.addOWLDocument(document);
    	OWLOntologyGraph[] ontoList = parser.parseOWLDocument(document);
    	
    	return ontoList[0].getgraph();
    	
    }
	
    static void saveEcore(EPackage epackage, String outputEcoreFile) throws IOException {
        //save ecore model
		ResourceSet resourceSet = new ResourceSetImpl();
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
                .put(Resource.Factory.Registry.DEFAULT_EXTENSION,
                        new EcoreResourceFactoryImpl());

        Resource resource = resourceSet.createResource(URI
                .createFileURI(new File(outputEcoreFile).getAbsolutePath()));
        resource.getContents().add(epackage);

        resource.save(Collections.EMPTY_MAP);
    }
	
    private static String OWL_THING = "http://www.w3.org/2002/07/owl#Thing";
    
    private static String XSD_BOOLEAN = "http://www.w3.org/2001/XMLSchema#boolean";
    private static String XSD_FLOAT = "http://www.w3.org/2001/XMLSchema#float";
    private static String XSD_BYTE = "http://www.w3.org/2001/XMLSchema#byte";
    private static String XSD_INT = "http://www.w3.org/2001/XMLSchema#int";
    private static String XSD_LONG = "http://www.w3.org/2001/XMLSchema#long";
    private static String XSD_DOUBLE = "http://www.w3.org/2001/XMLSchema#double";
    private static String XSD_SHORT = "http://www.w3.org/2001/XMLSchema#short";
    //private static String XSD_INTEGER = "http://www.w3.org/2001/XMLSchema#integer";
    private static String XSD_STRING = "http://www.w3.org/2001/XMLSchema#string";
}
