/**
 * Copyright (c) 2016 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Van Cam Pham        <VanCam.PHAM@cea.fr>
 */
package org.eclipse.papyrus.designer.languages.c.codegen.lib;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.c.codegen.lib.FunctionScript;
import org.eclipse.papyrus.designer.languages.common.base.GenUtils;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Event;
import org.eclipse.uml2.uml.FinalState;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Pseudostate;
import org.eclipse.uml2.uml.PseudostateKind;
import org.eclipse.uml2.uml.Region;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.Vertex;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class TransformationUtil {
  public static Map<org.eclipse.uml2.uml.Class, StateMachine> getStateMachineClass(final org.eclipse.uml2.uml.Package pack) {
    final Map<org.eclipse.uml2.uml.Class, StateMachine> map = new HashMap<org.eclipse.uml2.uml.Class, StateMachine>();
    final Consumer<org.eclipse.uml2.uml.Class> _function = new Consumer<org.eclipse.uml2.uml.Class>() {
      @Override
      public void accept(final org.eclipse.uml2.uml.Class it) {
        StateMachine behavior = IterableExtensions.<StateMachine>head(Iterables.<StateMachine>filter(it.getOwnedBehaviors(), StateMachine.class));
        if ((behavior != null)) {
          map.put(it, behavior);
        }
      }
    };
    Iterables.<org.eclipse.uml2.uml.Class>filter(pack.getOwnedElements(), org.eclipse.uml2.uml.Class.class).forEach(_function);
    return map;
  }
  
  public static Pseudostate firstPseudoState(final Region region, final PseudostateKind kind) {
    EList<Vertex> _subvertices = region.getSubvertices();
    for (final Vertex vertex : _subvertices) {
      if ((vertex instanceof Pseudostate)) {
        Pseudostate pseudoState = ((Pseudostate) vertex);
        PseudostateKind _kind = pseudoState.getKind();
        boolean _equals = Objects.equal(_kind, kind);
        if (_equals) {
          return pseudoState;
        }
      }
    }
    return null;
  }
  
  public static List<State> findCompositeStatesInRegion(final Vertex des1, final Vertex des2) {
    ArrayList<State> ret = new ArrayList<State>();
    final ArrayList<State> path1 = new ArrayList<State>();
    if ((des1 instanceof State)) {
      path1.add(((State)des1));
    }
    State owner1 = des1.getContainer().getState();
    while ((owner1 != null)) {
      {
        path1.add(owner1);
        owner1 = owner1.getContainer().getState();
      }
    }
    Vertex owner2 = des2;
    while (((owner2 != null) && ret.isEmpty())) {
      {
        for (final State s : path1) {
          if ((Objects.equal(s.getContainer(), owner2.getContainer()) && (owner2 instanceof State))) {
            ret.add(s);
            ret.add(((State) owner2));
          }
        }
        owner2 = owner2.getContainer().getState();
      }
    }
    return ret;
  }
  
  public static State findInitialState(final Region r) {
    Pseudostate pseudoDefault = TransformationUtil.firstPseudoState(r, PseudostateKind.INITIAL_LITERAL);
    if ((pseudoDefault != null)) {
      Vertex _target = IterableExtensions.<Transition>head(pseudoDefault.getOutgoings()).getTarget();
      return ((State) _target);
    }
    return null;
  }
  
  public static String getInitialEffect(final Region r) {
    Pseudostate pseudoDefault = TransformationUtil.firstPseudoState(r, PseudostateKind.INITIAL_LITERAL);
    if ((pseudoDefault != null)) {
      Transition t = IterableExtensions.<Transition>head(pseudoDefault.getOutgoings());
      if (((t.getEffect() != null) && (t.getEffect() instanceof OpaqueBehavior))) {
        Behavior _effect = t.getEffect();
        return IterableExtensions.<String>head(((OpaqueBehavior) _effect).getBodies());
      }
    }
    return "//no initial effect is defined";
  }
  
  public static List<State> transitiveSubStates(final State parent) {
    final ArrayList<State> ret = new ArrayList<State>();
    boolean _isComposite = parent.isComposite();
    if (_isComposite) {
      EList<Region> _regions = parent.getRegions();
      for (final Region r : _regions) {
        {
          Iterables.<State>addAll(ret, Iterables.<State>filter(r.getSubvertices(), State.class));
          final Consumer<State> _function = new Consumer<State>() {
            @Override
            public void accept(final State it) {
              ret.addAll(TransformationUtil.transitiveSubStates(it));
            }
          };
          Iterables.<State>filter(r.getSubvertices(), State.class).forEach(_function);
        }
      }
    }
    return ret;
  }
  
  public static List<State> transitiveSubStates(final Region parent) {
    final ArrayList<State> ret = new ArrayList<State>();
    final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf((!(it instanceof FinalState)));
      }
    };
    final Consumer<State> _function_1 = new Consumer<State>() {
      @Override
      public void accept(final State it) {
        ret.add(it);
        ret.addAll(TransformationUtil.transitiveSubStates(it));
      }
    };
    IterableExtensions.<State>filter(Iterables.<State>filter(parent.getSubvertices(), State.class), _function).forEach(_function_1);
    return ret;
  }
  
  /**
   * Return the name of an event (remove non-ASCII characters)
   */
  public static String eventName(final Event event) {
    String _xblockexpression = null;
    {
      String name = "";
      boolean repeat = false;
      char[] _charArray = event.getName().toCharArray();
      for (final char c : _charArray) {
        if ((Character.isAlphabetic(c) || Character.isDigit(c))) {
          String _name = name;
          name = (_name + Character.valueOf(c));
          repeat = false;
        } else {
          if ((!repeat)) {
            String _name_1 = name;
            name = (_name_1 + "_");
            repeat = true;
          }
        }
      }
      _xblockexpression = name;
    }
    return _xblockexpression;
  }
  
  /**
   * Return the ID of an event
   */
  public static String eventID(final Event event) {
    String _upperCase = TransformationUtil.eventName(event).toUpperCase();
    return (_upperCase + "_ID");
  }
  
  public static boolean isSavehistory(final Region topRegion, final Region r) {
    final Function1<Pseudostate, Boolean> _function = new Function1<Pseudostate, Boolean>() {
      @Override
      public Boolean apply(final Pseudostate it) {
        PseudostateKind _kind = it.getKind();
        return Boolean.valueOf(Objects.equal(_kind, PseudostateKind.SHALLOW_HISTORY_LITERAL));
      }
    };
    int _size = IterableExtensions.size(IterableExtensions.<Pseudostate>filter(Iterables.<Pseudostate>filter(r.getSubvertices(), Pseudostate.class), _function));
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      return true;
    }
    return TransformationUtil.isSaveDeepHistory(topRegion, r);
  }
  
  public static boolean isBehaviorExist(final Behavior b) {
    if (((b != null) && (b instanceof OpaqueBehavior))) {
      boolean _isEmpty = GenUtils.getBodyFromOB(((OpaqueBehavior) b), FunctionScript.CLANGUAGE).isEmpty();
      return (!_isEmpty);
    }
    return false;
  }
  
  public static boolean hasTriggerlessTransition(final State s) {
    final Function1<Transition, Boolean> _function = new Function1<Transition, Boolean>() {
      @Override
      public Boolean apply(final Transition it) {
        final Function1<Trigger, Event> _function = new Function1<Trigger, Event>() {
          @Override
          public Event apply(final Trigger it) {
            return it.getEvent();
          }
        };
        return Boolean.valueOf(ListExtensions.<Trigger, Event>map(it.getTriggers(), _function).isEmpty());
      }
    };
    int _size = IterableExtensions.size(IterableExtensions.<Transition>filter(s.getOutgoings(), _function));
    return (_size > 0);
  }
  
  public static boolean hasTriggerlessTransition(final StateMachine sm) {
    final Function1<State, Boolean> _function = new Function1<State, Boolean>() {
      @Override
      public Boolean apply(final State it) {
        return Boolean.valueOf((!(it instanceof FinalState)));
      }
    };
    Iterable<State> states = IterableExtensions.<State>filter(Iterables.<State>filter(IterableExtensions.<Region>head(sm.getRegions()).getSubvertices(), State.class), _function);
    for (final State s : states) {
      final Function1<Transition, Boolean> _function_1 = new Function1<Transition, Boolean>() {
        @Override
        public Boolean apply(final Transition it) {
          final Function1<Trigger, Event> _function = new Function1<Trigger, Event>() {
            @Override
            public Event apply(final Trigger it) {
              return it.getEvent();
            }
          };
          return Boolean.valueOf(ListExtensions.<Trigger, Event>map(it.getTriggers(), _function).isEmpty());
        }
      };
      int _size = IterableExtensions.size(IterableExtensions.<Transition>filter(s.getOutgoings(), _function_1));
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        return true;
      }
    }
    return false;
  }
  
  public static boolean isSaveDeepHistory(final Region topRegion, final Region r) {
    final Function1<Pseudostate, Boolean> _function = new Function1<Pseudostate, Boolean>() {
      @Override
      public Boolean apply(final Pseudostate it) {
        PseudostateKind _kind = it.getKind();
        return Boolean.valueOf(Objects.equal(_kind, PseudostateKind.DEEP_HISTORY_LITERAL));
      }
    };
    int _size = IterableExtensions.size(IterableExtensions.<Pseudostate>filter(Iterables.<Pseudostate>filter(r.getSubvertices(), Pseudostate.class), _function));
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      return true;
    }
    boolean _notEquals = (!Objects.equal(r, topRegion));
    if (_notEquals) {
      Region nextRegion = r.getState().getContainer();
      return TransformationUtil.isSaveDeepHistory(topRegion, nextRegion);
    }
    return false;
  }
}
