/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * The event indicates that the target has produced some output.
 * <p>
 * Represents the {@code body} of {@code OutputEvent} defined in spec.
 */
@SuppressWarnings("all")
public class OutputEventArguments {
  /**
   * The output category. If not specified, 'console' is assumed.
   * <p>
   * This is an optional property.
   * <p>
   * Possible values include - but not limited to those defined in {@link OutputEventArgumentsCategory}
   */
  private String category;

  /**
   * The output to report.
   * <p>
   * ANSI escape sequences may be used to influence text color and styling if `supportsANSIStyling` is present in
   * both the adapter's `Capabilities` and the client's `InitializeRequestArguments`. A client may strip any
   * unrecognized ANSI sequences.
   * <p>
   * If the `supportsANSIStyling` capabilities are not both true, then the client should display the output literally.
   */
  @NonNull
  private String output;

  /**
   * Support for keeping an output log organized by grouping related messages.
   * <p>
   * This is an optional property.
   */
  private OutputEventArgumentsGroup group;

  /**
   * If an attribute `variablesReference` exists and its value is &gt; 0, the output
   * contains objects which can be retrieved by passing `variablesReference` to the
   * `variables` request as long as execution remains suspended. See 'Lifetime of
   * Object References' in the {@link DebugProtocol#Overview} section for details.
   * <p>
   * This is an optional property.
   */
  private Integer variablesReference;

  /**
   * The source location where the output was produced.
   * <p>
   * This is an optional property.
   */
  private Source source;

  /**
   * The source location's line where the output was produced.
   * <p>
   * This is an optional property.
   */
  private Integer line;

  /**
   * The position in `line` where the output was produced. It is measured in UTF-16 code units and the
   * client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines whether it is 0- or 1-based.
   * <p>
   * This is an optional property.
   */
  private Integer column;

  /**
   * Additional data to report. For the 'telemetry' category the data is sent to telemetry, for the other
   * categories the data is shown in JSON format.
   * <p>
   * This is an optional property.
   */
  private Object data;

  /**
   * A reference that allows the client to request the location where the new value is declared. For example,
   * if the logged value is function pointer, the adapter may be able to look up the function's location. This should
   * be present only if the adapter is likely to be able to resolve the location.
   * <p>
   * This reference shares the same lifetime as the `variablesReference`.
   * See 'Lifetime of Object References' in the Overview section for details.
   * <p>
   * This is an optional property.
   */
  @ProtocolSince("1.68")
  private Integer locationReference;

  /**
   * The output category. If not specified, 'console' is assumed.
   * <p>
   * This is an optional property.
   * <p>
   * Possible values include - but not limited to those defined in {@link OutputEventArgumentsCategory}
   */
  public String getCategory() {
    return this.category;
  }

  /**
   * The output category. If not specified, 'console' is assumed.
   * <p>
   * This is an optional property.
   * <p>
   * Possible values include - but not limited to those defined in {@link OutputEventArgumentsCategory}
   */
  public void setCategory(final String category) {
    this.category = category;
  }

  /**
   * The output to report.
   * <p>
   * ANSI escape sequences may be used to influence text color and styling if `supportsANSIStyling` is present in
   * both the adapter's `Capabilities` and the client's `InitializeRequestArguments`. A client may strip any
   * unrecognized ANSI sequences.
   * <p>
   * If the `supportsANSIStyling` capabilities are not both true, then the client should display the output literally.
   */
  @NonNull
  public String getOutput() {
    return this.output;
  }

  /**
   * The output to report.
   * <p>
   * ANSI escape sequences may be used to influence text color and styling if `supportsANSIStyling` is present in
   * both the adapter's `Capabilities` and the client's `InitializeRequestArguments`. A client may strip any
   * unrecognized ANSI sequences.
   * <p>
   * If the `supportsANSIStyling` capabilities are not both true, then the client should display the output literally.
   */
  public void setOutput(@NonNull final String output) {
    this.output = Preconditions.checkNotNull(output, "output");
  }

  /**
   * Support for keeping an output log organized by grouping related messages.
   * <p>
   * This is an optional property.
   */
  public OutputEventArgumentsGroup getGroup() {
    return this.group;
  }

  /**
   * Support for keeping an output log organized by grouping related messages.
   * <p>
   * This is an optional property.
   */
  public void setGroup(final OutputEventArgumentsGroup group) {
    this.group = group;
  }

  /**
   * If an attribute `variablesReference` exists and its value is &gt; 0, the output
   * contains objects which can be retrieved by passing `variablesReference` to the
   * `variables` request as long as execution remains suspended. See 'Lifetime of
   * Object References' in the {@link DebugProtocol#Overview} section for details.
   * <p>
   * This is an optional property.
   */
  public Integer getVariablesReference() {
    return this.variablesReference;
  }

  /**
   * If an attribute `variablesReference` exists and its value is &gt; 0, the output
   * contains objects which can be retrieved by passing `variablesReference` to the
   * `variables` request as long as execution remains suspended. See 'Lifetime of
   * Object References' in the {@link DebugProtocol#Overview} section for details.
   * <p>
   * This is an optional property.
   */
  public void setVariablesReference(final Integer variablesReference) {
    this.variablesReference = variablesReference;
  }

  /**
   * The source location where the output was produced.
   * <p>
   * This is an optional property.
   */
  public Source getSource() {
    return this.source;
  }

  /**
   * The source location where the output was produced.
   * <p>
   * This is an optional property.
   */
  public void setSource(final Source source) {
    this.source = source;
  }

  /**
   * The source location's line where the output was produced.
   * <p>
   * This is an optional property.
   */
  public Integer getLine() {
    return this.line;
  }

  /**
   * The source location's line where the output was produced.
   * <p>
   * This is an optional property.
   */
  public void setLine(final Integer line) {
    this.line = line;
  }

  /**
   * The position in `line` where the output was produced. It is measured in UTF-16 code units and the
   * client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines whether it is 0- or 1-based.
   * <p>
   * This is an optional property.
   */
  public Integer getColumn() {
    return this.column;
  }

  /**
   * The position in `line` where the output was produced. It is measured in UTF-16 code units and the
   * client capability {@link InitializeRequestArguments#getColumnsStartAt1} determines whether it is 0- or 1-based.
   * <p>
   * This is an optional property.
   */
  public void setColumn(final Integer column) {
    this.column = column;
  }

  /**
   * Additional data to report. For the 'telemetry' category the data is sent to telemetry, for the other
   * categories the data is shown in JSON format.
   * <p>
   * This is an optional property.
   */
  public Object getData() {
    return this.data;
  }

  /**
   * Additional data to report. For the 'telemetry' category the data is sent to telemetry, for the other
   * categories the data is shown in JSON format.
   * <p>
   * This is an optional property.
   */
  public void setData(final Object data) {
    this.data = data;
  }

  /**
   * A reference that allows the client to request the location where the new value is declared. For example,
   * if the logged value is function pointer, the adapter may be able to look up the function's location. This should
   * be present only if the adapter is likely to be able to resolve the location.
   * <p>
   * This reference shares the same lifetime as the `variablesReference`.
   * See 'Lifetime of Object References' in the Overview section for details.
   * <p>
   * This is an optional property.
   */
  @ProtocolSince("1.68")
  public Integer getLocationReference() {
    return this.locationReference;
  }

  /**
   * A reference that allows the client to request the location where the new value is declared. For example,
   * if the logged value is function pointer, the adapter may be able to look up the function's location. This should
   * be present only if the adapter is likely to be able to resolve the location.
   * <p>
   * This reference shares the same lifetime as the `variablesReference`.
   * See 'Lifetime of Object References' in the Overview section for details.
   * <p>
   * This is an optional property.
   */
  @ProtocolSince("1.68")
  public void setLocationReference(final Integer locationReference) {
    this.locationReference = locationReference;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("category", this.category);
    b.add("output", this.output);
    b.add("group", this.group);
    b.add("variablesReference", this.variablesReference);
    b.add("source", this.source);
    b.add("line", this.line);
    b.add("column", this.column);
    b.add("data", this.data);
    b.add("locationReference", this.locationReference);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    OutputEventArguments other = (OutputEventArguments) obj;
    if (this.category == null) {
      if (other.category != null)
        return false;
    } else if (!this.category.equals(other.category))
      return false;
    if (this.output == null) {
      if (other.output != null)
        return false;
    } else if (!this.output.equals(other.output))
      return false;
    if (this.group == null) {
      if (other.group != null)
        return false;
    } else if (!this.group.equals(other.group))
      return false;
    if (this.variablesReference == null) {
      if (other.variablesReference != null)
        return false;
    } else if (!this.variablesReference.equals(other.variablesReference))
      return false;
    if (this.source == null) {
      if (other.source != null)
        return false;
    } else if (!this.source.equals(other.source))
      return false;
    if (this.line == null) {
      if (other.line != null)
        return false;
    } else if (!this.line.equals(other.line))
      return false;
    if (this.column == null) {
      if (other.column != null)
        return false;
    } else if (!this.column.equals(other.column))
      return false;
    if (this.data == null) {
      if (other.data != null)
        return false;
    } else if (!this.data.equals(other.data))
      return false;
    if (this.locationReference == null) {
      if (other.locationReference != null)
        return false;
    } else if (!this.locationReference.equals(other.locationReference))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.category== null) ? 0 : this.category.hashCode());
    result = prime * result + ((this.output== null) ? 0 : this.output.hashCode());
    result = prime * result + ((this.group== null) ? 0 : this.group.hashCode());
    result = prime * result + ((this.variablesReference== null) ? 0 : this.variablesReference.hashCode());
    result = prime * result + ((this.source== null) ? 0 : this.source.hashCode());
    result = prime * result + ((this.line== null) ? 0 : this.line.hashCode());
    result = prime * result + ((this.column== null) ? 0 : this.column.hashCode());
    result = prime * result + ((this.data== null) ? 0 : this.data.hashCode());
    return prime * result + ((this.locationReference== null) ? 0 : this.locationReference.hashCode());
  }
}
