package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Ipam}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableIpam.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableIpam implements Ipam {
  private final String driver;
  private final @Nullable List<IpamConfig> config;
  private final @Nullable Map<String, String> options;

  private ImmutableIpam(
      String driver,
      @Nullable List<IpamConfig> config,
      @Nullable Map<String, String> options) {
    this.driver = driver;
    this.config = config;
    this.options = options;
  }

  /**
   * @return The value of the {@code driver} attribute
   */
  @JsonProperty("Driver")
  @Override
  public String driver() {
    return driver;
  }

  /**
   * @return The value of the {@code config} attribute
   */
  @JsonProperty("Config")
  @Override
  public @Nullable List<IpamConfig> config() {
    return config;
  }

  /**
   * @return The value of the {@code options} attribute
   */
  @JsonProperty("Options")
  @Override
  public @Nullable Map<String, String> options() {
    return options;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Ipam#driver() driver} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for driver
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableIpam withDriver(String value) {
    String newValue = Objects.requireNonNull(value, "driver");
    if (this.driver.equals(newValue)) return this;
    return new ImmutableIpam(newValue, this.config, this.options);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Ipam#config() config}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableIpam withConfig(@Nullable IpamConfig... elements) {
    if (elements == null) {
      if (this.config == null) return this;
      return new ImmutableIpam(this.driver, null, this.options);
    }
    List<IpamConfig> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableIpam(this.driver, newValue, this.options);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Ipam#config() config}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of config elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableIpam withConfig(@Nullable Iterable<? extends IpamConfig> elements) {
    if (this.config == elements) return this;
    @Nullable List<IpamConfig> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableIpam(this.driver, newValue, this.options);
  }

  /**
   * Copy the current immutable object by replacing the {@link Ipam#options() options} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the options map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableIpam withOptions(@Nullable Map<String, ? extends String> entries) {
    if (this.options == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableIpam(this.driver, this.config, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableIpam} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableIpam
        && equalsByValue((ImmutableIpam) another);
  }

  private boolean equalsByValue(ImmutableIpam another) {
    return driver.equals(another.driver)
        && Objects.equals(config, another.config)
        && Objects.equals(options, another.options);
  }

  /**
   * Computes a hash code from attributes: {@code driver}, {@code config}, {@code options}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + driver.hashCode();
    h += (h << 5) + Objects.hashCode(config);
    h += (h << 5) + Objects.hashCode(options);
    return h;
  }

  /**
   * Prints the immutable value {@code Ipam} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Ipam{"
        + "driver=" + driver
        + ", config=" + config
        + ", options=" + options
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Ipam} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Ipam instance
   */
  public static ImmutableIpam copyOf(Ipam instance) {
    if (instance instanceof ImmutableIpam) {
      return (ImmutableIpam) instance;
    }
    return ImmutableIpam.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableIpam ImmutableIpam}.
   * <pre>
   * ImmutableIpam.builder()
   *    .driver(String) // required {@link Ipam#driver() driver}
   *    .config(List&lt;org.mandas.docker.client.messages.IpamConfig&gt; | null) // nullable {@link Ipam#config() config}
   *    .options(Map&lt;String, String&gt; | null) // nullable {@link Ipam#options() options}
   *    .build();
   * </pre>
   * @return A new ImmutableIpam builder
   */
  public static ImmutableIpam.Builder builder() {
    return new ImmutableIpam.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableIpam ImmutableIpam}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Ipam.Builder {
    private static final long INIT_BIT_DRIVER = 0x1L;
    private long initBits = 0x1L;

    private String driver;
    private List<IpamConfig> config = null;
    private Map<String, String> options = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Ipam} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Ipam instance) {
      Objects.requireNonNull(instance, "instance");
      this.driver(instance.driver());
      @Nullable List<IpamConfig> configValue = instance.config();
      if (configValue != null) {
        addAllConfig(configValue);
      }
      @Nullable Map<String, String> optionsValue = instance.options();
      if (optionsValue != null) {
        putAllOptions(optionsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Ipam#driver() driver} attribute.
     * @param driver The value for driver 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Driver")
    public final Builder driver(String driver) {
      this.driver = Objects.requireNonNull(driver, "driver");
      initBits &= ~INIT_BIT_DRIVER;
      return this;
    }

    /**
     * Adds one element to {@link Ipam#config() config} list.
     * @param element A config element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder config(IpamConfig element) {
      if (this.config == null) {
        this.config = new ArrayList<IpamConfig>();
      }
      this.config.add(Objects.requireNonNull(element, "config element"));
      return this;
    }

    /**
     * Adds elements to {@link Ipam#config() config} list.
     * @param elements An array of config elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder config(IpamConfig... elements) {
      if (this.config == null) {
        this.config = new ArrayList<IpamConfig>();
      }
      for (IpamConfig element : elements) {
        this.config.add(Objects.requireNonNull(element, "config element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Ipam#config() config} list.
     * @param elements An iterable of config elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Config")
    public final Builder config(@Nullable Iterable<? extends IpamConfig> elements) {
      if (elements == null) {
        this.config = null;
        return this;
      }
      this.config = new ArrayList<IpamConfig>();
      return addAllConfig(elements);
    }

    /**
     * Adds elements to {@link Ipam#config() config} list.
     * @param elements An iterable of config elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllConfig(Iterable<? extends IpamConfig> elements) {
      Objects.requireNonNull(elements, "config element");
      if (this.config == null) {
        this.config = new ArrayList<IpamConfig>();
      }
      for (IpamConfig element : elements) {
        this.config.add(Objects.requireNonNull(element, "config element"));
      }
      return this;
    }

    /**
     * Put one entry to the {@link Ipam#options() options} map.
     * @param key The key in the options map
     * @param value The associated value in the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(String key, String value) {
      if (this.options == null) {
        this.options = new LinkedHashMap<String, String>();
      }
      this.options.put(
          Objects.requireNonNull(key, "options key"),
          Objects.requireNonNull(value, value == null ? "options value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link Ipam#options() options} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(Map.Entry<String, ? extends String> entry) {
      if (this.options == null) {
        this.options = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.options.put(
          Objects.requireNonNull(k, "options key"),
          Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link Ipam#options() options} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Options")
    public final Builder options(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.options = null;
        return this;
      }
      this.options = new LinkedHashMap<String, String>();
      return putAllOptions(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link Ipam#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllOptions(Map<String, ? extends String> entries) {
      if (this.options == null) {
        this.options = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.options.put(
            Objects.requireNonNull(k, "options key"),
            Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableIpam ImmutableIpam}.
     * @return An immutable instance of Ipam
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableIpam build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableIpam(
          driver,
          config == null ? null : createUnmodifiableList(true, config),
          options == null ? null : createUnmodifiableMap(false, false, options));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DRIVER) != 0) attributes.add("driver");
      return "Cannot build Ipam, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
