/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.processors;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.ASTHelper;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelTypeExtensions;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.MelangeFactory;
import fr.inria.diverse.melange.metamodel.melange.ModelType;
import fr.inria.diverse.melange.metamodel.melange.ModelTypingSpace;
import fr.inria.diverse.melange.metamodel.melange.Subtyping;
import fr.inria.diverse.melange.typesystem.MelangeTypesRegistry;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Computes the subtyping hierarchy among {@link ModelType}s and the
 * implementation relations between {@link Language}s and {@link ModelType}s.
 * Populates the #implements and #subtypingRelations references accordingly.
 * Also populates the {@link MelangeTypesRegistry} that is used in our
 * specialized Xbase model-oriented type system and compiler.
 */
@SuppressWarnings("all")
public class TypingInferrer extends DispatchMelangeProcessor {
  @Inject
  @Extension
  private ASTHelper _aSTHelper;

  @Inject
  @Extension
  private ModelTypeExtensions _modelTypeExtensions;

  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  @Inject
  private MelangeTypesRegistry typesRegistry;

  protected void _preProcess(final ModelTypingSpace root, final boolean preLinkingPhase) {
    this.typesRegistry.clear();
    final Function1<Language, Set<Language>> _function = (Language it) -> {
      return this._languageExtensions.getAllDependencies(it);
    };
    final Function1<Language, EObject> _function_1 = (Language it) -> {
      return it.eContainer();
    };
    final Function1<ModelTypingSpace, Boolean> _function_2 = (ModelTypingSpace it) -> {
      return Boolean.valueOf((it != root));
    };
    final Set<ModelTypingSpace> externalRoots = IterableExtensions.<ModelTypingSpace>toSet(IterableExtensions.<ModelTypingSpace>filter(Iterables.<ModelTypingSpace>filter(IterableExtensions.<Language, EObject>map(Iterables.<Language>concat(IterableExtensions.<Language, Set<Language>>map(this._aSTHelper.getLanguages(root), _function)), _function_1), ModelTypingSpace.class), _function_2));
    final Function1<ModelTypingSpace, Iterable<ModelType>> _function_3 = (ModelTypingSpace it) -> {
      return this._aSTHelper.getModelTypes(it);
    };
    final List<ModelType> allMts = IterableExtensions.<ModelType>toList(Iterables.<ModelType>concat(IterableExtensions.<ModelTypingSpace, Iterable<ModelType>>map(externalRoots, _function_3)));
    Iterables.<ModelType>addAll(allMts, this._aSTHelper.getModelTypes(root));
    final Function1<ModelType, Boolean> _function_4 = (ModelType it) -> {
      return Boolean.valueOf(this._modelTypeExtensions.isComparable(it));
    };
    final Consumer<ModelType> _function_5 = (ModelType mt1) -> {
      final Function1<ModelType, Boolean> _function_6 = (ModelType mt2) -> {
        return Boolean.valueOf((((!Objects.equal(mt2, mt1)) && (!IterableExtensions.<Subtyping>exists(mt1.getSubtypingRelations(), ((Function1<Subtyping, Boolean>) (Subtyping it) -> {
          String _name = it.getSuperType().getName();
          String _name_1 = mt2.getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        })))) && this._modelTypeExtensions.isSubtypeOf(mt1, mt2)));
      };
      final Consumer<ModelType> _function_7 = (ModelType mt2) -> {
        EList<Subtyping> _subtypingRelations = mt1.getSubtypingRelations();
        Subtyping _createSubtyping = MelangeFactory.eINSTANCE.createSubtyping();
        final Procedure1<Subtyping> _function_8 = (Subtyping it) -> {
          it.setSubType(mt1);
          it.setSuperType(mt2);
        };
        Subtyping _doubleArrow = ObjectExtensions.<Subtyping>operator_doubleArrow(_createSubtyping, _function_8);
        _subtypingRelations.add(_doubleArrow);
        this.typesRegistry.registerSubtyping(
          this._iQualifiedNameProvider.getFullyQualifiedName(mt1).toString(), mt2);
      };
      IterableExtensions.<ModelType>filter(allMts, _function_6).forEach(_function_7);
      final Function1<Language, Boolean> _function_8 = (Language l) -> {
        return Boolean.valueOf(((this._languageExtensions.isTypable(l) && (!IterableExtensions.<ModelType>exists(l.getImplements(), ((Function1<ModelType, Boolean>) (ModelType it) -> {
          String _name = it.getName();
          String _name_1 = mt1.getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        })))) && this._languageExtensions.doesImplement(l, mt1)));
      };
      final Consumer<Language> _function_9 = (Language l) -> {
        EList<ModelType> _implements = l.getImplements();
        _implements.add(mt1);
        this.typesRegistry.registerImplementation(
          this._iQualifiedNameProvider.getFullyQualifiedName(l).toString(), mt1);
      };
      IterableExtensions.<Language>filter(this._aSTHelper.getLanguages(root), _function_8).forEach(_function_9);
    };
    IterableExtensions.<ModelType>filter(allMts, _function_4).forEach(_function_5);
  }

  protected void _postProcess(final ModelTypingSpace root) {
    final Consumer<Language> _function = (Language it) -> {
      it.getImplements().clear();
    };
    this._aSTHelper.getLanguages(root).forEach(_function);
    final Consumer<ModelType> _function_1 = (ModelType it) -> {
      it.getSubtypingRelations().clear();
    };
    this._aSTHelper.getModelTypes(root).forEach(_function_1);
  }

  public void preProcess(final EObject root, final boolean preLinkingPhase) {
    if (root instanceof ModelTypingSpace) {
      _preProcess((ModelTypingSpace)root, preLinkingPhase);
      return;
    } else if (root != null) {
      _preProcess(root, preLinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(root, preLinkingPhase).toString());
    }
  }

  public void postProcess(final EObject root) {
    if (root instanceof ModelTypingSpace) {
      _postProcess((ModelTypingSpace)root);
      return;
    } else if (root != null) {
      _postProcess(root);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(root).toString());
    }
  }
}
