/**
 * Copyright (c) 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package fr.inria.diverse.melange.utils;

import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;
import com.google.inject.Inject;
import fr.inria.diverse.melange.ast.LanguageExtensions;
import fr.inria.diverse.melange.ast.ModelingElementExtensions;
import fr.inria.diverse.melange.lib.EcoreExtensions;
import fr.inria.diverse.melange.metamodel.melange.Aspect;
import fr.inria.diverse.melange.metamodel.melange.ClassBinding;
import fr.inria.diverse.melange.metamodel.melange.Language;
import fr.inria.diverse.melange.metamodel.melange.PackageBinding;
import fr.inria.diverse.melange.metamodel.melange.PropertyBinding;
import fr.inria.diverse.melange.metamodel.melange.Weave;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jface.text.Document;
import org.eclipse.text.edits.TextEdit;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * This class recomputes the dispatch of Aspect's methods and rewrite
 * their body
 */
@SuppressWarnings("all")
public class DispatchOverrider {
  /**
   * This visitor replace body of recomputed dispatch methods
   */
  public static class MethodBodyReplacer extends ASTVisitor {
    private Class<?> aspect;

    private HashMap<Method, String> bodies;

    public MethodBodyReplacer(final Class<?> aspect, final HashMap<Method, String> bodies) {
      this.aspect = aspect;
      this.bodies = bodies;
    }

    @Override
    public boolean visit(final MethodDeclaration node) {
      boolean _xblockexpression = false;
      {
        final String rawBody = this.findBody(node);
        boolean _isEmpty = rawBody.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final ASTParser parser = ASTParser.newParser(AST.JLS8);
          parser.setSource(rawBody.toCharArray());
          parser.setKind(ASTParser.K_STATEMENTS);
          final ASTNode genNode = parser.createAST(null);
          if ((genNode instanceof Block)) {
            ASTNode _copySubtree = ASTNode.copySubtree(node.getAST(), genNode);
            node.setBody(((Block) _copySubtree));
          }
        }
        _xblockexpression = super.visit(node);
      }
      return _xblockexpression;
    }

    private String findBody(final MethodDeclaration method) {
      String _xblockexpression = null;
      {
        final Function1<Method, Boolean> _function = (Method it) -> {
          Class<?> _declaringClass = it.getDeclaringClass();
          return Boolean.valueOf(Objects.equal(_declaringClass, this.aspect));
        };
        final Function1<Method, Boolean> _function_1 = (Method it) -> {
          return Boolean.valueOf(this.isMatching(it, method));
        };
        final Method match = IterableExtensions.<Method>findFirst(IterableExtensions.<Method>filter(this.bodies.keySet(), _function), _function_1);
        String _xifexpression = null;
        if ((match != null)) {
          return this.bodies.get(match);
        } else {
          _xifexpression = "";
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    }

    private boolean isMatching(final Method m1, final MethodDeclaration m2) {
      return ((Objects.equal(m1.getName(), m2.getName().toString()) && (m1.getParameterTypes().length == ((Object[])Conversions.unwrapArray(m2.parameters(), Object.class)).length)) && IterableExtensions.<Class<?>>forall(((Iterable<Class<?>>)Conversions.doWrapArray(m1.getParameterTypes())), ((Function1<Class<?>, Boolean>) (Class<?> param) -> {
        boolean _xblockexpression = false;
        {
          final int index = ((List<Class<?>>)Conversions.doWrapArray(m1.getParameterTypes())).indexOf(param);
          boolean _xifexpression = false;
          if ((index != 0)) {
            String _simpleName = param.getSimpleName();
            Object _get = m2.parameters().get(index);
            String _string = ((SingleVariableDeclaration) _get).getType().toString();
            _xifexpression = Objects.equal(_simpleName, _string);
          } else {
            _xifexpression = true;
          }
          _xblockexpression = _xifexpression;
        }
        return Boolean.valueOf(_xblockexpression);
      })));
    }
  }

  @Inject
  @Extension
  private ModelingElementExtensions _modelingElementExtensions;

  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;

  @Inject
  @Extension
  private EcoreExtensions _ecoreExtensions;

  @Inject
  @Extension
  private LanguageExtensions _languageExtensions;

  private static final Logger log = Logger.getLogger(DispatchOverrider.class);

  /**
   * Link Language to its Aspects' Java classes
   */
  private SetMultimap<Language, Class<?>> aspectsByLang = HashMultimap.<Language, Class<?>>create();

  /**
   * Link Aspect's Java class to aspected EClass
   */
  private HashMap<Class<?>, EClass> aspected = new HashMap<Class<?>, EClass>();

  /**
   * Link Java class to JDT class
   */
  private HashMap<Class<?>, IType> source = new HashMap<Class<?>, IType>();

  /**
   * Link aspected EClass to containing Language
   */
  private HashMap<EClass, Language> eClassToLanguage = new HashMap<EClass, Language>();

  /**
   * Link EClass to all its subtyping EClasses
   */
  private SetMultimap<EClass, EClass> subTypes = HashMultimap.<EClass, EClass>create();

  private List<Pair<Aspect, List<PackageBinding>>> sourceRenaming;

  private Language language;

  public void overrideDispatch(final Language lang, final IJavaProject melangeProject) {
    boolean _isGeneratedByMelange = this._languageExtensions.isGeneratedByMelange(lang);
    boolean _not = (!_isGeneratedByMelange);
    if (_not) {
      return;
    }
    this.language = lang;
    this.aspectsByLang = HashMultimap.<Language, Class<?>>create();
    HashMap<Class<?>, EClass> _hashMap = new HashMap<Class<?>, EClass>();
    this.aspected = _hashMap;
    HashMap<Class<?>, IType> _hashMap_1 = new HashMap<Class<?>, IType>();
    this.source = _hashMap_1;
    HashMap<EClass, Language> _hashMap_2 = new HashMap<EClass, Language>();
    this.eClassToLanguage = _hashMap_2;
    this.subTypes = HashMultimap.<EClass, EClass>create();
    this.sourceRenaming = CollectionLiterals.<Pair<Aspect, List<PackageBinding>>>newArrayList();
    final ClassLoader loader = this.createClassLoader(melangeProject);
    this.initSubClasses(lang);
    final HashSet<Class<?>> aspects = new HashSet<Class<?>>();
    final Consumer<Aspect> _function = (Aspect asp) -> {
      try {
        final IType type = melangeProject.findType(asp.getAspectTypeRef().getType().getQualifiedName());
        final Class<?> aspType = loader.loadClass(asp.getAspectTypeRef().getType().getQualifiedName());
        this.aspected.put(aspType, asp.getAspectedClass());
        this.source.put(aspType, type);
        aspects.add(aspType);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    lang.getSemantics().forEach(_function);
    this.aspectsByLang.putAll(lang, aspects);
    this.processLanguage(lang);
  }

  private SetMultimap<Class<?>, Class<?>> getHierachy(final Language l) {
    final Set<Class<?>> pool = this.aspectsByLang.get(l);
    final SetMultimap<Class<?>, Class<?>> subClasses = HashMultimap.<Class<?>, Class<?>>create();
    final Consumer<Class<?>> _function = (Class<?> cls1) -> {
      final Function1<Class<?>, Boolean> _function_1 = (Class<?> cls2) -> {
        return Boolean.valueOf(((cls1 != cls2) && cls1.isAssignableFrom(cls2)));
      };
      final Iterable<Class<?>> subCls = IterableExtensions.<Class<?>>filter(pool, _function_1);
      subClasses.putAll(cls1, subCls);
    };
    pool.forEach(_function);
    return subClasses;
  }

  private void processLanguage(final Language l) {
    final SetMultimap<Class<?>, Class<?>> hierarchy = this.getHierachy(l);
    final Function1<Class<?>, Boolean> _function = (Class<?> it) -> {
      return Boolean.valueOf(this.isAspect(it));
    };
    final Consumer<Class<?>> _function_1 = (Class<?> asp) -> {
      this.processAspect(asp, hierarchy);
    };
    IterableExtensions.<Class<?>>filter(this.aspectsByLang.get(l), _function).forEach(_function_1);
  }

  private void processAspect(final Class<?> aspect, final SetMultimap<Class<?>, Class<?>> hierarchy) {
    final HashMap<Method, String> bodies = new HashMap<Method, String>();
    final Consumer<Method> _function = (Method m) -> {
      final String body = this.genCode(m, hierarchy);
      bodies.put(m, body);
    };
    this.getDispatchMethods(aspect).forEach(_function);
    this.rewriteBodies(aspect, bodies);
  }

  private Iterable<Method> getDispatchMethods(final Class<?> type) {
    final Function1<Method, Boolean> _function = (Method m) -> {
      return Boolean.valueOf(((Modifier.isPublic(m.getModifiers()) && (!m.getName().startsWith("_privk3"))) && (!m.getName().startsWith("super_"))));
    };
    return IterableExtensions.<Method>filter(((Iterable<Method>)Conversions.doWrapArray(type.getDeclaredMethods())), _function);
  }

  private String genCode(final Method m, final SetMultimap<Class<?>, Class<?>> hierarchy) {
    String _xblockexpression = null;
    {
      final StringBuffer res = new StringBuffer();
      String _methodProcessingChangeBody = this.methodProcessingChangeBody(m, hierarchy);
      String _plus = (_methodProcessingChangeBody + "\n\n");
      res.append(_plus);
      _xblockexpression = res.toString();
    }
    return _xblockexpression;
  }

  /**
   * Return the sorted list of all methods overriding m
   */
  private Iterable<Method> getRedefinition(final Method m, final SetMultimap<Class<?>, Class<?>> hierarchy) {
    Iterable<Method> _xblockexpression = null;
    {
      final Class<?> type = m.getDeclaringClass();
      final Function1<Class<?>, Method> _function = (Class<?> it) -> {
        return this.getSimilar(it, m);
      };
      _xblockexpression = IterableExtensions.<Method>filterNull(IterableExtensions.<Class<?>, Method>map(this.sortByOverridingPriority(hierarchy.get(type)), _function));
    }
    return _xblockexpression;
  }

  /**
   * Return the method from {@link type} that match {@link m} by
   * same name and same argument types
   */
  private Method getSimilar(final Class<?> type, final Method m) {
    final Function1<Method, Boolean> _function = (Method it) -> {
      return Boolean.valueOf(this.isMatching(it, m));
    };
    return IterableExtensions.<Method>findFirst(((Iterable<Method>)Conversions.doWrapArray(type.getDeclaredMethods())), _function);
  }

  private boolean isMatching(final Method m1, final Method m2) {
    return ((Objects.equal(m1.getName(), m2.getName()) && (m1.getParameterTypes().length == m2.getParameterTypes().length)) && IterableExtensions.<Class<?>>forall(((Iterable<Class<?>>)Conversions.doWrapArray(m1.getParameterTypes())), ((Function1<Class<?>, Boolean>) (Class<?> param) -> {
      boolean _xblockexpression = false;
      {
        final int index = ((List<Class<?>>)Conversions.doWrapArray(m1.getParameterTypes())).indexOf(param);
        boolean _xifexpression = false;
        if ((index != 0)) {
          String _simpleName = param.getSimpleName();
          String _simpleName_1 = (m2.getParameterTypes()[index]).getSimpleName();
          _xifexpression = Objects.equal(_simpleName, _simpleName_1);
        } else {
          _xifexpression = true;
        }
        _xblockexpression = _xifexpression;
      }
      return Boolean.valueOf(_xblockexpression);
    })));
  }

  /**
   * Sort types by
   * 1) aspected class hierachy
   * 2) inheritance
   * 3) source Language
   */
  private Iterable<Class<?>> sortByOverridingPriority(final Iterable<Class<?>> types) {
    boolean _isEmpty = IterableExtensions.isEmpty(types);
    if (_isEmpty) {
      return types;
    }
    final EList<Aspect> originalOrder = this.language.getSemantics();
    final Comparator<Class<?>> _function = (Class<?> asp1, Class<?> asp2) -> {
      int _xblockexpression = (int) 0;
      {
        final EClass cls1 = this.aspected.get(asp1);
        final EClass cls2 = this.aspected.get(asp2);
        int _xifexpression = (int) 0;
        if ((((cls1 != null) && (cls2 != null)) && cls2.getEAllSuperTypes().contains(cls1))) {
          _xifexpression = 1;
        } else {
          int _xifexpression_1 = (int) 0;
          if ((((cls1 != null) && (cls2 != null)) && cls1.getEAllSuperTypes().contains(cls2))) {
            _xifexpression_1 = (-1);
          } else {
            int _xifexpression_2 = (int) 0;
            boolean _isAssignableFrom = asp1.isAssignableFrom(asp2);
            if (_isAssignableFrom) {
              _xifexpression_2 = 1;
            } else {
              int _xifexpression_3 = (int) 0;
              boolean _isAssignableFrom_1 = asp2.isAssignableFrom(asp1);
              if (_isAssignableFrom_1) {
                _xifexpression_3 = (-1);
              } else {
                int _xblockexpression_1 = (int) 0;
                {
                  final Function1<Aspect, Boolean> _function_1 = (Aspect it) -> {
                    String _string = this._iQualifiedNameProvider.getFullyQualifiedName(it.getAspectTypeRef().getType()).toString();
                    String _name = asp1.getName();
                    return Boolean.valueOf(Objects.equal(_string, _name));
                  };
                  final Aspect aspect1 = IterableExtensions.<Aspect>findFirst(originalOrder, _function_1);
                  final Function1<Aspect, Boolean> _function_2 = (Aspect it) -> {
                    String _string = this._iQualifiedNameProvider.getFullyQualifiedName(it.getAspectTypeRef().getType()).toString();
                    String _name = asp2.getName();
                    return Boolean.valueOf(Objects.equal(_string, _name));
                  };
                  final Aspect aspect2 = IterableExtensions.<Aspect>findFirst(originalOrder, _function_2);
                  if (((aspect1 == null) || (aspect2 == null))) {
                    return 0;
                  }
                  final int index1 = originalOrder.indexOf(aspect1);
                  final int index2 = originalOrder.indexOf(aspect2);
                  int _xifexpression_4 = (int) 0;
                  if ((index1 > index2)) {
                    _xifexpression_4 = 1;
                  } else {
                    int _xifexpression_5 = (int) 0;
                    if ((index2 > index1)) {
                      _xifexpression_5 = (-1);
                    } else {
                      _xifexpression_5 = 0;
                    }
                    _xifexpression_4 = _xifexpression_5;
                  }
                  _xblockexpression_1 = _xifexpression_4;
                }
                _xifexpression_3 = _xblockexpression_1;
              }
              _xifexpression_2 = _xifexpression_3;
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    };
    return IterableExtensions.<Class<?>>sortWith(types, _function);
  }

  /**
   * Names of parameter may not be in the bytecode (depends of compiler options)
   * so we rely on the JDT
   */
  private String getParameterList(final Method m1) {
    try {
      String _xblockexpression = null;
      {
        final IType type = this.source.get(m1.getDeclaringClass());
        final Function1<IMethod, Boolean> _function = (IMethod m2) -> {
          return Boolean.valueOf(((Objects.equal(m2.getElementName(), m1.getName()) && (m2.getParameterTypes().length == m1.getParameterTypes().length)) && IterableExtensions.<String>forall(((Iterable<String>)Conversions.doWrapArray(m2.getParameterTypes())), ((Function1<String, Boolean>) (String param) -> {
            boolean _xblockexpression_1 = false;
            {
              final int index = ((List<String>)Conversions.doWrapArray(m2.getParameterTypes())).indexOf(param);
              boolean _xifexpression = false;
              if ((index != 0)) {
                String _simpleName = (m1.getParameterTypes()[index]).getSimpleName();
                String _string = Signature.toString(param);
                _xifexpression = Objects.equal(_simpleName, _string);
              } else {
                _xifexpression = true;
              }
              _xblockexpression_1 = _xifexpression;
            }
            return Boolean.valueOf(_xblockexpression_1);
          }))));
        };
        final IMethod res = IterableExtensions.<IMethod>findFirst(((Iterable<IMethod>)Conversions.doWrapArray(type.getMethods())), _function);
        String _xifexpression = null;
        if ((res != null)) {
          _xifexpression = IterableExtensions.join(((Iterable<?>)Conversions.doWrapArray(res.getParameterNames())), ",");
        } else {
          _xifexpression = DispatchOverrider.SELF_VAR_NAME;
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private ClassLoader createClassLoader(final IJavaProject project) {
    try {
      final String[] classPathEntries = JavaRuntime.computeDefaultRuntimeClassPath(project);
      final ArrayList<URL> urlList = new ArrayList<URL>();
      for (int i = 0; (i < classPathEntries.length); i++) {
        {
          final String entry = classPathEntries[i];
          final Path path = new Path(entry);
          final URL url = path.toFile().toURI().toURL();
          urlList.add(url);
        }
      }
      final ClassLoader parentClassLoader = Thread.currentThread().getContextClassLoader();
      return new URLClassLoader(((URL[])Conversions.unwrapArray(urlList, URL.class)), parentClassLoader);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * K3 Stuff
   */
  public static final String CTX_NAME = "AspectContext";

  public static final String PROP_NAME = "AspectProperties";

  public static final String PRIV_PREFIX = "_privk3_";

  public static final String PROP_VAR_NAME = "_self_";

  public static final String SELF_VAR_NAME = "_self";

  private String methodProcessingChangeBody(final Method m, final SetMultimap<Class<?>, Class<?>> hierarchy) {
    final Class<?> clazz = m.getDeclaringClass();
    final String className = this.getAspectedSimpleName(clazz);
    final String s = this.getParameterList(m);
    final boolean isStep = this.isStep(m);
    final String ret = this.getReturnInstruction(m);
    final StringBuilder call = new StringBuilder();
    final Iterable<Method> redefs = this.getRedefinition(m, hierarchy);
    boolean _isEmpty = IterableExtensions.isEmpty(redefs);
    boolean _not = (!_isEmpty);
    if (_not) {
      final Function1<Method, Class<?>> _function = (Method it) -> {
        return it.getDeclaringClass();
      };
      final List<Class<?>> declTypes = IterableExtensions.<Class<?>>toList(IterableExtensions.<Method, Class<?>>map(redefs, _function));
      declTypes.add(m.getDeclaringClass());
      final String ifst = this.transformIfStatements(m, declTypes, s, ret, isStep, className);
      StringBuilder _append = call.append(ifst);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(" ");
      _builder.append("{ throw new IllegalArgumentException(\"Unhandled parameter types: \" + java.util.Arrays.<Object>asList(");
      _builder.append(DispatchOverrider.SELF_VAR_NAME, " ");
      _builder.append(").toString()); }");
      _append.append(_builder);
    } else {
      final String instruction = this.transformNormalStatement(m, s, isStep, className);
      call.append(instruction);
    }
    return this.getBody(clazz, className, call.toString(), ret);
  }

  private String getBody(final Class<?> clazz, final String className, final String call, final String returnStatement) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("final ");
    String _canonicalName = clazz.getCanonicalName();
    String _plus = (_canonicalName + className);
    String _plus_1 = (_plus + DispatchOverrider.PROP_NAME);
    _builder.append(_plus_1);
    _builder.append(" ");
    _builder.append(DispatchOverrider.PROP_VAR_NAME);
    _builder.append(" = ");
    String _canonicalName_1 = clazz.getCanonicalName();
    String _plus_2 = (_canonicalName_1 + className);
    String _plus_3 = (_plus_2 + DispatchOverrider.CTX_NAME);
    _builder.append(_plus_3);
    _builder.append(".getSelf(");
    _builder.append(DispatchOverrider.SELF_VAR_NAME);
    _builder.append(");");
    _builder.newLineIfNotEmpty();
    {
      boolean _contains = returnStatement.contains("return");
      if (_contains) {
        _builder.append("Object result = null;");
        _builder.newLine();
      }
    }
    String _string = call.toString();
    _builder.append(_string);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append(returnStatement);
    return _builder.toString();
  }

  private String getReturnInstruction(final Method method) {
    String _xifexpression = null;
    boolean _hasReturnType = this.hasReturnType(method);
    if (_hasReturnType) {
      String _canonicalName = method.getReturnType().getCanonicalName();
      String _plus = ("return (" + _canonicalName);
      _xifexpression = (_plus + ")result;");
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }

  private boolean hasReturnType(final Method method) {
    Class<?> _returnType = method.getReturnType();
    return (!Objects.equal(_returnType, Void.TYPE));
  }

  private String transformIfStatements(final Method m, final List<Class<?>> declTypes, final String parameters, final String returnStatement, final boolean isStep, final String className) {
    final boolean hasReturn = returnStatement.contains("return");
    final String resultVar = "result";
    final StringBuilder sb = new StringBuilder();
    final ArrayList<EClass> dispatchOrder = this.getEAllSubTypes(this.aspected.get(m.getDeclaringClass()));
    for (final EClass cls : dispatchOrder) {
      {
        final Class<?> dt = this.bestCandidate(cls, declTypes);
        String call = "";
        Class<?> _declaringClass = m.getDeclaringClass();
        boolean _equals = Objects.equal(_declaringClass, dt);
        if (_equals) {
          StringConcatenation _builder = new StringConcatenation();
          String _canonicalName = dt.getCanonicalName();
          _builder.append(_canonicalName);
          _builder.append(".");
          String _originalName = this.originalName(m);
          String _plus = (DispatchOverrider.PRIV_PREFIX + _originalName);
          _builder.append(_plus);
          _builder.append("(_self_, ");
          String _javaFqn = this.getJavaFqn(cls);
          String _plus_1 = ("(" + _javaFqn);
          String _plus_2 = (_plus_1 + ")");
          String _plus_3 = (_plus_2 + DispatchOverrider.SELF_VAR_NAME);
          String _replaceFirst = parameters.replaceFirst(DispatchOverrider.SELF_VAR_NAME, _plus_3);
          _builder.append(_replaceFirst);
          _builder.append(")");
          call = _builder.toString();
          if (isStep) {
            call = this.surroundWithStepCommandExecution(className, m.getName(), call, hasReturn, resultVar);
          } else {
            if (hasReturn) {
              StringConcatenation _builder_1 = new StringConcatenation();
              _builder_1.append(resultVar);
              _builder_1.append(" = ");
              _builder_1.append(call);
              call = _builder_1.toString();
            }
          }
        } else {
          StringConcatenation _builder_2 = new StringConcatenation();
          String _canonicalName_1 = dt.getCanonicalName();
          _builder_2.append(_canonicalName_1);
          _builder_2.append(".");
          String _name = m.getName();
          _builder_2.append(_name);
          _builder_2.append("(");
          String _javaFqn_1 = this.getJavaFqn(cls);
          String _plus_4 = ("(" + _javaFqn_1);
          String _plus_5 = (_plus_4 + ")");
          String _plus_6 = (_plus_5 + DispatchOverrider.SELF_VAR_NAME);
          String _replaceFirst_1 = parameters.replaceFirst(DispatchOverrider.SELF_VAR_NAME, _plus_6);
          _builder_2.append(_replaceFirst_1);
          _builder_2.append(")");
          call = _builder_2.toString();
          if (hasReturn) {
            StringConcatenation _builder_3 = new StringConcatenation();
            _builder_3.append(resultVar);
            _builder_3.append(" = ");
            _builder_3.append(call);
            call = _builder_3.toString();
          }
        }
        StringConcatenation _builder_4 = new StringConcatenation();
        _builder_4.append(" ");
        _builder_4.append("if (");
        _builder_4.append(DispatchOverrider.SELF_VAR_NAME, " ");
        _builder_4.append(" instanceof ");
        String _javaFqn_2 = this.getJavaFqn(cls);
        _builder_4.append(_javaFqn_2, " ");
        _builder_4.append("){");
        _builder_4.newLineIfNotEmpty();
        _builder_4.append("\t\t\t\t\t");
        _builder_4.append(call, "\t\t\t\t\t");
        _builder_4.append(";");
        _builder_4.newLineIfNotEmpty();
        _builder_4.append("} else ");
        sb.append(_builder_4);
      }
    }
    return sb.toString();
  }

  private String transformNormalStatement(final Method method, final String parameters, final boolean isStep, final String className) {
    final boolean hasReturn = this.hasReturnType(method);
    final String resultVar = "result";
    StringConcatenation _builder = new StringConcatenation();
    String _originalName = this.originalName(method);
    String _plus = (DispatchOverrider.PRIV_PREFIX + _originalName);
    _builder.append(_plus);
    _builder.append("(_self_, ");
    _builder.append(parameters);
    _builder.append(")");
    String call = _builder.toString();
    if (isStep) {
      call = this.surroundWithStepCommandExecution(className, method.getName(), call, hasReturn, resultVar);
    } else {
      if (hasReturn) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(resultVar);
        _builder_1.append(" = ");
        _builder_1.append(call);
        call = _builder_1.toString();
      }
    }
    return (call + ";");
  }

  private String surroundWithStepCommandExecution(final String className, final String methodName, final String code, final boolean hasReturn, final String resultVar) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("fr.inria.diverse.k3.al.annotationprocessor.stepmanager.StepCommand command = new fr.inria.diverse.k3.al.annotationprocessor.stepmanager.StepCommand() {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("@Override");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("public void execute() {");
    _builder.newLine();
    {
      if (hasReturn) {
        _builder.append("\t\t");
        _builder.append("addToResult(");
        _builder.append(code, "\t\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("\t\t");
        _builder.append(code, "\t\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("};");
    _builder.newLine();
    _builder.append("fr.inria.diverse.k3.al.annotationprocessor.stepmanager.IStepManager manager = fr.inria.diverse.k3.al.annotationprocessor.stepmanager.StepManagerRegistry.getInstance().findStepManager(_self);");
    _builder.newLine();
    _builder.append("if (manager != null) {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("manager.executeStep(_self,command,\"");
    _builder.append(className, "\t");
    _builder.append("\",\"");
    _builder.append(methodName, "\t");
    _builder.append("\");");
    _builder.newLineIfNotEmpty();
    _builder.append("} else {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("command.execute();");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    {
      if (hasReturn) {
        _builder.append(resultVar);
        _builder.append(" = command.getResult();");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }

  private String getAspectedSimpleName(final Class<?> type) {
    String _xblockexpression = null;
    {
      final EClass eClass = this.aspected.get(type);
      String _xifexpression = null;
      if ((eClass != null)) {
        return eClass.getName();
      } else {
        _xifexpression = "";
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  private boolean isAspect(final Class<?> cls) {
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      String _canonicalName = it.annotationType().getCanonicalName();
      return Boolean.valueOf(Objects.equal(_canonicalName, "fr.inria.diverse.k3.al.annotationprocessor.Aspect"));
    };
    return IterableExtensions.<Annotation>exists(((Iterable<Annotation>)Conversions.doWrapArray(cls.getDeclaredAnnotations())), _function);
  }

  private boolean isStep(final Method m) {
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      String _canonicalName = it.annotationType().getCanonicalName();
      return Boolean.valueOf(Objects.equal(_canonicalName, "fr.inria.diverse.k3.al.annotationprocessor.Step"));
    };
    return IterableExtensions.<Annotation>exists(((Iterable<Annotation>)Conversions.doWrapArray(m.getDeclaredAnnotations())), _function);
  }

  private void initSubClasses(final Language lang) {
    final Iterable<EClass> allClasses = this._modelingElementExtensions.getAllClasses(lang.getSyntax());
    final Consumer<EClass> _function = (EClass cls) -> {
      this.eClassToLanguage.put(cls, lang);
    };
    allClasses.forEach(_function);
    final Consumer<EClass> _function_1 = (EClass cls1) -> {
      final Function1<EClass, Boolean> _function_2 = (EClass cls2) -> {
        return Boolean.valueOf(cls1.isSuperTypeOf(cls2));
      };
      final Iterable<EClass> subClasses = IterableExtensions.<EClass>filter(allClasses, _function_2);
      this.subTypes.putAll(cls1, subClasses);
    };
    allClasses.forEach(_function_1);
    Stack<List<PackageBinding>> _stack = new Stack<List<PackageBinding>>();
    this.sourceRenaming = this._languageExtensions.getAllAspectsWithRenaming(lang, _stack);
  }

  /**
   * Among {@link declTypes}, get the best applicable aspects on {@link cls}
   */
  private Class<?> bestCandidate(final EClass cls, final List<Class<?>> declTypes) {
    final Function1<Class<?>, Boolean> _function = (Class<?> it) -> {
      boolean _xblockexpression = false;
      {
        final EClass dtCls = this.aspected.get(it);
        _xblockexpression = (Objects.equal(cls, dtCls) || cls.getEAllSuperTypes().contains(dtCls));
      }
      return Boolean.valueOf(_xblockexpression);
    };
    return IterableExtensions.<Class<?>>head(IterableExtensions.<Class<?>>filter(declTypes, _function));
  }

  /**
   * Returns ordered list of the subclasses of cls, including itself
   */
  private ArrayList<EClass> getEAllSubTypes(final EClass cls) {
    final Set<EClass> set = this.subTypes.get(cls);
    final ArrayList<EClass> res = new ArrayList<EClass>();
    final Consumer<EClass> _function = (EClass cls1) -> {
      int insertionIndex = res.size();
      for (int i = (res.size() - 1); (i >= 0); i--) {
        {
          final EClass cls2 = res.get(i);
          boolean _isSuperTypeOf = cls2.isSuperTypeOf(cls1);
          if (_isSuperTypeOf) {
            insertionIndex = i;
          }
        }
      }
      res.add(insertionIndex, cls1);
    };
    set.forEach(_function);
    return res;
  }

  private String getJavaFqn(final EClass cls) {
    final Language l = this.eClassToLanguage.get(cls);
    final String prefix = this._iQualifiedNameProvider.getFullyQualifiedName(l).toLowerCase().toString();
    String _uniqueId = this._ecoreExtensions.getUniqueId(cls);
    return ((prefix + ".") + _uniqueId);
  }

  private void rewriteBodies(final Class<?> aspect, final HashMap<Method, String> bodies) {
    try {
      final ICompilationUnit sourceUnit = this.source.get(aspect).getCompilationUnit();
      final String source = sourceUnit.getSource();
      final Document document = new Document(source);
      final ASTParser parser = ASTParser.newParser(AST.JLS8);
      parser.setSource(sourceUnit);
      ASTNode _createAST = parser.createAST(null);
      final CompilationUnit astRoot = ((CompilationUnit) _createAST);
      astRoot.recordModifications();
      DispatchOverrider.MethodBodyReplacer _methodBodyReplacer = new DispatchOverrider.MethodBodyReplacer(aspect, bodies);
      astRoot.accept(_methodBodyReplacer);
      final TextEdit edits = astRoot.rewrite(document, sourceUnit.getJavaProject().getOptions(true));
      edits.apply(document);
      String newSource = document.get();
      sourceUnit.getBuffer().setContents(newSource);
      sourceUnit.getBuffer().save(null, true);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        String _name = aspect.getName();
        String _plus = ("Couldn\'t apply rewriteBodies on " + _name);
        DispatchOverrider.log.error(_plus, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }

  /**
   * If {@link m} is a getter to a renamed property, try to find the original name.
   * Return the same name otherwise
   */
  private String originalName(final Method m) {
    final Class<?> aspType = m.getDeclaringClass();
    final EClass aspectedCls = this.aspected.get(aspType);
    if ((aspectedCls == null)) {
      return m.getName();
    }
    final Function1<Aspect, Boolean> _function = (Aspect it) -> {
      String _qualifiedName = it.getAspectTypeRef().getType().getQualifiedName();
      String _name = aspType.getName();
      return Boolean.valueOf(Objects.equal(_qualifiedName, _name));
    };
    final Aspect asp = IterableExtensions.<Aspect>findFirst(this.language.getSemantics(), _function);
    final Function1<Pair<Aspect, List<PackageBinding>>, Boolean> _function_1 = (Pair<Aspect, List<PackageBinding>> pair) -> {
      Weave _source = pair.getKey().getSource();
      Weave _source_1 = asp.getSource();
      return Boolean.valueOf(Objects.equal(_source, _source_1));
    };
    Pair<Aspect, List<PackageBinding>> _findFirst = IterableExtensions.<Pair<Aspect, List<PackageBinding>>>findFirst(this.sourceRenaming, _function_1);
    List<PackageBinding> _value = null;
    if (_findFirst!=null) {
      _value=_findFirst.getValue();
    }
    final List<PackageBinding> renaming = _value;
    Iterable<PackageBinding> _filter = null;
    if (renaming!=null) {
      final Function1<PackageBinding, Boolean> _function_2 = (PackageBinding it) -> {
        String _to = it.getTo();
        String _uniqueId = this._ecoreExtensions.getUniqueId(aspectedCls.getEPackage());
        return Boolean.valueOf(Objects.equal(_to, _uniqueId));
      };
      _filter=IterableExtensions.<PackageBinding>filter(renaming, _function_2);
    }
    Iterable<EList<ClassBinding>> _map = null;
    if (_filter!=null) {
      final Function1<PackageBinding, EList<ClassBinding>> _function_3 = (PackageBinding it) -> {
        return it.getClasses();
      };
      _map=IterableExtensions.<PackageBinding, EList<ClassBinding>>map(_filter, _function_3);
    }
    Iterable<ClassBinding> _flatten = null;
    if (_map!=null) {
      _flatten=Iterables.<ClassBinding>concat(_map);
    }
    Iterable<ClassBinding> _filter_1 = null;
    if (_flatten!=null) {
      final Function1<ClassBinding, Boolean> _function_4 = (ClassBinding it) -> {
        String _to = it.getTo();
        String _name = aspectedCls.getName();
        return Boolean.valueOf(Objects.equal(_to, _name));
      };
      _filter_1=IterableExtensions.<ClassBinding>filter(_flatten, _function_4);
    }
    Iterable<EList<PropertyBinding>> _map_1 = null;
    if (_filter_1!=null) {
      final Function1<ClassBinding, EList<PropertyBinding>> _function_5 = (ClassBinding it) -> {
        return it.getProperties();
      };
      _map_1=IterableExtensions.<ClassBinding, EList<PropertyBinding>>map(_filter_1, _function_5);
    }
    Iterable<PropertyBinding> _flatten_1 = null;
    if (_map_1!=null) {
      _flatten_1=Iterables.<PropertyBinding>concat(_map_1);
    }
    PropertyBinding _findFirst_1 = null;
    if (_flatten_1!=null) {
      final Function1<PropertyBinding, Boolean> _function_6 = (PropertyBinding it) -> {
        String _to = it.getTo();
        String _name = m.getName();
        return Boolean.valueOf(Objects.equal(_to, _name));
      };
      _findFirst_1=IterableExtensions.<PropertyBinding>findFirst(_flatten_1, _function_6);
    }
    final PropertyBinding candidateRule = _findFirst_1;
    if ((candidateRule != null)) {
      return candidateRule.getFrom();
    }
    return m.getName();
  }
}
