/*******************************************************************************
 * Copyright (c) 2017 INRIA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     INRIA - initial API and implementation
 *     I3S Laboratory - API update and bug fix
 *******************************************************************************/
package org.eclipse.gemoc.execution.concurrent.ccsljavaengine.dsa.executors.explorer;

import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.lang3.tuple.Pair;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gemoc.execution.concurrent.ccsljavaengine.commons.MoccmlModelExecutionContext;
import org.eclipse.gemoc.execution.concurrent.ccsljavaengine.engine.MoccmlExecutionEngine;
import org.eclipse.gemoc.execution.concurrent.ccsljavaengine.extensions.k3.dsa.helper.IK3ModelStateHelper;
import org.eclipse.gemoc.execution.concurrent.ccsljavaxdsml.api.dsa.executors.CodeExecutionException;
import org.eclipse.gemoc.execution.concurrent.ccsljavaxdsml.api.moc.ICCSLExplorer;
import org.eclipse.gemoc.execution.concurrent.ccsljavaxdsml.api.moc.ICCSLSolver;
import org.eclipse.gemoc.moccml.mapping.feedback.feedback.ModelSpecificEvent;
import org.eclipse.gemoc.trace.commons.model.generictrace.GenericParallelStep;
import org.eclipse.gemoc.trace.commons.model.generictrace.GenericStep;
import org.eclipse.gemoc.trace.commons.model.trace.SmallStep;
import org.eclipse.gemoc.trace.commons.model.trace.Step;

import grph.Grph;

/**
 * Experimental feature to explore the whole state space of a model. Seen for
 * now as an execution engine but may before a dedicated tool that delegate
 * actual execution to a gemoc engine.
 * 
 * @see MoccmlExecutionEngine
 * 
 * @author julien.deantoni@univ-cotedazur.fr
 * @param <T>
 * 
 */
public class ExhaustiveConcurrentExecutionEngine extends MoccmlExecutionEngine {

	public ExhaustiveConcurrentExecutionEngine(MoccmlModelExecutionContext concurrentexecutionContext, ICCSLSolver s)
			throws CoreException {
		super(concurrentexecutionContext, s);
	}

	public StateSpace stateSpace = new StateSpace();
	protected ArrayList<ControlAndRTDState> statesToExplore = new ArrayList<ControlAndRTDState>();
	private boolean savedDotRegularly = true; //only for debugging purpose. Otherwise should be false

	/**
	 * actually performs all the execution steps...
	 * 
	 * @throws CodeExecutionException
	 */
	@Override
	protected void performExecutionStep() {
		String fullLanguageName = this._executionContext.getLanguageDefinitionExtension().getName();
		int lastDot = fullLanguageName.lastIndexOf(".");
		if (lastDot == -1)
			lastDot = 0;
		String languageName = fullLanguageName.substring(lastDot + 1);
		String languageToUpperFirst = languageName.substring(0, 1).toUpperCase() + languageName.substring(1);

		IK3ModelStateHelper modelStateHelper = null;
		try {
			modelStateHelper = (IK3ModelStateHelper) this._executionContext.getDslBundle().loadClass(
					languageToUpperFirst.toLowerCase() + ".xdsml.api.impl." + languageToUpperFirst + "ModelStateHelper")
					.newInstance();
		} catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
			e.printStackTrace();
		}
		EObject model = this._executionContext.getResourceModel().getContents().get(0);
//		System.out.println(model);
		
		((ICCSLExplorer)this._solver).initSolverForExploration();
		
		ControlAndRTDState initialState = new ControlAndRTDState(modelStateHelper.getK3StateSpaceModelState(model),
				this._solver.getState(), this.saveState());
		stateSpace.initialState = initialState;
		stateSpace.addVertex(initialState);
		statesToExplore.add(initialState);

		DateTimeFormatter dtf = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss");  
		LocalDateTime now = LocalDateTime.now();  
		System.out.println("starts exploring state space on "+now);

		int exploredStates = 0;
		while (!statesToExplore.isEmpty()) {
			System.out.println("###################  explored "+exploredStates+"  ##################### still " + statesToExplore.size()+ " steps to explore ###########");
			exploredStates++;
			_possibleLogicalSteps = null;
			ControlAndRTDState currentState = statesToExplore.remove(0);
			modelStateHelper.restoreModelState(currentState.modelState);
			this._solver.setState(currentState.moCCState); //Arrays.copyOf( ?
			this.restoreState(Pair.of(currentState.nextEventToForce, currentState.futurActions));
			// set the possibleLogicalSteps for this state
			beforeUpdatePossibleLogicalSteps(); //filter according to DSA returned value
			((ICCSLExplorer)this._solver).computeAndGetPossibleLogicalStepsForExploration();
			_possibleLogicalSteps = ((ICCSLExplorer)this._solver).updatePossibleLogicalStepsForExploration();
			// 2- compute all states accessible from the currenState when using the
			// possibleLogicalStates
			int originalPossibleLogicalStepSize = _possibleLogicalSteps.size();
			for (int i = 0; i < _possibleLogicalSteps.size(); i++) {
				
//				System.out.println(i+") current model state = "+currentState);
				if (getPossibleLogicalSteps().size() != originalPossibleLogicalStepSize) {
					System.err.println("something went wrong during mocc state save/restore");
				}
				((ICCSLExplorer)this._solver).prepareSolverForNewStepForExploration();
				
				Step<?> aStep = getPossibleLogicalSteps().get(i);
				setSelectedLogicalStep(aStep);
				try {
					executeSelectedLogicalStep();
				} catch (Throwable t) {
					throw new RuntimeException(t);
				}
				((ICCSLExplorer)this._solver).applyLogicalStepForExploration(aStep);
				engineStatus.incrementNbLogicalStepRun();
				ControlAndRTDState newState = new ControlAndRTDState(modelStateHelper.getK3StateSpaceModelState(model),
						this._solver.getState(), this.saveState());//this.saveState());

				ControlAndRTDState theExistingState = null;
				for (ControlAndRTDState s : stateSpace.getVertices()) {
					if (newState.equals(s)) {
						theExistingState = s;
						break;
					}
				}
				if (theExistingState == null) {
					stateSpace.addVertex(newState);
					StringBuffer buf = new StringBuffer(prettyPrint((GenericParallelStep) aStep));
					stateSpace.addDirectedSimpleEdge(currentState, buf, newState);
					statesToExplore.add(newState);
				} else {
					assert (theExistingState != null);
//					System.out.println("there is a loop");
					StringBuffer buf = new StringBuffer(prettyPrint((GenericParallelStep) aStep));
					stateSpace.addDirectedSimpleEdge(currentState, buf, theExistingState);
				}
				modelStateHelper.restoreModelState(currentState.modelState);
				this._solver.setState(currentState.moCCState); //Arrays.copyOf( ?
				this.restoreState(Pair.of(currentState.nextEventToForce, currentState.futurActions));
			}
			((ICCSLExplorer)this._solver).resetCurrentStepForExploration();
			//for debugging purpose
																if(savedDotRegularly && exploredStates%100 == 0) {
																	String modelPath = this._executionContext.getResourceModel().getURI().toPlatformString(true);
																	IProject modelProject = ResourcesPlugin.getWorkspace().getRoot()
																			.getProject(modelPath.substring(1, modelPath.substring(1).indexOf('/') + 1));
																	IFile autFile = modelProject
																			.getFile(modelPath.replace("/" + modelProject.getName() + "/", "") + "_temp_statespace.dot");
																	PrintStream psDot = null;
																	try {
																		psDot = new PrintStream(autFile.getLocationURI().toString().substring(5));
																	} catch (FileNotFoundException e) {
																		e.printStackTrace();
																	}
																	Grph internalGrph = stateSpace.getGrph();
																	psDot.print(internalGrph.toDot().replaceAll("##", "#"));
																	psDot.close();
																}
		}
//		stepExecutor.clearFiredClock();
//		stepExecutor = null;
		this._solver = null;
		
		stop();
		PrintStream psDot = null;
		PrintStream psAut = null;
		String modelPath = this._executionContext.getResourceModel().getURI().toPlatformString(true);
		IProject modelProject = ResourcesPlugin.getWorkspace().getRoot()
				.getProject(modelPath.substring(1, modelPath.substring(1).indexOf('/') + 1));
		IFile dotFile = modelProject
				.getFile(modelPath.replace("/" + modelProject.getName() + "/", "") + "_statespace.dot");
		IFile autFile = modelProject
				.getFile(modelPath.replace("/" + modelProject.getName() + "/", "") + "_statespace.aut");
				
		try {
			psDot = new PrintStream(dotFile.getLocationURI().toString().substring(5));
			psAut = new PrintStream(autFile.getLocationURI().toString().substring(5));
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		Grph internalGrph = stateSpace.getGrph();
		now = LocalDateTime.now();  
		System.out.println("just finished exploring state space on "+now);
		System.out.println("################################################res: " + internalGrph.getVertices().size()
				+ " states and " + internalGrph.getEdges().size() + " transitions");
		psDot.print(internalGrph.toDot().replaceAll("##", "#").replaceAll("MSE_","").replaceAll("([^_]*)_[^_]*_([^ ]*)","$1::$2"));
		psDot.close();
		
		createAutStateSpaceFormat(psAut);
	}

	private String prettyPrint(GenericParallelStep aStep) {
		StringBuilder sbStep = new StringBuilder();
		for (GenericStep s : aStep.getSubSteps()) {
			sbStep.append(((SmallStep<?>) s).getMseoccurrence().getMse().getName() + " ");
		}
		return sbStep.toString();
	}

	@Override
	public String engineKindName() {
		return "GEMOC Moccml Exhaustive Concurrent Engine";
	}
	

	private void createAutStateSpaceFormat(PrintStream psAut) {		
				
			Iterator<ControlAndRTDState> iterVertices = stateSpace.getVertices().iterator();
			ControlAndRTDState aState = iterVertices.next();
			if (aState == null){
				System.err.println("no State space to serialize");
				return;
			}
			StringBuilder fileContent = new StringBuilder();
			fileContent.append("des(");
			fileContent.append(stateSpace.v2i(stateSpace.initialState));
			fileContent.append(",");
			fileContent.append(stateSpace.getGrph().getEdges().size());
			fileContent.append(",");
			fileContent.append(stateSpace.getGrph().getVertices().size());
			fileContent.append(")\n");

			//print all transitions
			//TODO check with Luc
//			Iterator<String> iterEdges = stateSpace.getEdges().iterator();
//			while (iterEdges.hasNext()){
//				String transition = iterEdges.next();
//				String aLine= serializeTransition(transition, clockNameToIndex);
//				fileContent.append(aLine);
//			}
			
			for(ControlAndRTDState s1 : stateSpace.getVertices()){
				for(ControlAndRTDState s2 : stateSpace.getVertices()){
					for(StringBuffer t: stateSpace.getEdges(s1, s2)){
						String aLine= "("+(stateSpace.v2i(s1)+", "+ mclFormat(t) +"\", "+stateSpace.v2i(s2))+")";
						fileContent.append(aLine);
						fileContent.append("\n");
					}
				}
			}
			psAut.print(fileContent.toString());
			psAut.close();
		
		
		
	}











	private String mclFormat(StringBuffer t) {
		StringBuffer res = new StringBuffer("\"LS !");
		for(String s : t.toString().split(" ")) {
			res.append(":");
			res.append(s);
		}
		return res.toString().replaceAll("\\[", "").replaceAll("\\]", "");
	}
	
	
	
}
