//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.checkers.checks;

import static org.eclipse.escet.common.java.Lists.list;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.checkers.CifCheck;
import org.eclipse.escet.cif.checkers.CifCheckViolations;
import org.eclipse.escet.cif.common.CifAddressableUtils;
import org.eclipse.escet.cif.common.CifAddressableUtils.DuplVarAsgnException;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.common.RangeCompat;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ProjectionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TupleExpression;
import org.eclipse.escet.cif.metamodel.cif.functions.AssignmentFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.BreakFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.ContinueFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.WhileFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;

/** CIF check to disallow specific statements in internal user-defined functions. */
public class FuncNoSpecificIntUserDefFuncStatsCheck extends CifCheck {
    /** The disallowed statements. */
    private final EnumSet<NoSpecificStatement> disAlloweds;

    /**
     * Constructor of the {@link FuncNoSpecificIntUserDefFuncStatsCheck} class.
     *
     * @param disAlloweds The disallowed statements.
     */
    public FuncNoSpecificIntUserDefFuncStatsCheck(EnumSet<NoSpecificStatement> disAlloweds) {
        this.disAlloweds = disAlloweds;
    }

    /**
     * Constructor of the {@link FuncNoSpecificIntUserDefFuncStatsCheck} class.
     *
     * @param disAlloweds The disallowed statements.
     */
    public FuncNoSpecificIntUserDefFuncStatsCheck(NoSpecificStatement... disAlloweds) {
        this(Arrays.stream(disAlloweds)
                .collect(Collectors.toCollection(() -> EnumSet.noneOf(NoSpecificStatement.class))));
    }

    @Override
    protected void preprocessAssignmentFuncStatement(AssignmentFuncStatement asgStat, CifCheckViolations violations) {
        if (disAlloweds.contains(NoSpecificStatement.MULTI_ASSIGNMENT)) {
            Expression addr = asgStat.getAddressable();
            if (addr instanceof TupleExpression) {
                violations.add(addr, "Internal user-defined function has a multi-assignment");
            }
        }

        if (disAlloweds.contains(NoSpecificStatement.PARTIAL_ASSIGNMENT)) {
            Expression addr = asgStat.getAddressable();
            List<ProjectionExpression> projs = CifAddressableUtils.collectProjs(addr, list());
            for (ProjectionExpression proj: projs) {
                violations.add(proj, "Internal user-defined function has a partial variable assignment");
            }
        }

        if (disAlloweds.contains(NoSpecificStatement.ASSIGN_MULTI_PARTS_SAME_VAR)
                && !disAlloweds.contains(NoSpecificStatement.MULTI_ASSIGNMENT)
                && !disAlloweds.contains(NoSpecificStatement.PARTIAL_ASSIGNMENT))
        {
            Expression addr = asgStat.getAddressable();
            try {
                CifAddressableUtils.getRefs(addr);
            } catch (DuplVarAsgnException ex) {
                violations.add(asgStat, "Internal user-defined function has a multi-assignment "
                        + "that assigns multiple (non-overlapping) parts of a single variable");
            }
        }

        if (disAlloweds.contains(NoSpecificStatement.ASSIGN_OUT_OF_RANGE)) {
            CifType addrType = asgStat.getAddressable().getType();
            CifType valueType = asgStat.getValue().getType();
            if (!CifTypeUtils.checkTypeCompat(addrType, valueType, RangeCompat.CONTAINED)) {
                violations.add(asgStat, "Assignment in a function may fail with an out-of-range error, as the value of "
                        + "the computed expression does not always fit in the assigned variable");
            }
        }
    }

    @Override
    protected void preprocessBreakFuncStatement(BreakFuncStatement breakStat, CifCheckViolations violations) {
        if (disAlloweds.contains(NoSpecificStatement.BREAK)) {
            violations.add(breakStat, "Internal user-defined function has a 'break' statement");
        }
    }

    @Override
    protected void preprocessContinueFuncStatement(ContinueFuncStatement continueStat, CifCheckViolations violations) {
        if (disAlloweds.contains(NoSpecificStatement.CONTINUE)) {
            violations.add(continueStat, "Internal user-defined function has a 'continue' statement");
        }
    }

    @Override
    protected void preprocessWhileFuncStatement(WhileFuncStatement whileStat, CifCheckViolations violations) {
        if (disAlloweds.contains(NoSpecificStatement.WHILE)) {
            violations.add(whileStat, "Internal user-defined function has a 'while' statement");
        }
    }

    /** Disallowed user-defined function statement. */
    public static enum NoSpecificStatement {
        /**
         * Do not allow multi-assignments that assign multiple (parts of) variables in a single assignment statement.
         */
        MULTI_ASSIGNMENT,

        /** Do not allow partial variable assignments assign a part of a variable. */
        PARTIAL_ASSIGNMENT,

        /**
         * Do not allow an assignment to multiple parts of the same variable in internal user-defined functions. Ignored
         * if {@link #MULTI_ASSIGNMENT} or {@link #PARTIAL_ASSIGNMENT} is also specified as being disallowed.
         */
        ASSIGN_MULTI_PARTS_SAME_VAR,

        /**
         * Do not allow an assignment, in internal user-defined functions, that could potentially result in an
         * out-of-range runtime error.
         */
        ASSIGN_OUT_OF_RANGE,

        /** Do not allow the 'break' statement in internal user-defined functions. */
        BREAK,

        /** Do not allow the 'continue' statement in internal user-defined functions. */
        CONTINUE,

        /** Do not allow the 'while' statement in internal user-defined functions. */
        WHILE,
    }
}
