# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::PopulateVulnerabilityDismissalFields, schema: 20230718020825, feature_category: :vulnerability_management do # rubocop:disable Layout/LineLength
  let(:users) { table(:users) }
  let(:namespaces) { table(:namespaces) }
  let(:projects) { table(:projects) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:findings) { table(:vulnerability_occurrences) }
  let(:scanners) { table(:vulnerability_scanners) }
  let(:identifiers) { table(:vulnerability_identifiers) }
  let(:feedback) { table(:vulnerability_feedback) }

  let(:user) { users.create!(name: 'test', email: 'test@example.com', projects_limit: 5) }
  let(:namespace) { namespaces.create!(name: 'gitlab', path: 'gitlab-org') }
  let(:project) { projects.create!(namespace_id: namespace.id, name: 'foo', project_namespace_id: namespace.id) }
  let(:vulnerability_1) do
    vulnerabilities.create!(title: 'title', state: 2, severity: 0,
      confidence: 5, report_type: 2, project_id: project.id, author_id: user.id
    )
  end

  let(:vulnerability_2) do
    vulnerabilities.create!(title: 'title', state: 2, severity: 0,
      confidence: 5, report_type: 2, project_id: project.id, author_id: user.id
    )
  end

  let(:scanner) { scanners.create!(project_id: project.id, external_id: 'foo', name: 'bar') }
  let(:identifier) do
    identifiers.create!(project_id: project.id, fingerprint: 'foo',
      external_type: 'bar', external_id: 'zoo', name: 'identifier'
    )
  end

  let(:uuid) { SecureRandom.uuid }

  before do
    feedback.create!(feedback_type: 0,
      category: 'sast',
      project_fingerprint: '418291a26024a1445b23fe64de9380cdcdfd1fa8',
      project_id: project.id,
      author_id: user.id,
      created_at: Time.current,
      finding_uuid: uuid
    )

    findings.create!(name: 'Finding',
      report_type: 'sast',
      project_fingerprint: '418291a26024a1445b23fe64de9380cdcdfd1f98',
      location_fingerprint: 'bar',
      severity: 1,
      confidence: 1,
      metadata_version: 1,
      raw_metadata: '',
      details: {},
      uuid: uuid,
      project_id: project.id,
      vulnerability_id: vulnerability_1.id,
      scanner_id: scanner.id,
      primary_identifier_id: identifier.id
    )

    allow(::Gitlab::BackgroundMigration::Logger).to receive_messages(info: true, warn: true, error: true)
  end

  subject do
    described_class.new(
      start_id: vulnerability_1.id,
      end_id: vulnerability_2.id,
      batch_table: :vulnerabilities,
      batch_column: :id,
      sub_batch_size: 200,
      pause_ms: 2.minutes,
      connection: ApplicationRecord.connection
    )
  end

  describe '#perform' do
    it 'updates the missing dismissal information of the vulnerability' do
      expect { subject.perform }.to change { vulnerability_1.reload.dismissed_at }
        .from(nil)
        .and change { vulnerability_1.reload.dismissed_by_id }.from(nil).to(user.id)
    end

    it 'writes log messages', :aggregate_failures do
      subject.perform

      expect(::Gitlab::BackgroundMigration::Logger).to have_received(:info).with(migrator: described_class.name,
        message: 'Dismissal information has been copied',
        count: 2
      )
      expect(::Gitlab::BackgroundMigration::Logger).to have_received(:warn).with(migrator: described_class.name,
        message: 'Could not update vulnerability!',
        vulnerability_id: vulnerability_2.id
      )
    end

    context 'when logger throws exception StandardError' do
      before do
        allow(::Gitlab::BackgroundMigration::Logger).to receive(:warn).and_raise(StandardError)
      end

      it 'logs StandardError' do
        expect(::Gitlab::BackgroundMigration::Logger).to receive(:error).with({
          migrator: described_class.name, message: StandardError.to_s, vulnerability_id: vulnerability_2.id
        })

        subject.perform
      end
    end
  end
end
