﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/EdgeOutputConfig.h>
#include <aws/sagemaker/model/Tag.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateEdgePackagingJobRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateEdgePackagingJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEdgePackagingJob"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the edge packaging job.</p>
   */
  inline const Aws::String& GetEdgePackagingJobName() const { return m_edgePackagingJobName; }
  inline bool EdgePackagingJobNameHasBeenSet() const { return m_edgePackagingJobNameHasBeenSet; }
  template <typename EdgePackagingJobNameT = Aws::String>
  void SetEdgePackagingJobName(EdgePackagingJobNameT&& value) {
    m_edgePackagingJobNameHasBeenSet = true;
    m_edgePackagingJobName = std::forward<EdgePackagingJobNameT>(value);
  }
  template <typename EdgePackagingJobNameT = Aws::String>
  CreateEdgePackagingJobRequest& WithEdgePackagingJobName(EdgePackagingJobNameT&& value) {
    SetEdgePackagingJobName(std::forward<EdgePackagingJobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the SageMaker Neo compilation job that will be used to locate
   * model artifacts for packaging.</p>
   */
  inline const Aws::String& GetCompilationJobName() const { return m_compilationJobName; }
  inline bool CompilationJobNameHasBeenSet() const { return m_compilationJobNameHasBeenSet; }
  template <typename CompilationJobNameT = Aws::String>
  void SetCompilationJobName(CompilationJobNameT&& value) {
    m_compilationJobNameHasBeenSet = true;
    m_compilationJobName = std::forward<CompilationJobNameT>(value);
  }
  template <typename CompilationJobNameT = Aws::String>
  CreateEdgePackagingJobRequest& WithCompilationJobName(CompilationJobNameT&& value) {
    SetCompilationJobName(std::forward<CompilationJobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the model.</p>
   */
  inline const Aws::String& GetModelName() const { return m_modelName; }
  inline bool ModelNameHasBeenSet() const { return m_modelNameHasBeenSet; }
  template <typename ModelNameT = Aws::String>
  void SetModelName(ModelNameT&& value) {
    m_modelNameHasBeenSet = true;
    m_modelName = std::forward<ModelNameT>(value);
  }
  template <typename ModelNameT = Aws::String>
  CreateEdgePackagingJobRequest& WithModelName(ModelNameT&& value) {
    SetModelName(std::forward<ModelNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the model.</p>
   */
  inline const Aws::String& GetModelVersion() const { return m_modelVersion; }
  inline bool ModelVersionHasBeenSet() const { return m_modelVersionHasBeenSet; }
  template <typename ModelVersionT = Aws::String>
  void SetModelVersion(ModelVersionT&& value) {
    m_modelVersionHasBeenSet = true;
    m_modelVersion = std::forward<ModelVersionT>(value);
  }
  template <typename ModelVersionT = Aws::String>
  CreateEdgePackagingJobRequest& WithModelVersion(ModelVersionT&& value) {
    SetModelVersion(std::forward<ModelVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role that enables Amazon SageMaker
   * to download and upload the model, and to contact SageMaker Neo.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateEdgePackagingJobRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides information about the output location for the packaged model.</p>
   */
  inline const EdgeOutputConfig& GetOutputConfig() const { return m_outputConfig; }
  inline bool OutputConfigHasBeenSet() const { return m_outputConfigHasBeenSet; }
  template <typename OutputConfigT = EdgeOutputConfig>
  void SetOutputConfig(OutputConfigT&& value) {
    m_outputConfigHasBeenSet = true;
    m_outputConfig = std::forward<OutputConfigT>(value);
  }
  template <typename OutputConfigT = EdgeOutputConfig>
  CreateEdgePackagingJobRequest& WithOutputConfig(OutputConfigT&& value) {
    SetOutputConfig(std::forward<OutputConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services KMS key to use when encrypting the EBS volume the
   * edge packaging job runs on.</p>
   */
  inline const Aws::String& GetResourceKey() const { return m_resourceKey; }
  inline bool ResourceKeyHasBeenSet() const { return m_resourceKeyHasBeenSet; }
  template <typename ResourceKeyT = Aws::String>
  void SetResourceKey(ResourceKeyT&& value) {
    m_resourceKeyHasBeenSet = true;
    m_resourceKey = std::forward<ResourceKeyT>(value);
  }
  template <typename ResourceKeyT = Aws::String>
  CreateEdgePackagingJobRequest& WithResourceKey(ResourceKeyT&& value) {
    SetResourceKey(std::forward<ResourceKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Creates tags for the packaging job.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateEdgePackagingJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateEdgePackagingJobRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_edgePackagingJobName;

  Aws::String m_compilationJobName;

  Aws::String m_modelName;

  Aws::String m_modelVersion;

  Aws::String m_roleArn;

  EdgeOutputConfig m_outputConfig;

  Aws::String m_resourceKey;

  Aws::Vector<Tag> m_tags;
  bool m_edgePackagingJobNameHasBeenSet = false;
  bool m_compilationJobNameHasBeenSet = false;
  bool m_modelNameHasBeenSet = false;
  bool m_modelVersionHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_outputConfigHasBeenSet = false;
  bool m_resourceKeyHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
