﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/partnercentral-selling/PartnerCentralSellingRequest.h>
#include <aws/partnercentral-selling/PartnerCentralSelling_EXPORTS.h>
#include <aws/partnercentral-selling/model/AssigneeContact.h>

#include <utility>

namespace Aws {
namespace PartnerCentralSelling {
namespace Model {

/**
 */
class AssignOpportunityRequest : public PartnerCentralSellingRequest {
 public:
  AWS_PARTNERCENTRALSELLING_API AssignOpportunityRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssignOpportunity"; }

  AWS_PARTNERCENTRALSELLING_API Aws::String SerializePayload() const override;

  AWS_PARTNERCENTRALSELLING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the catalog associated with the request. This field takes a string
   * value from a predefined list: <code>AWS</code> or <code>Sandbox</code>. The
   * catalog determines which environment the opportunity is assigned in. Use
   * <code>AWS</code> to assign real opportunities in the Amazon Web Services
   * catalog, and <code>Sandbox</code> for testing in secure, isolated
   * environments.</p>
   */
  inline const Aws::String& GetCatalog() const { return m_catalog; }
  inline bool CatalogHasBeenSet() const { return m_catalogHasBeenSet; }
  template <typename CatalogT = Aws::String>
  void SetCatalog(CatalogT&& value) {
    m_catalogHasBeenSet = true;
    m_catalog = std::forward<CatalogT>(value);
  }
  template <typename CatalogT = Aws::String>
  AssignOpportunityRequest& WithCatalog(CatalogT&& value) {
    SetCatalog(std::forward<CatalogT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Requires the <code>Opportunity</code>'s unique identifier when you want to
   * assign it to another user. Provide the correct identifier so the intended
   * opportunity is reassigned.</p>
   */
  inline const Aws::String& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = Aws::String>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = Aws::String>
  AssignOpportunityRequest& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the user or team member responsible for managing the assigned
   * opportunity. This field identifies the <i>Assignee</i> based on the partner's
   * internal team structure. Ensure that the email address is associated with a
   * registered user in your Partner Central account.</p>
   */
  inline const AssigneeContact& GetAssignee() const { return m_assignee; }
  inline bool AssigneeHasBeenSet() const { return m_assigneeHasBeenSet; }
  template <typename AssigneeT = AssigneeContact>
  void SetAssignee(AssigneeT&& value) {
    m_assigneeHasBeenSet = true;
    m_assignee = std::forward<AssigneeT>(value);
  }
  template <typename AssigneeT = AssigneeContact>
  AssignOpportunityRequest& WithAssignee(AssigneeT&& value) {
    SetAssignee(std::forward<AssigneeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalog;

  Aws::String m_identifier;

  AssigneeContact m_assignee;
  bool m_catalogHasBeenSet = false;
  bool m_identifierHasBeenSet = false;
  bool m_assigneeHasBeenSet = false;
};

}  // namespace Model
}  // namespace PartnerCentralSelling
}  // namespace Aws
