﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/neptunedata/NeptunedataRequest.h>
#include <aws/neptunedata/Neptunedata_EXPORTS.h>
#include <aws/neptunedata/model/CustomModelTrainingParameters.h>

#include <utility>

namespace Aws {
namespace neptunedata {
namespace Model {

/**
 */
class StartMLModelTrainingJobRequest : public NeptunedataRequest {
 public:
  AWS_NEPTUNEDATA_API StartMLModelTrainingJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartMLModelTrainingJob"; }

  AWS_NEPTUNEDATA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A unique identifier for the new job. The default is An autogenerated
   * UUID.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  StartMLModelTrainingJobRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job ID of a completed model-training job that you want to update
   * incrementally based on updated data.</p>
   */
  inline const Aws::String& GetPreviousModelTrainingJobId() const { return m_previousModelTrainingJobId; }
  inline bool PreviousModelTrainingJobIdHasBeenSet() const { return m_previousModelTrainingJobIdHasBeenSet; }
  template <typename PreviousModelTrainingJobIdT = Aws::String>
  void SetPreviousModelTrainingJobId(PreviousModelTrainingJobIdT&& value) {
    m_previousModelTrainingJobIdHasBeenSet = true;
    m_previousModelTrainingJobId = std::forward<PreviousModelTrainingJobIdT>(value);
  }
  template <typename PreviousModelTrainingJobIdT = Aws::String>
  StartMLModelTrainingJobRequest& WithPreviousModelTrainingJobId(PreviousModelTrainingJobIdT&& value) {
    SetPreviousModelTrainingJobId(std::forward<PreviousModelTrainingJobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job ID of the completed data-processing job that has created the data
   * that the training will work with.</p>
   */
  inline const Aws::String& GetDataProcessingJobId() const { return m_dataProcessingJobId; }
  inline bool DataProcessingJobIdHasBeenSet() const { return m_dataProcessingJobIdHasBeenSet; }
  template <typename DataProcessingJobIdT = Aws::String>
  void SetDataProcessingJobId(DataProcessingJobIdT&& value) {
    m_dataProcessingJobIdHasBeenSet = true;
    m_dataProcessingJobId = std::forward<DataProcessingJobIdT>(value);
  }
  template <typename DataProcessingJobIdT = Aws::String>
  StartMLModelTrainingJobRequest& WithDataProcessingJobId(DataProcessingJobIdT&& value) {
    SetDataProcessingJobId(std::forward<DataProcessingJobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The location in Amazon S3 where the model artifacts are to be stored.</p>
   */
  inline const Aws::String& GetTrainModelS3Location() const { return m_trainModelS3Location; }
  inline bool TrainModelS3LocationHasBeenSet() const { return m_trainModelS3LocationHasBeenSet; }
  template <typename TrainModelS3LocationT = Aws::String>
  void SetTrainModelS3Location(TrainModelS3LocationT&& value) {
    m_trainModelS3LocationHasBeenSet = true;
    m_trainModelS3Location = std::forward<TrainModelS3LocationT>(value);
  }
  template <typename TrainModelS3LocationT = Aws::String>
  StartMLModelTrainingJobRequest& WithTrainModelS3Location(TrainModelS3LocationT&& value) {
    SetTrainModelS3Location(std::forward<TrainModelS3LocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of an IAM role for SageMaker execution.This must be listed in your DB
   * cluster parameter group or an error will occur.</p>
   */
  inline const Aws::String& GetSagemakerIamRoleArn() const { return m_sagemakerIamRoleArn; }
  inline bool SagemakerIamRoleArnHasBeenSet() const { return m_sagemakerIamRoleArnHasBeenSet; }
  template <typename SagemakerIamRoleArnT = Aws::String>
  void SetSagemakerIamRoleArn(SagemakerIamRoleArnT&& value) {
    m_sagemakerIamRoleArnHasBeenSet = true;
    m_sagemakerIamRoleArn = std::forward<SagemakerIamRoleArnT>(value);
  }
  template <typename SagemakerIamRoleArnT = Aws::String>
  StartMLModelTrainingJobRequest& WithSagemakerIamRoleArn(SagemakerIamRoleArnT&& value) {
    SetSagemakerIamRoleArn(std::forward<SagemakerIamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of an IAM role that provides Neptune access to SageMaker and Amazon
   * S3 resources. This must be listed in your DB cluster parameter group or an error
   * will occur.</p>
   */
  inline const Aws::String& GetNeptuneIamRoleArn() const { return m_neptuneIamRoleArn; }
  inline bool NeptuneIamRoleArnHasBeenSet() const { return m_neptuneIamRoleArnHasBeenSet; }
  template <typename NeptuneIamRoleArnT = Aws::String>
  void SetNeptuneIamRoleArn(NeptuneIamRoleArnT&& value) {
    m_neptuneIamRoleArnHasBeenSet = true;
    m_neptuneIamRoleArn = std::forward<NeptuneIamRoleArnT>(value);
  }
  template <typename NeptuneIamRoleArnT = Aws::String>
  StartMLModelTrainingJobRequest& WithNeptuneIamRoleArn(NeptuneIamRoleArnT&& value) {
    SetNeptuneIamRoleArn(std::forward<NeptuneIamRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of ML instance used in preparing and managing training of ML models.
   * This is a CPU instance chosen based on memory requirements for processing the
   * training data and model.</p>
   */
  inline const Aws::String& GetBaseProcessingInstanceType() const { return m_baseProcessingInstanceType; }
  inline bool BaseProcessingInstanceTypeHasBeenSet() const { return m_baseProcessingInstanceTypeHasBeenSet; }
  template <typename BaseProcessingInstanceTypeT = Aws::String>
  void SetBaseProcessingInstanceType(BaseProcessingInstanceTypeT&& value) {
    m_baseProcessingInstanceTypeHasBeenSet = true;
    m_baseProcessingInstanceType = std::forward<BaseProcessingInstanceTypeT>(value);
  }
  template <typename BaseProcessingInstanceTypeT = Aws::String>
  StartMLModelTrainingJobRequest& WithBaseProcessingInstanceType(BaseProcessingInstanceTypeT&& value) {
    SetBaseProcessingInstanceType(std::forward<BaseProcessingInstanceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of ML instance used for model training. All Neptune ML models
   * support CPU, GPU, and multiGPU training. The default is
   * <code>ml.p3.2xlarge</code>. Choosing the right instance type for training
   * depends on the task type, graph size, and your budget.</p>
   */
  inline const Aws::String& GetTrainingInstanceType() const { return m_trainingInstanceType; }
  inline bool TrainingInstanceTypeHasBeenSet() const { return m_trainingInstanceTypeHasBeenSet; }
  template <typename TrainingInstanceTypeT = Aws::String>
  void SetTrainingInstanceType(TrainingInstanceTypeT&& value) {
    m_trainingInstanceTypeHasBeenSet = true;
    m_trainingInstanceType = std::forward<TrainingInstanceTypeT>(value);
  }
  template <typename TrainingInstanceTypeT = Aws::String>
  StartMLModelTrainingJobRequest& WithTrainingInstanceType(TrainingInstanceTypeT&& value) {
    SetTrainingInstanceType(std::forward<TrainingInstanceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The disk volume size of the training instance. Both input data and the output
   * model are stored on disk, so the volume size must be large enough to hold both
   * data sets. The default is 0. If not specified or 0, Neptune ML selects a disk
   * volume size based on the recommendation generated in the data processing
   * step.</p>
   */
  inline int GetTrainingInstanceVolumeSizeInGB() const { return m_trainingInstanceVolumeSizeInGB; }
  inline bool TrainingInstanceVolumeSizeInGBHasBeenSet() const { return m_trainingInstanceVolumeSizeInGBHasBeenSet; }
  inline void SetTrainingInstanceVolumeSizeInGB(int value) {
    m_trainingInstanceVolumeSizeInGBHasBeenSet = true;
    m_trainingInstanceVolumeSizeInGB = value;
  }
  inline StartMLModelTrainingJobRequest& WithTrainingInstanceVolumeSizeInGB(int value) {
    SetTrainingInstanceVolumeSizeInGB(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Timeout in seconds for the training job. The default is 86,400 (1 day).</p>
   */
  inline int GetTrainingTimeOutInSeconds() const { return m_trainingTimeOutInSeconds; }
  inline bool TrainingTimeOutInSecondsHasBeenSet() const { return m_trainingTimeOutInSecondsHasBeenSet; }
  inline void SetTrainingTimeOutInSeconds(int value) {
    m_trainingTimeOutInSecondsHasBeenSet = true;
    m_trainingTimeOutInSeconds = value;
  }
  inline StartMLModelTrainingJobRequest& WithTrainingTimeOutInSeconds(int value) {
    SetTrainingTimeOutInSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Maximum total number of training jobs to start for the hyperparameter tuning
   * job. The default is 2. Neptune ML automatically tunes the hyperparameters of the
   * machine learning model. To obtain a model that performs well, use at least 10
   * jobs (in other words, set <code>maxHPONumberOfTrainingJobs</code> to 10). In
   * general, the more tuning runs, the better the results.</p>
   */
  inline int GetMaxHPONumberOfTrainingJobs() const { return m_maxHPONumberOfTrainingJobs; }
  inline bool MaxHPONumberOfTrainingJobsHasBeenSet() const { return m_maxHPONumberOfTrainingJobsHasBeenSet; }
  inline void SetMaxHPONumberOfTrainingJobs(int value) {
    m_maxHPONumberOfTrainingJobsHasBeenSet = true;
    m_maxHPONumberOfTrainingJobs = value;
  }
  inline StartMLModelTrainingJobRequest& WithMaxHPONumberOfTrainingJobs(int value) {
    SetMaxHPONumberOfTrainingJobs(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Maximum number of parallel training jobs to start for the hyperparameter
   * tuning job. The default is 2. The number of parallel jobs you can run is limited
   * by the available resources on your training instance.</p>
   */
  inline int GetMaxHPOParallelTrainingJobs() const { return m_maxHPOParallelTrainingJobs; }
  inline bool MaxHPOParallelTrainingJobsHasBeenSet() const { return m_maxHPOParallelTrainingJobsHasBeenSet; }
  inline void SetMaxHPOParallelTrainingJobs(int value) {
    m_maxHPOParallelTrainingJobsHasBeenSet = true;
    m_maxHPOParallelTrainingJobs = value;
  }
  inline StartMLModelTrainingJobRequest& WithMaxHPOParallelTrainingJobs(int value) {
    SetMaxHPOParallelTrainingJobs(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the subnets in the Neptune VPC. The default is None.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnets() const { return m_subnets; }
  inline bool SubnetsHasBeenSet() const { return m_subnetsHasBeenSet; }
  template <typename SubnetsT = Aws::Vector<Aws::String>>
  void SetSubnets(SubnetsT&& value) {
    m_subnetsHasBeenSet = true;
    m_subnets = std::forward<SubnetsT>(value);
  }
  template <typename SubnetsT = Aws::Vector<Aws::String>>
  StartMLModelTrainingJobRequest& WithSubnets(SubnetsT&& value) {
    SetSubnets(std::forward<SubnetsT>(value));
    return *this;
  }
  template <typename SubnetsT = Aws::String>
  StartMLModelTrainingJobRequest& AddSubnets(SubnetsT&& value) {
    m_subnetsHasBeenSet = true;
    m_subnets.emplace_back(std::forward<SubnetsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The VPC security group IDs. The default is None.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  StartMLModelTrainingJobRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  StartMLModelTrainingJobRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Key Management Service (KMS) key that SageMaker uses to encrypt
   * data on the storage volume attached to the ML compute instances that run the
   * training job. The default is None.</p>
   */
  inline const Aws::String& GetVolumeEncryptionKMSKey() const { return m_volumeEncryptionKMSKey; }
  inline bool VolumeEncryptionKMSKeyHasBeenSet() const { return m_volumeEncryptionKMSKeyHasBeenSet; }
  template <typename VolumeEncryptionKMSKeyT = Aws::String>
  void SetVolumeEncryptionKMSKey(VolumeEncryptionKMSKeyT&& value) {
    m_volumeEncryptionKMSKeyHasBeenSet = true;
    m_volumeEncryptionKMSKey = std::forward<VolumeEncryptionKMSKeyT>(value);
  }
  template <typename VolumeEncryptionKMSKeyT = Aws::String>
  StartMLModelTrainingJobRequest& WithVolumeEncryptionKMSKey(VolumeEncryptionKMSKeyT&& value) {
    SetVolumeEncryptionKMSKey(std::forward<VolumeEncryptionKMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Key Management Service (KMS) key that SageMaker uses to encrypt
   * the output of the processing job. The default is none.</p>
   */
  inline const Aws::String& GetS3OutputEncryptionKMSKey() const { return m_s3OutputEncryptionKMSKey; }
  inline bool S3OutputEncryptionKMSKeyHasBeenSet() const { return m_s3OutputEncryptionKMSKeyHasBeenSet; }
  template <typename S3OutputEncryptionKMSKeyT = Aws::String>
  void SetS3OutputEncryptionKMSKey(S3OutputEncryptionKMSKeyT&& value) {
    m_s3OutputEncryptionKMSKeyHasBeenSet = true;
    m_s3OutputEncryptionKMSKey = std::forward<S3OutputEncryptionKMSKeyT>(value);
  }
  template <typename S3OutputEncryptionKMSKeyT = Aws::String>
  StartMLModelTrainingJobRequest& WithS3OutputEncryptionKMSKey(S3OutputEncryptionKMSKeyT&& value) {
    SetS3OutputEncryptionKMSKey(std::forward<S3OutputEncryptionKMSKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optimizes the cost of training machine-learning models by using Amazon
   * Elastic Compute Cloud spot instances. The default is <code>False</code>.</p>
   */
  inline bool GetEnableManagedSpotTraining() const { return m_enableManagedSpotTraining; }
  inline bool EnableManagedSpotTrainingHasBeenSet() const { return m_enableManagedSpotTrainingHasBeenSet; }
  inline void SetEnableManagedSpotTraining(bool value) {
    m_enableManagedSpotTrainingHasBeenSet = true;
    m_enableManagedSpotTraining = value;
  }
  inline StartMLModelTrainingJobRequest& WithEnableManagedSpotTraining(bool value) {
    SetEnableManagedSpotTraining(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration for custom model training. This is a JSON object.</p>
   */
  inline const CustomModelTrainingParameters& GetCustomModelTrainingParameters() const { return m_customModelTrainingParameters; }
  inline bool CustomModelTrainingParametersHasBeenSet() const { return m_customModelTrainingParametersHasBeenSet; }
  template <typename CustomModelTrainingParametersT = CustomModelTrainingParameters>
  void SetCustomModelTrainingParameters(CustomModelTrainingParametersT&& value) {
    m_customModelTrainingParametersHasBeenSet = true;
    m_customModelTrainingParameters = std::forward<CustomModelTrainingParametersT>(value);
  }
  template <typename CustomModelTrainingParametersT = CustomModelTrainingParameters>
  StartMLModelTrainingJobRequest& WithCustomModelTrainingParameters(CustomModelTrainingParametersT&& value) {
    SetCustomModelTrainingParameters(std::forward<CustomModelTrainingParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_previousModelTrainingJobId;

  Aws::String m_dataProcessingJobId;

  Aws::String m_trainModelS3Location;

  Aws::String m_sagemakerIamRoleArn;

  Aws::String m_neptuneIamRoleArn;

  Aws::String m_baseProcessingInstanceType;

  Aws::String m_trainingInstanceType;

  int m_trainingInstanceVolumeSizeInGB{0};

  int m_trainingTimeOutInSeconds{0};

  int m_maxHPONumberOfTrainingJobs{0};

  int m_maxHPOParallelTrainingJobs{0};

  Aws::Vector<Aws::String> m_subnets;

  Aws::Vector<Aws::String> m_securityGroupIds;

  Aws::String m_volumeEncryptionKMSKey;

  Aws::String m_s3OutputEncryptionKMSKey;

  bool m_enableManagedSpotTraining{false};

  CustomModelTrainingParameters m_customModelTrainingParameters;
  bool m_idHasBeenSet = false;
  bool m_previousModelTrainingJobIdHasBeenSet = false;
  bool m_dataProcessingJobIdHasBeenSet = false;
  bool m_trainModelS3LocationHasBeenSet = false;
  bool m_sagemakerIamRoleArnHasBeenSet = false;
  bool m_neptuneIamRoleArnHasBeenSet = false;
  bool m_baseProcessingInstanceTypeHasBeenSet = false;
  bool m_trainingInstanceTypeHasBeenSet = false;
  bool m_trainingInstanceVolumeSizeInGBHasBeenSet = false;
  bool m_trainingTimeOutInSecondsHasBeenSet = false;
  bool m_maxHPONumberOfTrainingJobsHasBeenSet = false;
  bool m_maxHPOParallelTrainingJobsHasBeenSet = false;
  bool m_subnetsHasBeenSet = false;
  bool m_securityGroupIdsHasBeenSet = false;
  bool m_volumeEncryptionKMSKeyHasBeenSet = false;
  bool m_s3OutputEncryptionKMSKeyHasBeenSet = false;
  bool m_enableManagedSpotTrainingHasBeenSet = false;
  bool m_customModelTrainingParametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace neptunedata
}  // namespace Aws
