﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lookoutequipment/LookoutEquipmentRequest.h>
#include <aws/lookoutequipment/LookoutEquipment_EXPORTS.h>
#include <aws/lookoutequipment/model/DatasetSchema.h>
#include <aws/lookoutequipment/model/Tag.h>

#include <utility>

namespace Aws {
namespace LookoutEquipment {
namespace Model {

/**
 */
class CreateDatasetRequest : public LookoutEquipmentRequest {
 public:
  AWS_LOOKOUTEQUIPMENT_API CreateDatasetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDataset"; }

  AWS_LOOKOUTEQUIPMENT_API Aws::String SerializePayload() const override;

  AWS_LOOKOUTEQUIPMENT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the dataset being created. </p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  CreateDatasetRequest& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A JSON description of the data that is in each time series dataset, including
   * names, column names, and data types. </p>
   */
  inline const DatasetSchema& GetDatasetSchema() const { return m_datasetSchema; }
  inline bool DatasetSchemaHasBeenSet() const { return m_datasetSchemaHasBeenSet; }
  template <typename DatasetSchemaT = DatasetSchema>
  void SetDatasetSchema(DatasetSchemaT&& value) {
    m_datasetSchemaHasBeenSet = true;
    m_datasetSchema = std::forward<DatasetSchemaT>(value);
  }
  template <typename DatasetSchemaT = DatasetSchema>
  CreateDatasetRequest& WithDatasetSchema(DatasetSchemaT&& value) {
    SetDatasetSchema(std::forward<DatasetSchemaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides the identifier of the KMS key used to encrypt dataset data by Amazon
   * Lookout for Equipment. </p>
   */
  inline const Aws::String& GetServerSideKmsKeyId() const { return m_serverSideKmsKeyId; }
  inline bool ServerSideKmsKeyIdHasBeenSet() const { return m_serverSideKmsKeyIdHasBeenSet; }
  template <typename ServerSideKmsKeyIdT = Aws::String>
  void SetServerSideKmsKeyId(ServerSideKmsKeyIdT&& value) {
    m_serverSideKmsKeyIdHasBeenSet = true;
    m_serverSideKmsKeyId = std::forward<ServerSideKmsKeyIdT>(value);
  }
  template <typename ServerSideKmsKeyIdT = Aws::String>
  CreateDatasetRequest& WithServerSideKmsKeyId(ServerSideKmsKeyIdT&& value) {
    SetServerSideKmsKeyId(std::forward<ServerSideKmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A unique identifier for the request. If you do not set the client request
   * token, Amazon Lookout for Equipment generates one. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateDatasetRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Any tags associated with the ingested data described in the dataset. </p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDatasetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDatasetRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_datasetName;

  DatasetSchema m_datasetSchema;

  Aws::String m_serverSideKmsKeyId;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Vector<Tag> m_tags;
  bool m_datasetNameHasBeenSet = false;
  bool m_datasetSchemaHasBeenSet = false;
  bool m_serverSideKmsKeyIdHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace LookoutEquipment
}  // namespace Aws
