﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/ComponentParameterDetail.h>
#include <aws/imagebuilder/model/ComponentState.h>
#include <aws/imagebuilder/model/ComponentType.h>
#include <aws/imagebuilder/model/Platform.h>
#include <aws/imagebuilder/model/ProductCodeListItem.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace imagebuilder {
namespace Model {

/**
 * <p>A detailed view of a component.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/imagebuilder-2019-12-02/Component">AWS
 * API Reference</a></p>
 */
class Component {
 public:
  AWS_IMAGEBUILDER_API Component() = default;
  AWS_IMAGEBUILDER_API Component(Aws::Utils::Json::JsonView jsonValue);
  AWS_IMAGEBUILDER_API Component& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IMAGEBUILDER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the component.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Component& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the component.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Component& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the component.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  Component& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the component.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Component& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes what change has been made in this version of the component, or what
   * makes this version different from other versions of the component.</p>
   */
  inline const Aws::String& GetChangeDescription() const { return m_changeDescription; }
  inline bool ChangeDescriptionHasBeenSet() const { return m_changeDescriptionHasBeenSet; }
  template <typename ChangeDescriptionT = Aws::String>
  void SetChangeDescription(ChangeDescriptionT&& value) {
    m_changeDescriptionHasBeenSet = true;
    m_changeDescription = std::forward<ChangeDescriptionT>(value);
  }
  template <typename ChangeDescriptionT = Aws::String>
  Component& WithChangeDescription(ChangeDescriptionT&& value) {
    SetChangeDescription(std::forward<ChangeDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The component type specifies whether Image Builder uses the component to
   * build the image or only to test it.</p>
   */
  inline ComponentType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(ComponentType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline Component& WithType(ComponentType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system platform of the component.</p>
   */
  inline Platform GetPlatform() const { return m_platform; }
  inline bool PlatformHasBeenSet() const { return m_platformHasBeenSet; }
  inline void SetPlatform(Platform value) {
    m_platformHasBeenSet = true;
    m_platform = value;
  }
  inline Component& WithPlatform(Platform value) {
    SetPlatform(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The operating system (OS) version supported by the component. If the OS
   * information is available, Image Builder performs a prefix match against the base
   * image OS version during image recipe creation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSupportedOsVersions() const { return m_supportedOsVersions; }
  inline bool SupportedOsVersionsHasBeenSet() const { return m_supportedOsVersionsHasBeenSet; }
  template <typename SupportedOsVersionsT = Aws::Vector<Aws::String>>
  void SetSupportedOsVersions(SupportedOsVersionsT&& value) {
    m_supportedOsVersionsHasBeenSet = true;
    m_supportedOsVersions = std::forward<SupportedOsVersionsT>(value);
  }
  template <typename SupportedOsVersionsT = Aws::Vector<Aws::String>>
  Component& WithSupportedOsVersions(SupportedOsVersionsT&& value) {
    SetSupportedOsVersions(std::forward<SupportedOsVersionsT>(value));
    return *this;
  }
  template <typename SupportedOsVersionsT = Aws::String>
  Component& AddSupportedOsVersions(SupportedOsVersionsT&& value) {
    m_supportedOsVersionsHasBeenSet = true;
    m_supportedOsVersions.emplace_back(std::forward<SupportedOsVersionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the current status of the component.</p>
   */
  inline const ComponentState& GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  template <typename StateT = ComponentState>
  void SetState(StateT&& value) {
    m_stateHasBeenSet = true;
    m_state = std::forward<StateT>(value);
  }
  template <typename StateT = ComponentState>
  Component& WithState(StateT&& value) {
    SetState(std::forward<StateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains parameter details for each of the parameters that the component
   * document defined for the component.</p>
   */
  inline const Aws::Vector<ComponentParameterDetail>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<ComponentParameterDetail>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<ComponentParameterDetail>>
  Component& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = ComponentParameterDetail>
  Component& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The owner of the component.</p>
   */
  inline const Aws::String& GetOwner() const { return m_owner; }
  inline bool OwnerHasBeenSet() const { return m_ownerHasBeenSet; }
  template <typename OwnerT = Aws::String>
  void SetOwner(OwnerT&& value) {
    m_ownerHasBeenSet = true;
    m_owner = std::forward<OwnerT>(value);
  }
  template <typename OwnerT = Aws::String>
  Component& WithOwner(OwnerT&& value) {
    SetOwner(std::forward<OwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Component data contains the YAML document content for the component.</p>
   */
  inline const Aws::String& GetData() const { return m_data; }
  inline bool DataHasBeenSet() const { return m_dataHasBeenSet; }
  template <typename DataT = Aws::String>
  void SetData(DataT&& value) {
    m_dataHasBeenSet = true;
    m_data = std::forward<DataT>(value);
  }
  template <typename DataT = Aws::String>
  Component& WithData(DataT&& value) {
    SetData(std::forward<DataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key identifier used to encrypt the component. This can be either the
   * Key ARN or the Alias ARN. For more information, see <a
   * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id-key-ARN">Key
   * identifiers (KeyId)</a> in the <i>Key Management Service Developer
   * Guide</i>.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  Component& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption status of the component.</p>
   */
  inline bool GetEncrypted() const { return m_encrypted; }
  inline bool EncryptedHasBeenSet() const { return m_encryptedHasBeenSet; }
  inline void SetEncrypted(bool value) {
    m_encryptedHasBeenSet = true;
    m_encrypted = value;
  }
  inline Component& WithEncrypted(bool value) {
    SetEncrypted(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date that Image Builder created the component.</p>
   */
  inline const Aws::String& GetDateCreated() const { return m_dateCreated; }
  inline bool DateCreatedHasBeenSet() const { return m_dateCreatedHasBeenSet; }
  template <typename DateCreatedT = Aws::String>
  void SetDateCreated(DateCreatedT&& value) {
    m_dateCreatedHasBeenSet = true;
    m_dateCreated = std::forward<DateCreatedT>(value);
  }
  template <typename DateCreatedT = Aws::String>
  Component& WithDateCreated(DateCreatedT&& value) {
    SetDateCreated(std::forward<DateCreatedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags that apply to the component.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  Component& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  Component& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains the name of the publisher if this is a third-party component.
   * Otherwise, this property is empty.</p>
   */
  inline const Aws::String& GetPublisher() const { return m_publisher; }
  inline bool PublisherHasBeenSet() const { return m_publisherHasBeenSet; }
  template <typename PublisherT = Aws::String>
  void SetPublisher(PublisherT&& value) {
    m_publisherHasBeenSet = true;
    m_publisher = std::forward<PublisherT>(value);
  }
  template <typename PublisherT = Aws::String>
  Component& WithPublisher(PublisherT&& value) {
    SetPublisher(std::forward<PublisherT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether component source is hidden from view in the console, and
   * from component detail results for API, CLI, or SDK operations.</p>
   */
  inline bool GetObfuscate() const { return m_obfuscate; }
  inline bool ObfuscateHasBeenSet() const { return m_obfuscateHasBeenSet; }
  inline void SetObfuscate(bool value) {
    m_obfuscateHasBeenSet = true;
    m_obfuscate = value;
  }
  inline Component& WithObfuscate(bool value) {
    SetObfuscate(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains product codes that are used for billing purposes for Amazon Web
   * Services Marketplace components.</p>
   */
  inline const Aws::Vector<ProductCodeListItem>& GetProductCodes() const { return m_productCodes; }
  inline bool ProductCodesHasBeenSet() const { return m_productCodesHasBeenSet; }
  template <typename ProductCodesT = Aws::Vector<ProductCodeListItem>>
  void SetProductCodes(ProductCodesT&& value) {
    m_productCodesHasBeenSet = true;
    m_productCodes = std::forward<ProductCodesT>(value);
  }
  template <typename ProductCodesT = Aws::Vector<ProductCodeListItem>>
  Component& WithProductCodes(ProductCodesT&& value) {
    SetProductCodes(std::forward<ProductCodesT>(value));
    return *this;
  }
  template <typename ProductCodesT = ProductCodeListItem>
  Component& AddProductCodes(ProductCodesT&& value) {
    m_productCodesHasBeenSet = true;
    m_productCodes.emplace_back(std::forward<ProductCodesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;

  Aws::String m_name;

  Aws::String m_version;

  Aws::String m_description;

  Aws::String m_changeDescription;

  ComponentType m_type{ComponentType::NOT_SET};

  Platform m_platform{Platform::NOT_SET};

  Aws::Vector<Aws::String> m_supportedOsVersions;

  ComponentState m_state;

  Aws::Vector<ComponentParameterDetail> m_parameters;

  Aws::String m_owner;

  Aws::String m_data;

  Aws::String m_kmsKeyId;

  bool m_encrypted{false};

  Aws::String m_dateCreated;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_publisher;

  bool m_obfuscate{false};

  Aws::Vector<ProductCodeListItem> m_productCodes;
  bool m_arnHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_versionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_changeDescriptionHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_platformHasBeenSet = false;
  bool m_supportedOsVersionsHasBeenSet = false;
  bool m_stateHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
  bool m_ownerHasBeenSet = false;
  bool m_dataHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_encryptedHasBeenSet = false;
  bool m_dateCreatedHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_publisherHasBeenSet = false;
  bool m_obfuscateHasBeenSet = false;
  bool m_productCodesHasBeenSet = false;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
