﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/Filter.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DescribeStoreImageTasksRequest : public EC2Request {
 public:
  AWS_EC2_API DescribeStoreImageTasksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeStoreImageTasks"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The AMI IDs for which to show progress. Up to 20 AMI IDs can be included in a
   * request.</p>
   */
  inline const Aws::Vector<Aws::String>& GetImageIds() const { return m_imageIds; }
  inline bool ImageIdsHasBeenSet() const { return m_imageIdsHasBeenSet; }
  template <typename ImageIdsT = Aws::Vector<Aws::String>>
  void SetImageIds(ImageIdsT&& value) {
    m_imageIdsHasBeenSet = true;
    m_imageIds = std::forward<ImageIdsT>(value);
  }
  template <typename ImageIdsT = Aws::Vector<Aws::String>>
  DescribeStoreImageTasksRequest& WithImageIds(ImageIdsT&& value) {
    SetImageIds(std::forward<ImageIdsT>(value));
    return *this;
  }
  template <typename ImageIdsT = Aws::String>
  DescribeStoreImageTasksRequest& AddImageIds(ImageIdsT&& value) {
    m_imageIdsHasBeenSet = true;
    m_imageIds.emplace_back(std::forward<ImageIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DescribeStoreImageTasksRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filters.</p> <ul> <li> <p> <code>task-state</code> - Returns tasks in a
   * certain state (<code>InProgress</code> | <code>Completed</code> |
   * <code>Failed</code>)</p> </li> <li> <p> <code>bucket</code> - Returns task
   * information for tasks that targeted a specific bucket. For the filter value,
   * specify the bucket name.</p> </li> </ul>  <p>When you specify the
   * <code>ImageIds</code> parameter, any filters that you specify are ignored. To
   * use the filters, you must remove the <code>ImageIds</code> parameter.</p>
   *
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeStoreImageTasksRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeStoreImageTasksRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token returned from a previous paginated request. Pagination continues
   * from the end of the items returned by the previous request.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeStoreImageTasksRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return for this request. To get the next page
   * of items, make another request with the token returned in the output. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html#api-pagination">Pagination</a>.</p>
   * <p>You cannot specify this parameter and the <code>ImageIds</code> parameter in
   * the same call.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeStoreImageTasksRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_imageIds;

  bool m_dryRun{false};

  Aws::Vector<Filter> m_filters;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_imageIdsHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
