﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datasync/DataSyncRequest.h>
#include <aws/datasync/DataSync_EXPORTS.h>
#include <aws/datasync/model/FsxUpdateProtocol.h>

#include <utility>

namespace Aws {
namespace DataSync {
namespace Model {

/**
 */
class UpdateLocationFsxOntapRequest : public DataSyncRequest {
 public:
  AWS_DATASYNC_API UpdateLocationFsxOntapRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateLocationFsxOntap"; }

  AWS_DATASYNC_API Aws::String SerializePayload() const override;

  AWS_DATASYNC_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the Amazon Resource Name (ARN) of the FSx for ONTAP transfer
   * location that you're updating.</p>
   */
  inline const Aws::String& GetLocationArn() const { return m_locationArn; }
  inline bool LocationArnHasBeenSet() const { return m_locationArnHasBeenSet; }
  template <typename LocationArnT = Aws::String>
  void SetLocationArn(LocationArnT&& value) {
    m_locationArnHasBeenSet = true;
    m_locationArn = std::forward<LocationArnT>(value);
  }
  template <typename LocationArnT = Aws::String>
  UpdateLocationFsxOntapRequest& WithLocationArn(LocationArnT&& value) {
    SetLocationArn(std::forward<LocationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the data transfer protocol that DataSync uses to access your Amazon
   * FSx file system.</p>
   */
  inline const FsxUpdateProtocol& GetProtocol() const { return m_protocol; }
  inline bool ProtocolHasBeenSet() const { return m_protocolHasBeenSet; }
  template <typename ProtocolT = FsxUpdateProtocol>
  void SetProtocol(ProtocolT&& value) {
    m_protocolHasBeenSet = true;
    m_protocol = std::forward<ProtocolT>(value);
  }
  template <typename ProtocolT = FsxUpdateProtocol>
  UpdateLocationFsxOntapRequest& WithProtocol(ProtocolT&& value) {
    SetProtocol(std::forward<ProtocolT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a path to the file share in the storage virtual machine (SVM) where
   * you want to transfer data to or from.</p> <p>You can specify a junction path
   * (also known as a mount point), qtree path (for NFS file shares), or share name
   * (for SMB file shares). For example, your mount path might be <code>/vol1</code>,
   * <code>/vol1/tree1</code>, or <code>/share1</code>.</p>  <p>Don't specify a
   * junction path in the SVM's root volume. For more information, see <a
   * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/managing-svms.html">Managing
   * FSx for ONTAP storage virtual machines</a> in the <i>Amazon FSx for NetApp ONTAP
   * User Guide</i>.</p>
   */
  inline const Aws::String& GetSubdirectory() const { return m_subdirectory; }
  inline bool SubdirectoryHasBeenSet() const { return m_subdirectoryHasBeenSet; }
  template <typename SubdirectoryT = Aws::String>
  void SetSubdirectory(SubdirectoryT&& value) {
    m_subdirectoryHasBeenSet = true;
    m_subdirectory = std::forward<SubdirectoryT>(value);
  }
  template <typename SubdirectoryT = Aws::String>
  UpdateLocationFsxOntapRequest& WithSubdirectory(SubdirectoryT&& value) {
    SetSubdirectory(std::forward<SubdirectoryT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_locationArn;

  FsxUpdateProtocol m_protocol;

  Aws::String m_subdirectory;
  bool m_locationArnHasBeenSet = false;
  bool m_protocolHasBeenSet = false;
  bool m_subdirectoryHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataSync
}  // namespace Aws
