/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine.dataregion.wal.io;

import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.OpenOption;
import java.util.Objects;
import org.apache.iotdb.db.service.metrics.WritingMetrics;
import org.apache.iotdb.db.storageengine.dataregion.wal.io.WALFileVersion;
import org.apache.iotdb.db.storageengine.dataregion.wal.io.WALMetaData;
import org.apache.iotdb.db.utils.MmapUtil;
import org.apache.tsfile.compress.IUnCompressor;
import org.apache.tsfile.file.metadata.enums.CompressionType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WALInputStream
extends InputStream
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(WALInputStream.class);
    private final FileChannel channel;
    private final ByteBuffer segmentHeaderWithoutCompressedSizeBuffer = ByteBuffer.allocate(5);
    private final ByteBuffer compressedSizeBuffer = ByteBuffer.allocate(4);
    private ByteBuffer dataBuffer = null;
    private ByteBuffer compressedBuffer = null;
    private final long fileSize;
    File logFile;
    private long endOffset = -1L;
    WALFileVersion version;

    public WALInputStream(File logFile) throws IOException {
        this.channel = FileChannel.open(logFile.toPath(), new OpenOption[0]);
        this.logFile = logFile;
        try {
            this.fileSize = this.channel.size();
            this.analyzeFileVersion();
            this.getEndOffset();
        }
        catch (Exception e) {
            this.channel.close();
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void getEndOffset() throws IOException {
        if (this.channel.size() < (long)(WALFileVersion.V2.getVersionBytes().length + 4)) {
            this.endOffset = this.channel.size();
            return;
        }
        ByteBuffer metadataSizeBuf = ByteBuffer.allocate(4);
        try {
            long position;
            ByteBuffer magicStringBuffer;
            if (this.version == WALFileVersion.V2) {
                magicStringBuffer = ByteBuffer.allocate(this.version.getVersionBytes().length);
                this.channel.read(magicStringBuffer, this.channel.size() - (long)this.version.getVersionBytes().length);
                magicStringBuffer.flip();
                if (this.logFile.getName().endsWith(".checkpoint") || !new String(magicStringBuffer.array(), StandardCharsets.UTF_8).equals(this.version.getVersionString())) {
                    this.endOffset = this.channel.size();
                    return;
                }
                position = this.channel.size() - (long)this.version.getVersionBytes().length - 4L;
            } else {
                if (this.logFile.getName().endsWith(".checkpoint")) {
                    this.endOffset = this.channel.size();
                    return;
                }
                magicStringBuffer = ByteBuffer.allocate(this.version.getVersionBytes().length);
                this.channel.read(magicStringBuffer, this.channel.size() - (long)this.version.getVersionBytes().length);
                magicStringBuffer.flip();
                if (!new String(magicStringBuffer.array(), StandardCharsets.UTF_8).equals(this.version.getVersionString())) {
                    this.endOffset = this.channel.size();
                    return;
                }
                position = this.channel.size() - (long)this.version.getVersionBytes().length - 4L;
            }
            this.channel.read(metadataSizeBuf, position);
            metadataSizeBuf.flip();
            int metadataSize = metadataSizeBuf.getInt();
            this.endOffset = this.channel.size() - (long)this.version.getVersionBytes().length - 4L - (long)metadataSize;
        }
        finally {
            if (this.version == WALFileVersion.V2) {
                this.channel.position(this.version.getVersionBytes().length);
            } else {
                this.channel.position(0L);
            }
        }
    }

    private void analyzeFileVersion() throws IOException {
        this.version = WALFileVersion.getVersion(this.channel);
    }

    @Override
    public int read() throws IOException {
        if (Objects.isNull(this.dataBuffer) || this.dataBuffer.position() >= this.dataBuffer.limit()) {
            this.loadNextSegment();
        }
        return this.dataBuffer.get() & 0xFF;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        if (Objects.isNull(this.dataBuffer) || this.dataBuffer.position() >= this.dataBuffer.limit()) {
            this.loadNextSegment();
        }
        if (this.dataBuffer.remaining() >= len) {
            this.dataBuffer.get(b, off, len);
            return len;
        }
        int toBeRead = len;
        while (toBeRead > 0) {
            int remaining = this.dataBuffer.remaining();
            int bytesRead = Math.min(remaining, toBeRead);
            this.dataBuffer.get(b, off, bytesRead);
            off += bytesRead;
            if ((toBeRead -= bytesRead) <= 0) continue;
            this.loadNextSegment();
        }
        return len;
    }

    @Override
    public void close() throws IOException {
        this.channel.close();
        MmapUtil.clean(this.dataBuffer);
        MmapUtil.clean(this.compressedBuffer);
        this.dataBuffer = null;
        this.compressedBuffer = null;
    }

    @Override
    public int available() throws IOException {
        long size = this.endOffset - this.channel.position();
        if (!Objects.isNull(this.dataBuffer)) {
            size += (long)(this.dataBuffer.limit() - this.dataBuffer.position());
        }
        return (int)size;
    }

    private void loadNextSegment() throws IOException {
        if (this.channel.position() >= this.endOffset) {
            throw new EOFException("Reach the end offset of wal file");
        }
        long startTime = System.nanoTime();
        long startPosition = this.channel.position();
        if (this.version == WALFileVersion.V2) {
            this.loadNextSegmentV2();
        } else if (this.version == WALFileVersion.V1) {
            this.loadNextSegmentV1();
        } else {
            this.tryLoadSegment();
        }
        WritingMetrics.getInstance().recordWALRead(this.channel.position() - startPosition, System.nanoTime() - startTime);
    }

    private void loadNextSegmentV1() throws IOException {
        if (this.channel.position() >= this.fileSize) {
            throw new IOException("Unexpected end of file");
        }
        if (Objects.isNull(this.dataBuffer)) {
            this.dataBuffer = ByteBuffer.allocate(131072);
        }
        this.dataBuffer.clear();
        this.readWALBufferFromChannel(this.dataBuffer);
        this.dataBuffer.flip();
    }

    private void loadNextSegmentV2() throws IOException {
        long position = this.channel.position();
        SegmentInfo segmentInfo = this.getNextSegmentInfo();
        try {
            if (segmentInfo.compressionType != CompressionType.UNCOMPRESSED) {
                if (Objects.isNull(this.dataBuffer) || this.dataBuffer.capacity() < segmentInfo.uncompressedSize || this.dataBuffer.capacity() > segmentInfo.uncompressedSize * 2) {
                    MmapUtil.clean(this.dataBuffer);
                    this.dataBuffer = ByteBuffer.allocateDirect(segmentInfo.uncompressedSize);
                }
                this.dataBuffer.clear();
                if (Objects.isNull(this.compressedBuffer) || this.compressedBuffer.capacity() < segmentInfo.dataInDiskSize || this.compressedBuffer.capacity() > segmentInfo.dataInDiskSize * 2) {
                    MmapUtil.clean(this.compressedBuffer);
                    this.compressedBuffer = ByteBuffer.allocateDirect(segmentInfo.dataInDiskSize);
                }
                this.compressedBuffer.clear();
                this.compressedBuffer.limit(segmentInfo.dataInDiskSize);
                if (this.readWALBufferFromChannel(this.compressedBuffer) != segmentInfo.dataInDiskSize) {
                    throw new IOException("Unexpected end of file");
                }
                this.compressedBuffer.flip();
                IUnCompressor unCompressor = IUnCompressor.getUnCompressor((CompressionType)segmentInfo.compressionType);
                this.uncompressWALBuffer(this.compressedBuffer, this.dataBuffer, unCompressor);
            } else {
                if (Objects.isNull(this.dataBuffer) || this.dataBuffer.capacity() < segmentInfo.dataInDiskSize || this.dataBuffer.capacity() > segmentInfo.dataInDiskSize * 2) {
                    MmapUtil.clean(this.dataBuffer);
                    this.dataBuffer = ByteBuffer.allocateDirect(segmentInfo.dataInDiskSize);
                }
                this.dataBuffer.clear();
                this.dataBuffer.limit(segmentInfo.dataInDiskSize);
                if (this.readWALBufferFromChannel(this.dataBuffer) != segmentInfo.dataInDiskSize) {
                    throw new IOException("Unexpected end of file");
                }
            }
        }
        catch (Exception e) {
            logger.error("Unexpected error when loading a wal segment {} in {}@{}", new Object[]{segmentInfo, this.logFile, position, e});
            throw new IOException(e);
        }
        this.dataBuffer.flip();
    }

    private void tryLoadSegment() throws IOException {
        long originPosition = this.channel.position();
        try {
            this.loadNextSegmentV1();
            this.version = WALFileVersion.V1;
        }
        catch (Throwable e) {
            this.channel.position(originPosition);
            this.loadNextSegmentV2();
            this.version = WALFileVersion.V2;
            logger.info("Failed to load WAL segment in V1 way, try in V2 way successfully.");
        }
    }

    public void skipToGivenLogicalPosition(long pos) throws IOException {
        if (this.version == WALFileVersion.V2) {
            SegmentInfo segmentInfo;
            this.channel.position(this.version.getVersionBytes().length);
            long posRemain = pos;
            do {
                long currentPos = this.channel.position();
                segmentInfo = this.getNextSegmentInfo();
                if (posRemain < (long)segmentInfo.uncompressedSize) break;
                this.channel.position(currentPos + (long)segmentInfo.dataInDiskSize + (long)segmentInfo.headerSize());
            } while ((posRemain -= (long)segmentInfo.uncompressedSize) >= 0L);
            if (segmentInfo.compressionType != CompressionType.UNCOMPRESSED) {
                this.compressedBuffer = ByteBuffer.allocateDirect(segmentInfo.dataInDiskSize);
                this.readWALBufferFromChannel(this.compressedBuffer);
                this.compressedBuffer.flip();
                IUnCompressor unCompressor = IUnCompressor.getUnCompressor((CompressionType)segmentInfo.compressionType);
                this.dataBuffer = ByteBuffer.allocateDirect(segmentInfo.uncompressedSize);
                this.uncompressWALBuffer(this.compressedBuffer, this.dataBuffer, unCompressor);
                MmapUtil.clean(this.compressedBuffer);
                this.compressedBuffer = null;
            } else {
                this.dataBuffer = ByteBuffer.allocateDirect(segmentInfo.dataInDiskSize);
                this.readWALBufferFromChannel(this.dataBuffer);
                this.dataBuffer.flip();
            }
            this.dataBuffer.position((int)posRemain);
        } else {
            this.dataBuffer = null;
            this.channel.position(pos);
        }
    }

    public void read(ByteBuffer buffer) throws IOException {
        int currReadBytes;
        for (int totalBytesToBeRead = buffer.remaining(); totalBytesToBeRead > 0; totalBytesToBeRead -= currReadBytes) {
            if (this.dataBuffer.remaining() == 0) {
                this.loadNextSegment();
            }
            currReadBytes = Math.min(this.dataBuffer.remaining(), totalBytesToBeRead);
            this.dataBuffer.get(buffer.array(), buffer.position(), currReadBytes);
            buffer.position(buffer.position() + currReadBytes);
        }
        buffer.flip();
    }

    public long getFileCurrentPos() throws IOException {
        return this.channel.position();
    }

    public WALMetaData getWALMetaData() throws IOException {
        long position = this.channel.position();
        this.channel.position(0L);
        WALMetaData walMetaData = WALMetaData.readFromWALFile(this.logFile, this.channel);
        this.channel.position(position);
        return walMetaData;
    }

    private SegmentInfo getNextSegmentInfo() throws IOException {
        this.segmentHeaderWithoutCompressedSizeBuffer.clear();
        this.channel.read(this.segmentHeaderWithoutCompressedSizeBuffer);
        this.segmentHeaderWithoutCompressedSizeBuffer.flip();
        SegmentInfo info = new SegmentInfo();
        info.compressionType = CompressionType.deserialize((byte)this.segmentHeaderWithoutCompressedSizeBuffer.get());
        info.dataInDiskSize = this.segmentHeaderWithoutCompressedSizeBuffer.getInt();
        if (info.compressionType != CompressionType.UNCOMPRESSED) {
            this.compressedSizeBuffer.clear();
            this.readWALBufferFromChannel(this.compressedSizeBuffer);
            this.compressedSizeBuffer.flip();
            info.uncompressedSize = this.compressedSizeBuffer.getInt();
        } else {
            info.uncompressedSize = info.dataInDiskSize;
        }
        return info;
    }

    private int readWALBufferFromChannel(ByteBuffer buffer) throws IOException {
        long startTime = System.nanoTime();
        int size = this.channel.read(buffer);
        WritingMetrics.getInstance().recordWALRead(size, System.nanoTime() - startTime);
        return size;
    }

    private void uncompressWALBuffer(ByteBuffer compressed, ByteBuffer uncompressed, IUnCompressor unCompressor) throws IOException {
        long startTime = System.nanoTime();
        unCompressor.uncompress(compressed, uncompressed);
        WritingMetrics.getInstance().recordWALUncompressCost(System.nanoTime() - startTime);
    }

    private static class SegmentInfo {
        public CompressionType compressionType;
        public int dataInDiskSize;
        public int uncompressedSize;

        private SegmentInfo() {
        }

        int headerSize() {
            return this.compressionType == CompressionType.UNCOMPRESSED ? 5 : 9;
        }

        public String toString() {
            return "SegmentInfo{compressionType=" + this.compressionType + ", dataInDiskSize=" + this.dataInDiskSize + ", uncompressedSize=" + this.uncompressedSize + '}';
        }
    }
}

