/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.plc4x;

import java.util.Collections;
import java.util.Map;

import org.apache.camel.Processor;
import org.apache.camel.impl.DefaultCamelContext;
import org.apache.plc4x.java.api.exceptions.PlcConnectionException;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class Plc4XConsumerTest {

    private Plc4XEndpoint endpoint;
    private Processor processor;
    private Plc4XConsumer consumer;

    @BeforeEach
    void setUp() {
        endpoint = mock(Plc4XEndpoint.class);
        processor = mock(Processor.class);

        when(endpoint.getTags()).thenReturn(Collections.emptyMap());
        when(endpoint.getTrigger()).thenReturn(null); // untriggered
        when(endpoint.getCamelContext()).thenReturn(new DefaultCamelContext());

        consumer = new Plc4XConsumer(endpoint, processor);
    }

    @Test
    void doStart() throws Exception {

        doNothing().when(endpoint).setupConnection();
        consumer.doStart();
        verify(endpoint, times(1)).setupConnection();
    }

    @Test
    void doStartBadStart() throws Exception {

        doThrow(new PlcConnectionException("fail"))
                .when(endpoint).setupConnection();
        consumer.doStart();
        verify(endpoint).setupConnection();
        assertFalse(consumer.isStarted());
    }

    @Test
    void shouldStartTriggeredScraperWhenTriggerPresent() throws Exception {

        when(endpoint.getTrigger()).thenReturn("someTrigger");
        when(endpoint.getTags()).thenReturn(Map.of("tag1", "value1"));
        when(endpoint.getPeriod()).thenReturn(1000);
        when(endpoint.getUri()).thenReturn("mock:plc");

        // Avoid real connection
        doNothing().when(endpoint).setupConnection();
        doNothing().when(endpoint).reconnectIfNeeded();
        when(endpoint.createExchange()).thenReturn(mock(org.apache.camel.Exchange.class));
        consumer.doStart();
        verify(endpoint, atLeastOnce()).getTrigger();
    }

    @Test
    void doStop() throws Exception {
        consumer.doStop();
    }

}
