/*****************************************************************************
 * Copyright (c) 2021 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Yoann Farre (CIL4Sys Engineering) - Initial API and implementation
 *
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.docx.internal.util;

import java.util.Arrays;
import java.util.List;

import org.apache.poi.xwpf.usermodel.XWPFParagraph;
import org.apache.poi.xwpf.usermodel.XWPFRun;

/**
 * The purpose of this class is to manage white characters as tab and line break (Bug 571508)
 */
public class TextUtils {

	private static final String DROP_LINE = "\n"; //$NON-NLS-1$

	private static final String TAB = "\t"; //$NON-NLS-1$

	private static final List<String> CHARS_TO_CONVERT = Arrays.asList(DROP_LINE, TAB);

	/**
	 * Fill the paragraph with the given text and convert special chars to docx compliant chars.
	 *
	 * @param p
	 * @param text
	 */
	public static void fillParagraph(XWPFParagraph p, String text) {
		if (text != null) {
			String subString = text;

			while (!subString.isEmpty()) {
				int min = -1;
				String splitDelimiter = ""; //$NON-NLS-1$
				for (String s : CHARS_TO_CONVERT) {
					String[] replacement = subString.split(s, 2);
					if (min == -1) {
						min = replacement[0].length();
						splitDelimiter = s;
					} else if (min != -1 && replacement[0].length() < min) {
						min = replacement[0].length();
						splitDelimiter = s;
					}
				}

				String[] splittedString = subString.split(splitDelimiter, 2);
				String firstPart = splittedString[0];
				XWPFRun newRun = p.createRun();
				newRun.setText(firstPart);

				if (splittedString.length > 1) {
					switch (splitDelimiter) {
					case DROP_LINE:
						newRun.addBreak();
						break;
					case TAB:
						newRun.addTab();
						break;
					default:
						break;
					}
				}

				if (splittedString.length > 1) {
					subString = splittedString[1];
				} else {
					subString = ""; //$NON-NLS-1$
				}
			}
		}
	}
}
