//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2023, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.typechecker.annotations;

import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.annotations.Annotation;

/**
 * CIF annotation provider. Provides functionality to perform additional type checking on one or more annotations of a
 * specific type.
 *
 * <p>
 * In the specification, annotations may be used. The type checker uses annotation provider instances to check all the
 * annotations, one by one, by invoking {@link #checkAnnotation}. After each annotation has been checked individually,
 * the type checker allows the provider to check global constraints, by invoking {@link #checkGlobal}. Providers are not
 * constructed if there are no annotations in the specification for which they are registered. Their
 * {@link #checkGlobal} is thus also not invoked.
 * </p>
 *
 * <p>
 * Per specification, a single instance of an annotation provider is created for related annotations that occur in the
 * specification. Annotations are related when the extension points that register them indicate the same plugin name and
 * class name. By performing checking of related annotations in the same provider, results of the checks can be shared
 * and re-used for improved performance.
 * </p>
 *
 * <p>
 * The type checker will ensure that each annotation has unique named arguments (multiple unnamed arguments is also
 * allowed). Providers may thus assume this constraint holds.
 * </p>
 *
 * <p>
 * The type checker only provides annotations to the provider after component definition/instantiation has been
 * eliminated. This ensures that only concrete objects are annotated with annotations, and that all such objects will
 * have their annotations checked.
 * </p>
 *
 * <p>
 * The type checker first attempts to create the annotation provider with a {@link String} type argument containing the
 * name of the annotation that needs to be checked by the annotation provider. If instantiation fails, the type checker
 * then tries to create the annotation provider without any argument. For annotation providers that handle more than one
 * annotation, it is recommended to use the latter way to instantiate the annotation provider since the string parameter
 * of the former way to create the annotation provider contains only one of the supported annotations.
 * </p>
 *
 * @see AnnotationProviderHelper
 */
public abstract class AnnotationProvider {
    /**
     * Perform additional type checking on an annotation.
     *
     * @param annotatedObject The annotated object.
     * @param annotation The annotation to check.
     * @param reporter The reporter to use to report problems in the annotation.
     */
    public abstract void checkAnnotation(AnnotatedObject annotatedObject, Annotation annotation,
            AnnotationProblemReporter reporter);

    /**
     * Perform additional global type checking (for the entire specification), for the type of annotation provided by
     * this provider.
     *
     * @param spec The specification check.
     * @param reporter The reporter to use to report problems in the specification.
     */
    public abstract void checkGlobal(Specification spec, AnnotationProblemReporter reporter);
}
