//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.checkers.checks;

import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Strings.adjustForPlurals;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.checkers.CifCheck;
import org.eclipse.escet.cif.checkers.CifCheckViolations;
import org.eclipse.escet.cif.common.CifInternalFuncUtils.OrderInternalFunctions;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.functions.ExternalFunction;
import org.eclipse.escet.cif.metamodel.cif.functions.Function;
import org.eclipse.escet.cif.metamodel.cif.functions.InternalFunction;
import org.eclipse.escet.common.java.Assert;

/** CIF check that does not allow specific user-defined functions. */
public class FuncNoSpecificUserDefCheck extends CifCheck {
    /** Set of all kinds of user-defined functions. */
    private static final Set<NoSpecificUserDefFunc> ALL_USER_DEF_FUNCS = EnumSet.of(NoSpecificUserDefFunc.INTERNAL,
            NoSpecificUserDefFunc.EXTERNAL);

    /** The collection of disallowed specific user-defined functions. */
    private final Set<NoSpecificUserDefFunc> disalloweds;

    /**
     * Found internal user-defined functions.
     *
     * <p>
     * Is only used when the {@link NoSpecificUserDefFunc#INTERNAL_RECURSIVE} check is requested.
     * </p>
     */
    private final List<InternalFunction> internalFunctions = list();

    /**
     * Smallest allowed number of return values for an external user-defined function. Initialized with the default
     * value.
     */
    private int externalMinReturnValues = 1;

    /**
     * Smallest allowed number of return values for an internal user-defined function. Initialized with the default
     * value.
     */
    private int internalMinReturnValues = 1;

    /**
     * Largest allowed number of return values for an external user-defined function where {@code null} means infinite.
     * Initialized with the default value.
     */
    private Integer externalMaxReturnValues = null;

    /**
     * Largest allowed number of return values for an internal user-defined function, where {@code null} means infinite.
     * Initialized with the default value.
     */
    private Integer internalMaxReturnValues = null;

    /**
     * Constructor of the {@link FuncNoSpecificUserDefCheck} class.
     *
     * @param disalloweds The collection of disallowed specific user-defined functions.
     */
    public FuncNoSpecificUserDefCheck(Set<NoSpecificUserDefFunc> disalloweds) {
        this.disalloweds = disalloweds;
    }

    /**
     * Constructor of the {@link FuncNoSpecificUserDefCheck} class.
     *
     * @param disalloweds The collection of disallowed specific user-defined functions.
     */
    public FuncNoSpecificUserDefCheck(NoSpecificUserDefFunc... disalloweds) {
        this(Arrays.stream(disalloweds)
                .collect(Collectors.toCollection(() -> EnumSet.noneOf(NoSpecificUserDefFunc.class))));
    }

    /**
     * Set the minimum and maximum allowed number of return values for internal user-defined functions.
     *
     * @param minCount Smallest number of allowed return values. {@code null} means {@code 1}.
     * @param maxCount Largest number of allowed return values. {@code null} means infinite.
     * @return The check instance, for daisy-chaining.
     */
    public FuncNoSpecificUserDefCheck internalFuncReturnValuesLimits(Integer minCount, Integer maxCount) {
        Assert.check(minCount == null || minCount >= 1, "Minimum number of return values for internal functions is 1.");

        internalMinReturnValues = (minCount == null) ? 1 : minCount;
        internalMaxReturnValues = maxCount;

        Assert.check(internalMaxReturnValues == null || (internalMinReturnValues <= internalMaxReturnValues),
                "Internal function return value count limits are conflicting.");
        return this;
    }

    /**
     * Set the minimum and maximum allowed number of return values for external user-defined functions.
     *
     * @param minCount Smallest number of allowed return values. {@code null} means {@code 1}.
     * @param maxCount Largest number of allowed return values. {@code null} means infinite.
     * @return The check instance, for daisy-chaining.
     */
    public FuncNoSpecificUserDefCheck externalFuncReturnValuesLimits(Integer minCount, Integer maxCount) {
        Assert.check(minCount == null || minCount >= 1, "Minimum number of return values for external functions is 1.");

        externalMinReturnValues = (minCount == null) ? 1 : minCount;
        externalMaxReturnValues = maxCount;
        Assert.check(externalMaxReturnValues == null || (externalMinReturnValues <= externalMaxReturnValues),
                "External function return value count limits are conflicting.");
        return this;
    }

    @Override
    protected void preprocessInternalFunction(InternalFunction func, CifCheckViolations violations) {
        checkFunction(func, NoSpecificUserDefFunc.INTERNAL, violations);

        // Collect references to internal user-defined functions for recursion checking if necessary.
        if (!disalloweds.contains(NoSpecificUserDefFunc.INTERNAL) &&
                disalloweds.contains(NoSpecificUserDefFunc.INTERNAL_RECURSIVE))
        {
            internalFunctions.add(func);
        }
    }

    @Override
    protected void preprocessExternalFunction(ExternalFunction func, CifCheckViolations violations) {
        checkFunction(func, NoSpecificUserDefFunc.EXTERNAL, violations);
    }

    @Override
    protected void postprocessSpecification(Specification spec, CifCheckViolations violations) {
        if (!disalloweds.contains(NoSpecificUserDefFunc.INTERNAL) &&
                disalloweds.contains(NoSpecificUserDefFunc.INTERNAL_RECURSIVE) && !internalFunctions.isEmpty())
        {
            OrderInternalFunctions orderer = new OrderInternalFunctions();
            if (orderer.computeOrder(internalFunctions, true) == null) {
                // At least one cycle exists, report it for all functions in the cycle.
                List<InternalFunction> cycle = orderer.getCycle();
                if (cycle.size() == 1) {
                    violations.add(cycle.get(0), "Internal user-defined function calls itself recursively");
                } else {
                    String cyclePath = cycle.stream().map(intFunc -> CifTextUtils.getAbsName(intFunc, false))
                            .collect(Collectors.joining(" -> "));
                    String msg = fmt(
                            "Internal user-defined function is involved in a recursive call cycle: %s -> %s -> ..",
                            cyclePath, CifTextUtils.getAbsName(cycle.get(0), false));

                    for (InternalFunction intFunc: cycle) {
                        violations.add(intFunc, msg);
                    }
                }
            }
        }
    }

    /**
     * Check a user-defined function.
     *
     * @param func Function to check.
     * @param funcKind Kind of the function.
     * @param violations Destination of the reported violations, is modified in-place.
     */
    private void checkFunction(Function func, NoSpecificUserDefFunc funcKind, CifCheckViolations violations) {
        // Check the kind of the function.
        if (disalloweds.contains(funcKind)) {
            if (disalloweds.containsAll(ALL_USER_DEF_FUNCS)) {
                violations.add(func, "Function is a user-defined function");
                return;
            } else {
                String funcText = (funcKind == NoSpecificUserDefFunc.INTERNAL) ? "internal" : "external";
                violations.add(func, "Function is an %s user-defined function", funcText);
                return;
            }
        }

        // Check the number of parameters of the function.
        if (disalloweds.contains(NoSpecificUserDefFunc.NO_PARAMETER) && func.getParameters().isEmpty()) {
            violations.add(func, "Function is a user-defined function without parameters");
        }

        // Check the number of return values of the function.
        String errDesc, funcText;
        if (func instanceof InternalFunction) {
            errDesc = reportNumReturnValues(func, internalMinReturnValues, internalMaxReturnValues);
            funcText = "Internal";
        } else {
            errDesc = reportNumReturnValues(func, externalMinReturnValues, externalMaxReturnValues);
            funcText = "External";
        }

        if (errDesc != null) {
            violations.add(func, "%s user-defined function %s", funcText, errDesc);
        }
    }

    /**
     * Check whether the number of return values of the given function is within the given limits, and report about a
     * found violation.
     *
     * @param func Function to check.
     * @param minLimit Smallest number of allowed return values for the function.
     * @param maxLimit Largest number of allowed return values for the function. {@code null} means infinite.
     * @return {@code null} if the number of return values is within the specified limits, else a string describing what
     *     is wrong.
     */
    private static String reportNumReturnValues(Function func, int minLimit, Integer maxLimit) {
        int numReturnValues = func.getReturnTypes().size();
        if (numReturnValues < minLimit) {
            return adjustForPlurals("has less than {num} return value{p '' s}", minLimit);
        }
        if (maxLimit != null && numReturnValues > maxLimit) {
            return adjustForPlurals("has more than {num} return value{p '' s}", maxLimit);
        }
        return null;
    }

    /** Disallowed specific user-defined functions. */
    public static enum NoSpecificUserDefFunc {
        /** Disallow internal user-defined functions. */
        INTERNAL,

        /** Disallow external user-defined functions. */
        EXTERNAL,

        /** Disallow user-defined functions without a parameter. */
        NO_PARAMETER,

        /**
         * Disallow recursive calling of internal user-defined functions.
         *
         * <p>
         * This setting computes an approximation of the internal user-defined functions that can be involved in a call
         * cycle by searching for all occurrences of internal user-defined functions in expression context within all
         * internal user-defined functions.
         * </p>
         *
         * <p>
         * If all found references are actually used for calling other internal user-defined functions, this check is
         * reliable. If the found references are not actually called, they are used for other purposes, or additional
         * references are brought in at runtime through the parameters of the functions, then the check may draw
         * incorrect conclusions. Depending on the exact case, both false-positives and false-negatives may happen.
         * Disallow the use of functions as data using {@link TypeNoSpecificTypesCheck} and
         * {@link ExprNoSpecificExprsCheck} to prevent these false-positives and false-negatives.
         * </p>
         */
        INTERNAL_RECURSIVE,
    }
}
