## ----global_options, include=FALSE--------------------------------------------
library(knitr)

knitr::opts_chunk$set(
  fig.width = 7, fig.height = 7, warning = FALSE,
  message = FALSE, out.width = "70%"
)
knitr::opts_knit$set(root.dir = tempdir())

pkgs <- c(
  "bootES", "emmeans", "flextable", "ggplot2", "boot", "ggsignif",
  "ggpubr"
)
successfully_loaded <- vapply(pkgs, requireNamespace, FUN.VALUE = logical(1L), quietly = TRUE)
can_evaluate <- all(successfully_loaded)

if (can_evaluate) {
  knitr::opts_chunk$set(eval = TRUE)
  vapply(pkgs, require, FUN.VALUE = logical(1L), quietly = TRUE, character.only = TRUE)
} else {
  knitr::opts_chunk$set(eval = FALSE)
}

## -----------------------------------------------------------------------------
library(rempsyc)

## -----------------------------------------------------------------------------
pkgs <- c(
  "bootES", "emmeans", "flextable", "ggplot2", "boot",
  "ggsignif", "ggpubr"
)
install_if_not_installed(pkgs)

## -----------------------------------------------------------------------------
set.seed(100)
table.stats <- nice_contrasts(
  response = "Sepal.Length",
  group = "Species",
  data = iris
)
table.stats

## -----------------------------------------------------------------------------
(my_table <- nice_table(table.stats))

## ----eval = FALSE-------------------------------------------------------------
# # Open in Word
# print(my_table, preview = "docx")
# 
# # Save in Word
# flextable::save_as_docx(my_table, path = "contrasts.docx")

## ----fig.width=7, fig.height=7------------------------------------------------
(figure <- nice_violin(
  group = "Species",
  response = "Sepal.Length",
  data = iris,
  ytitle = "Length of Sepal",
  signif_annotation = c("***", "***", "***"),
  signif_yposition = c(8.7, 7.3, 8.2),
  signif_xmin = c("setosa", "setosa", "versicolor"),
  signif_xmax = c("virginica", "versicolor", "virginica")
))

## ----eval = FALSE-------------------------------------------------------------
# ggplot2::ggsave("Figure 1.pdf", figure,
#   width = 7, height = 7,
#   unit = "in", dpi = 300
# )

## ----eval = FALSE-------------------------------------------------------------
# data <- read.csv("https://osf.io/qkmnp//?action=download", header = TRUE)

## ----include = FALSE----------------------------------------------------------
data <- structure(list(
  id = 1:90, Group = c(
    "Embodied", "Embodied", "Control",
    "Control", "Mental", "Embodied", "Control", "Control", "Control",
    "Embodied", "Mental", "Mental", "Control", "Control", "Mental",
    "Control", "Mental", "Mental", "Embodied", "Embodied", "Control",
    "Embodied", "Mental", "Control", "Embodied", "Embodied", "Mental",
    "Control", "Mental", "Embodied", "Embodied", "Mental", "Mental",
    "Mental", "Embodied", "Mental", "Embodied", "Mental", "Control",
    "Embodied", "Mental", "Mental", "Embodied", "Mental", "Mental",
    "Embodied", "Control", "Control", "Control", "Mental", "Embodied",
    "Control", "Embodied", "Control", "Mental", "Embodied", "Embodied",
    "Embodied", "Control", "Control", "Control", "Control", "Mental",
    "Control", "Mental", "Embodied", "Embodied", "Mental", "Embodied",
    "Control", "Control", "Mental", "Mental", "Control", "Control",
    "Embodied", "Mental", "Control", "Embodied", "Embodied", "Embodied",
    "Mental", "Control", "Control", "Embodied", "Mental", "Mental",
    "Mental", "Embodied", "Control"
  ), IAT = c(
    -0.033, -0.1471, -0.3913,
    -0.3711, 0.1235, 0.3667, -0.4024, 0.0882, -0.1103, 0.3163, 0.1453,
    -0.1099, 0.4342, 0.6678, -0.1684, 0.1872, 0.0093, 0.1418, 0.0879,
    -0.045, -0.0305, -0.5016, -0.074, -0.0457, 0.3912, 0.5852, 0.012,
    0.4938, 0.0215, 0.1701, 0.2141, -0.5232, -0.0407, 0.0793, -0.2595,
    -0.2917, 0.1874, -0.1617, -0.308, -0.1015, -0.2107, -0.2237,
    -0.2446, 0.5154, 0.2848, 0.1525, -0.3533, 0.1532, -0.248, -0.3699,
    0.0188, 0.567, -0.0729, 0.6243, 0.035, -0.0102, -0.097, -0.0022,
    -0.1266, 0.0909, -0.2806, 0.2326, 0.1366, -0.119, 0.1286, 0.0549,
    -0.1104, 0.0009, -0.1505, 0.1463, 0.6284, 0.4233, -0.4322, -0.245,
    0.0545, -0.2442, 0.5217, -0.4947, 0.5377, -0.0413, 0.3531, 0.122,
    -0.3294, 0.1583, 0.2312, -0.2934, 0.0299, -0.0202, 0.1495, -0.5752
  ), SRS = c(
    1.25, 1.875, 1.375, 1.625, 1.625, 2, 1.625, 1.75,
    2.875, 1.625, 1.125, 1.875, 1.875, 2.375, 1.5, 1.125, 1.25, 1.25,
    1.75, 1.375, 2.125, 1.125, 1.875, 2.125, 2.125, 1.875, 1.5, 1.375,
    1.625, 1.25, 1.125, 1.125, 2.25, 1.875, 1.75, 2.125, 1.375, 1.125,
    1.875, 1.25, 1, 1, 1.25, 1.875, 1.75, 2, 1.5, 1.5, 1.875, 1.375,
    1.25, 1.25, 1.25, 1.5, 2.25, 1.625, 1.75, 1.75, 1.375, 2.5, 1.25,
    1, 1.625, 1.875, 2.125, 1, 2, 1, 1.625, 1.875, 2.375, 1.75, 1.5,
    1.75, 1.5, 2.125, 1.5, 1.125, 1.375, 1.5, 2.75, 1.25, 2, 2.375,
    1.25, 2.125, 1.125, 1.125, 1.625, 1
  ), QCAECE = c(
    3.4, 2.58333333333333,
    3.67777777777778, 3.25, 3.1, 3.89444444444444, 2.62222222222222,
    3.18888888888889, 2.5, 2.86111111111111, 3.4625, 3.01111111111111,
    3.34444444444444, 2.45555555555556, 3.67777777777778, 2.64444444444444,
    2.63333333333333, 2.65, 3.32777777777778, 3.52222222222222, 3.05,
    3.12222222222222, 3.56111111111111, 3.56111111111111, 3.09444444444444,
    2.83333333333333, 2.77222222222222, 2.92222222222222, 3.16666666666667,
    2.61666666666667, 3.46666666666667, 2.09444444444444, 3.45555555555556,
    3.52777777777778, 3.55, 3.2, 2.63888888888889, 3.41111111111111,
    3.11666666666667, 3.68888888888889, 3.45555555555556, 3.28888888888889,
    2.75, 2.67222222222222, 2.61666666666667, 3.54444444444444, 2.41666666666667,
    3.22222222222222, 3.11111111111111, 2.94444444444444, 3.47222222222222,
    3.41111111111111, 3.05555555555556, 3.18888888888889, 3.20555555555556,
    2.94444444444444, 2.15555555555556, 2.94444444444444, 3.78888888888889,
    3.83333333333333, 3.26111111111111, 3.62222222222222, 3.43333333333333,
    2.11111111111111, 3.34444444444444, 3.75, 3.07222222222222, 2.53333333333333,
    2.7625, 3.58888888888889, 3.41111111111111, 2.61111111111111,
    3.3, 2.61111111111111, 3.1, 2.25, 2.84444444444444, 3.48888888888889,
    2.43888888888889, 2.9, 3.03333333333333, 3.36666666666667, 3.21111111111111,
    3.47222222222222, 3.27222222222222, 3.11111111111111, 2.99444444444444,
    3.51666666666667, 3.23888888888889, 3.65
  ), QCAEAE = c(
    3.66666666666667,
    3, 3.41666666666667, 3.25, 3.5, 3.08333333333333, 1.91666666666667,
    3.16666666666667, 1.91666666666667, 3.66666666666667, 2.58333333333333,
    2.41666666666667, 2.75, 3.25, 3.66666666666667, 2.25, 3.5, 2.75,
    2.91666666666667, 3.5, 2.75, 2.58333333333333, 3.41666666666667,
    2.33333333333333, 3, 3.25, 3, 2.83333333333333, 2.91666666666667,
    2.16666666666667, 2.75, 3, 2.33333333333333, 3.25, 2.91666666666667,
    3.33333333333333, 3.16666666666667, 3.33333333333333, 2.58333333333333,
    3.75, 2.91666666666667, 3.83333333333333, 2.66666666666667, 3.25,
    2.5, 2.86111111111111, 2.58333333333333, 2.41666666666667, 3.5,
    3.08333333333333, 3.66666666666667, 2.66666666666667, 2.66666666666667,
    2.41666666666667, 3.33333333333333, 3, 3.5, 2.91666666666667,
    3.08333333333333, 4, 3.41666666666667, 3.75, 2.83333333333333,
    2.75, 3.25, 3.16666666666667, 2.83333333333333, 2.33333333333333,
    3.5, 3.33333333333333, 2.91666666666667, 3.33333333333333, 2.91666666666667,
    3.33333333333333, 2.16666666666667, 2.66666666666667, 3, 2.41666666666667,
    2.83333333333333, 2, 3, 3, 2.66666666666667, 3.83333333333333,
    2.58333333333333, 3.25, 2.91666666666667, 3.25, 3.66666666666667,
    3.91666666666667
  ), QCAEPT = c(
    2.8, 2.5, 3.8, 3.5, 3.2, 3.9, 2.8,
    3.6, 3, 2.5, 3.3, 2.8, 2.8, 1.8, 3.8, 2.4, 2.6, 2.3, 3.1, 3.6,
    3.1, 2.8, 3.9, 3.9, 3.3, 3, 2.1, 3.4, 3, 2.9, 3.6, 1.3, 3.8,
    3.5, 3.1, 3.4, 2.5, 3.6, 2.9, 3.6, 3.8, 2.8, 2.5, 2.9, 2.9, 3.2,
    2.5, 3, 3, 3, 3.5, 3.6, 3, 3.6, 3.3, 3, 2.2, 3, 3.8, 4, 3.3,
    3.8, 3.2, 2, 2.8, 3.5, 2.7, 2.4, 2.9, 3.4, 3.6, 3, 3.6, 3, 3.2,
    2.5, 2.8, 3.2, 2.1, 2.8, 3.4, 3.4, 3.2, 3.5, 3.1, 2.77777777777778,
    3.1, 3.7, 3.7, 3.3
  ), QCAEOS = c(
    4, 2.66666666666667, 3.55555555555556,
    3, 3, 3.88888888888889, 2.44444444444444, 2.77777777777778, 2,
    3.22222222222222, 3.625, 3.22222222222222, 3.88888888888889,
    3.11111111111111, 3.55555555555556, 2.88888888888889, 2.66666666666667,
    3, 3.55555555555556, 3.44444444444444, 3, 3.44444444444444, 3.22222222222222,
    3.22222222222222, 2.88888888888889, 2.66666666666667, 3.44444444444444,
    2.44444444444444, 3.33333333333333, 2.33333333333333, 3.33333333333333,
    2.88888888888889, 3.11111111111111, 3.55555555555556, 4, 3, 2.77777777777778,
    3.22222222222222, 3.33333333333333, 3.77777777777778, 3.11111111111111,
    3.77777777777778, 3, 2.44444444444444, 2.33333333333333, 3.88888888888889,
    2.33333333333333, 3.44444444444444, 3.22222222222222, 2.88888888888889,
    3.44444444444444, 3.22222222222222, 3.11111111111111, 2.77777777777778,
    3.11111111111111, 2.88888888888889, 2.11111111111111, 2.88888888888889,
    3.77777777777778, 3.66666666666667, 3.22222222222222, 3.44444444444444,
    3.66666666666667, 2.22222222222222, 3.88888888888889, 4, 3.44444444444444,
    2.66666666666667, 2.625, 3.77777777777778, 3.22222222222222,
    2.22222222222222, 3, 2.22222222222222, 3, 2, 2.88888888888889,
    3.77777777777778, 2.77777777777778, 3, 2.66666666666667, 3.33333333333333,
    3.22222222222222, 3.44444444444444, 3.44444444444444, 3.44444444444444,
    2.88888888888889, 3.33333333333333, 2.77777777777778, 4
  ), QCAEEC = c(
    4,
    2.75, 4, 2.75, 3.5, 3.25, 2, 3, 2, 4, 2.75, 3, 2.75, 2.75, 4,
    2, 3.5, 3, 3.5, 3.25, 3, 2.75, 3.25, 2, 2.5, 3.5, 3.25, 3.5,
    2.75, 2, 2, 3.75, 2.25, 3.25, 3, 2.5, 3.75, 3.25, 2.5, 3.25,
    2.75, 4, 2.75, 3.25, 3, 2.25, 3.5, 2.5, 4, 3.75, 3.75, 2, 1.75,
    2, 3.25, 2.5, 3.25, 2.5, 3, 4, 3, 3.75, 2.5, 3, 3, 3, 2.25, 2,
    4, 4, 3.5, 3.25, 2.75, 3.25, 2.75, 2.75, 2.75, 1.5, 3, 1.5, 3,
    2.75, 3, 4, 2, 3.75, 3, 2.75, 3.5, 4
  ), QCAEXR = c(
    3.5, 2.75,
    3.25, 3.5, 3.5, 3, 2.25, 3.75, 1.75, 3, 2.75, 2.25, 2.5, 3.75,
    3.75, 2.5, 3.75, 2.75, 2.5, 3.75, 2.5, 2.25, 3.5, 2.5, 3.5, 2.75,
    3, 2.5, 3.25, 1.75, 3.75, 2.75, 3, 3.5, 2.5, 3.5, 3.25, 3.5,
    2.5, 4, 3, 3.75, 2.5, 3.5, 2.25, 3.33333333333333, 2, 2.5, 4,
    2.5, 3.25, 3, 3, 2.25, 3.75, 2.75, 3.75, 2.75, 3.5, 4, 3.5, 4,
    3.25, 2, 4, 3, 3.25, 2.5, 3.75, 3.25, 2.5, 3.75, 3.5, 3.75, 2,
    2.25, 3.5, 2.25, 3.25, 2.25, 3, 3.25, 2.75, 4, 2.75, 3, 3.25,
    3.5, 3.75, 3.75
  ), QCAEPR = c(
    3.5, 3.5, 3, 3.5, 3.5, 3, 1.5, 2.75,
    2, 4, 2.25, 2, 3, 3.25, 3.25, 2.25, 3.25, 2.5, 2.75, 3.5, 2.75,
    2.75, 3.5, 2.5, 3, 3.5, 2.75, 2.5, 2.75, 2.75, 2.5, 2.5, 1.75,
    3, 3.25, 4, 2.5, 3.25, 2.75, 4, 3, 3.75, 2.75, 3, 2.25, 3, 2.25,
    2.25, 2.5, 3, 4, 3, 3.25, 3, 3, 3.75, 3.5, 3.5, 2.75, 4, 3.75,
    3.5, 2.75, 3.25, 2.75, 3.5, 3, 2.5, 2.75, 2.75, 2.75, 3, 2.5,
    3, 1.75, 3, 2.75, 3.5, 2.25, 2.25, 3, 3, 2.25, 3.5, 3, 3, 2.5,
    3.5, 3.75, 4
  ), IRIPT = c(
    4.57142857142857, 3.42857142857143,
    4.14285714285714, 3.71428571428571, 3.14285714285714, 4.71428571428571,
    3.14285714285714, 2.71428571428571, 2, 3.42857142857143, 4, 3.85714285714286,
    4.28571428571429, 3.85714285714286, 3.85714285714286, 4, 3.28571428571429,
    3.14285714285714, 4.28571428571429, 4.14285714285714, 4.14285714285714,
    4.33333333333333, 3.42857142857143, 4.14285714285714, 3.85714285714286,
    3.57142857142857, 4, 3.28571428571429, 3.57142857142857, 2.57142857142857,
    4, 3.71428571428571, 3.28571428571429, 4.42857142857143, 4.71428571428571,
    4.14285714285714, 4, 4, 4.14285714285714, 4.71428571428571, 4,
    4.57142857142857, 3.85714285714286, NA, 2.71428571428571, 4.42857142857143,
    2.85714285714286, 4.28571428571429, 3.71428571428571, 4, 4.14285714285714,
    3.85714285714286, 4, 3.71428571428571, 3.42857142857143, 3, 2.85714285714286,
    3.14285714285714, 3.71428571428571, 4.28571428571429, 3.42857142857143,
    4.28571428571429, 4.14285714285714, 2.71428571428571, 4.71428571428571,
    4.71428571428571, 4.57142857142857, 3.28571428571429, 3.14285714285714,
    4.57142857142857, 4.28571428571429, 2.42857142857143, 3.57142857142857,
    2.42857142857143, 3.57142857142857, 2.71428571428571, 3.57142857142857,
    4, 3.85714285714286, 3.85714285714286, 2.71428571428571, 4.14285714285714,
    4.14285714285714, 4.42857142857143, 3.85714285714286, 4.33333333333333,
    3.85714285714286, 4.57142857142857, 2.28571428571429, 5
  ),
  IRIFS = c(
    4.28571428571429,
    3.71428571428571, 4.28571428571429, 4.28571428571429, 4.14285714285714,
    4.28571428571429, 3.42857142857143, 3.57142857142857, 3.14285714285714,
    4.71428571428571, 2.57142857142857, 3, 4.42857142857143, 4.14285714285714,
    4.14285714285714, 2.14285714285714, 4.42857142857143, 3.28571428571429,
    2.85714285714286, 5, 3.57142857142857, 3.28571428571429, 3.42857142857143,
    4.14285714285714, 3, 4, 3.42857142857143, 3.71428571428571, 3.57142857142857,
    3.57142857142857, 2.14285714285714, 2.28571428571429, 2.57142857142857,
    3.71428571428571, 4.57142857142857, 4.57142857142857, 3, 4.71428571428571,
    3.42857142857143, 4.71428571428571, 4, 3.71428571428571, 3.71428571428571,
    NA, 3, 3.42857142857143, 3.28571428571429, 3.71428571428571,
    4, 3.71428571428571, 4.85714285714286, 3.85714285714286, 4.14285714285714,
    3, 4.42857142857143, 4.71428571428571, 3.85714285714286, 4.28571428571429,
    3.42857142857143, 5, 4.71428571428571, 5, 2.71428571428571, 4,
    4.14285714285714, 4.14285714285714, 2.85714285714286, 3.14285714285714,
    2.71428571428571, 3.57142857142857, 3.85714285714286, 3, 3.28571428571429,
    3, 1.71428571428571, 3.85714285714286, 2.57142857142857, 3.85714285714286,
    3.42857142857143, 2.57142857142857, 4, 4.14285714285714, 3.28571428571429,
    4.85714285714286, 4, 3, 3.14285714285714, 4, 4.42857142857143,
    4.85714285714286
  ), IRIEC = c(
    4.85714285714286, 3.42857142857143,
    3.85714285714286, 4.42857142857143, 4.71428571428571, 4.14285714285714,
    3.57142857142857, 3.57142857142857, 3.28571428571429, 4.14285714285714,
    3.28571428571429, 3.57142857142857, 4.42857142857143, 2.42857142857143,
    4.71428571428571, 3.28571428571429, 4.42857142857143, 3.71428571428571,
    3.71428571428571, 5, 3.42857142857143, 4, 3.71428571428571, 3.28571428571429,
    3.57142857142857, 4.28571428571429, 4.57142857142857, 4, 3.71428571428571,
    3, 4.57142857142857, 3.42857142857143, 3.28571428571429, 4.28571428571429,
    4.14285714285714, 4, 3.85714285714286, 4.42857142857143, 3.85714285714286,
    5, 4.42857142857143, 5, 4, NA, 2.85714285714286, 4.85714285714286,
    3.14285714285714, 3.71428571428571, 3.71428571428571, 3.28571428571429,
    5, 3.71428571428571, 3.42857142857143, 3.14285714285714, 4.28571428571429,
    4.42857142857143, 4.71428571428571, 4.14285714285714, 4.28571428571429,
    5, 5, 4.85714285714286, 3.85714285714286, 2.85714285714286, 4.71428571428571,
    5, 5, 3.85714285714286, 5, 3.71428571428571, 4, 3.85714285714286,
    4.42857142857143, 3.85714285714286, 3.57142857142857, 3.42857142857143,
    4.57142857142857, 4.28571428571429, 4, 3.42857142857143, 3.42857142857143,
    4, 4, 4.42857142857143, 4.28571428571429, 4, 4.85714285714286,
    4.57142857142857, 4.57142857142857, 5
  ), IRIPD = c(
    4.57142857142857,
    3.14285714285714, 1.85714285714286, 3, 2.42857142857143, 4.71428571428571,
    2, 2.71428571428571, 2.14285714285714, 2.71428571428571, 3.14285714285714,
    2.57142857142857, 3.42857142857143, 3, 3.57142857142857, 1.71428571428571,
    3.57142857142857, 1.71428571428571, 1.85714285714286, 3.57142857142857,
    1.57142857142857, 3.42857142857143, 2.28571428571429, 1.71428571428571,
    3, 4, 2.57142857142857, 2.42857142857143, 2.57142857142857, 2.28571428571429,
    2, 4.85714285714286, 2.57142857142857, 2.28571428571429, 3.42857142857143,
    2.28571428571429, 3.71428571428571, 2.42857142857143, 2.71428571428571,
    2.28571428571429, 1.57142857142857, 3.71428571428571, 2.85714285714286,
    NA, 2.14285714285714, 3, 2.85714285714286, 2.71428571428571,
    3.14285714285714, 3.71428571428571, 3, 2.28571428571429, 2.14285714285714,
    2.71428571428571, 3.28571428571429, 4.14285714285714, 4.28571428571429,
    3.71428571428571, 2.14285714285714, 4.14285714285714, 2.85714285714286,
    3.85714285714286, 2, 2.42857142857143, 2.71428571428571, 2.85714285714286,
    2.85714285714286, 3.28571428571429, 3.42857142857143, 2.57142857142857,
    2.71428571428571, 3, 2.71428571428571, 3, 2.28571428571429, 4.14285714285714,
    3.28571428571429, 2.85714285714286, 2, 2, 2.28571428571429, 3.5,
    2.85714285714286, 3.42857142857143, 2.14285714285714, 3.66666666666667,
    2.85714285714286, 2.42857142857143, 3.14285714285714, 3.57142857142857
  ), IOS = c(
    6L, 5L, 2L, 1L, 3L, 2L, 4L, 1L, 1L, 2L, 5L, 4L, 4L,
    1L, 5L, 3L, 4L, 2L, 4L, 3L, 1L, 5L, 1L, 1L, 3L, 7L, 2L, 1L, 6L,
    3L, 2L, 2L, 3L, 3L, 4L, 2L, 5L, 2L, 2L, 2L, 4L, 5L, 3L, 2L, 2L,
    5L, 4L, 3L, 1L, 2L, 4L, 3L, 5L, 2L, 2L, 5L, 2L, 2L, 1L, 2L, 4L,
    2L, 4L, 1L, 4L, 5L, 2L, 2L, 6L, 2L, 4L, 5L, 1L, 5L, 2L, 6L, 4L,
    1L, 3L, 3L, 5L, 3L, 3L, 2L, 6L, 4L, 2L, 2L, 5L, 5L
  ), SCCS = c(
    1.83333333333333,
    2.16666666666667, 1.66666666666667, 3.16666666666667, 2.91666666666667,
    1.91666666666667, 2.83333333333333, 2.75, 3.75, 3.08333333333333,
    2.41666666666667, 3.08333333333333, 2.66666666666667, 2.25, 2,
    4.25, 2.5, 4.16666666666667, 3.41666666666667, 1.5, 2.75, 2.08333333333333,
    2.91666666666667, 3.91666666666667, 2.41666666666667, 2, 2.25,
    1.91666666666667, 2.58333333333333, 2.25, 4.41666666666667, 2.33333333333333,
    1.83333333333333, 2.33333333333333, 3.91666666666667, 4.16666666666667,
    1.08333333333333, 3.58333333333333, 3.58333333333333, 4.16666666666667,
    4, 4.58333333333333, 2.75, 2.66666666666667, 1.58333333333333,
    4, 2.08333333333333, 2.66666666666667, 2.58333333333333, 3.66666666666667,
    3.66666666666667, 2.58333333333333, 3.41666666666667, 2.75, 3.33333333333333,
    1.58333333333333, 2.25, 3.08333333333333, 2.66666666666667, 2.91666666666667,
    2.41666666666667, 3.41666666666667, 3.58333333333333, 2.08333333333333,
    2.16666666666667, 4, 4.41666666666667, 2.16666666666667, 3.5,
    2.75, 2.5, 3, 1.58333333333333, 3, 4.25, 2.66666666666667, 2.25,
    2.58333333333333, 3.25, 3, 3.08333333333333, 2.25, 2.58333333333333,
    1.83333333333333, 4.08333333333333, 2.5, 3.08333333333333, 3.5,
    1.5, 3.66666666666667
  ), MC = c(
    6, 5, 4.66666666666667, 5, 4.33333333333333,
    5.9, 6.66666666666667, 6, 4, 2.9, 1, 3.66666666666667, 5.33333333333333,
    4, 4, 2, 5, 6.33333333333333, 4.7, 4.9, 4.33333333333333, 4.2,
    1.66666666666667, 6, 3.3, 5.3, 1.66666666666667, 6.66666666666667,
    5.66666666666667, 4.3, 5.1, 3.33333333333333, 4.33333333333333,
    3.33333333333333, 4.9, 2.66666666666667, 4.2, 4, 3.33333333333333,
    3.1, 3.66666666666667, 1, 4.6, 2.33333333333333, 2.33333333333333,
    4.5, 3.66666666666667, 5.66666666666667, 5.33333333333333, 4.33333333333333,
    4.8, 4, 3.1, 5.33333333333333, 2.33333333333333, 5.7, 3.6, 3.1,
    4.66666666666667, 6.33333333333333, 5.66666666666667, 7, 3.33333333333333,
    4.66666666666667, 5.66666666666667, 3.2, 1.7, 2, 4.3, 4.66666666666667,
    6.66666666666667, 4, 6, 3.66666666666667, 6.33333333333333, 5.1,
    2, 6.33333333333333, 3.7, 5.2, 4.4, 3.66666666666667, 5.33333333333333,
    6.66666666666667, 3.6, 3.66666666666667, 2.33333333333333, 0.666666666666667,
    4.5, 3.66666666666667
  )
), class = "data.frame", row.names = c(
  NA,
  -90L
))

## -----------------------------------------------------------------------------
(data$Group <- factor(data$Group, levels = c("Embodied", "Mental", "Control")))

## -----------------------------------------------------------------------------
library(dplyr)
(DV <- data %>% select(QCAEPR:IOS) %>% names())

## -----------------------------------------------------------------------------
set.seed(100)
table.stats <- nice_contrasts(
  response = DV,
  group = "Group",
  data = data
)
table.stats

## -----------------------------------------------------------------------------
table.stats[1] <- rep(c(
  "Peripheral Responsivity (QCAE)",
  "Perspective-Taking (IRI)",
  "Fantasy (IRI)", "Empathic Concern (IRI)",
  "Personal Distress (IRI)",
  "Inclusion of Other in the Self (IOS)"
), each = 3)

## -----------------------------------------------------------------------------
(my_table <- nice_table(table.stats,
  highlight = TRUE,
  title = c("Table 1", "Results of multiple regression with planned contrasts analyses for empathy subscales and self–other merging (confirmatory analyses with the exception of self–other merging)"),
  note = "\U1D451\U1D3F = robust Cohen’s \U1D451; CI = bootstrapped confidence interval. The comparisons were between-groups only (i.e., there were no within-subject pre/post comparisons). One participant did not complete the Interpersonal Reactivity Index. Bold/Grey background values represent statistically significant differences between the groups on that row and variable."
))

## ----eval = FALSE-------------------------------------------------------------
# # Open in Word
# print(my_table, preview = "docx")
# 
# # Save in Word
# flextable::save_as_docx(my_table, path = "contrasts.docx")

## -----------------------------------------------------------------------------
Data <- na.omit(data)

## -----------------------------------------------------------------------------
(EC <- nice_violin(
  group = "Group",
  response = "IRIEC",
  data = Data,
  colours = c("#00BA38", "#619CFF", "#F8766D"),
  ytitle = "Empathic Concern (from IRI)",
  signif_annotation = "*",
  signif_yposition = 5.2,
  signif_xmin = 1,
  signif_xmax = 3
))

## -----------------------------------------------------------------------------
(PD <- nice_violin(
  group = "Group",
  response = "IRIPD",
  data = Data,
  colours = c("#00BA38", "#619CFF", "#F8766D"),
  ytitle = "Personal Distress (from IRI)",
  signif_annotation = "*",
  signif_yposition = 5,
  signif_xmin = 1,
  signif_xmax = 3
))

## -----------------------------------------------------------------------------
(PR <- nice_violin(
  group = "Group",
  response = "QCAEPR",
  data = Data,
  colours = c("#00BA38", "#619CFF", "#F8766D"),
  ytitle = "Peripheral Responsivity (from QCAE)",
  signif_annotation = "*",
  signif_yposition = 4.2,
  signif_xmin = 1,
  signif_xmax = 3
))

## -----------------------------------------------------------------------------
(IOS <- nice_violin(
  group = "Group",
  response = "IOS",
  data = Data,
  colours = c("#00BA38", "#619CFF", "#F8766D"),
  ytitle = "Self-Other Merging (from IOS)",
  signif_annotation = c("***", "*", "*"),
  signif_yposition = c(8.25, 7.5, 6.75),
  signif_xmin = c(1, 1, 2),
  signif_xmax = c(3, 2, 3)
))

## ----fig.width=14, fig.height=14, out.width="100%"----------------------------
library(ggpubr)
(figure <- ggarrange(EC, PD, PR, IOS,
  labels = "AUTO",
  ncol = 2, nrow = 2
))

## ----eval = FALSE-------------------------------------------------------------
# ggplot2::ggsave("Figure 1.pdf", figure,
#   width = 14, height = 14,
#   unit = "in", dpi = 300
# )

