﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/Parameter.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Redshift {
namespace Model {

/**
 * <p>Describes the default cluster parameters for a parameter group
 * family.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/DefaultClusterParameters">AWS
 * API Reference</a></p>
 */
class DefaultClusterParameters {
 public:
  AWS_REDSHIFT_API DefaultClusterParameters() = default;
  AWS_REDSHIFT_API DefaultClusterParameters(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_REDSHIFT_API DefaultClusterParameters& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_REDSHIFT_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the cluster parameter group family to which the engine default
   * parameters apply.</p>
   */
  inline const Aws::String& GetParameterGroupFamily() const { return m_parameterGroupFamily; }
  inline bool ParameterGroupFamilyHasBeenSet() const { return m_parameterGroupFamilyHasBeenSet; }
  template <typename ParameterGroupFamilyT = Aws::String>
  void SetParameterGroupFamily(ParameterGroupFamilyT&& value) {
    m_parameterGroupFamilyHasBeenSet = true;
    m_parameterGroupFamily = std::forward<ParameterGroupFamilyT>(value);
  }
  template <typename ParameterGroupFamilyT = Aws::String>
  DefaultClusterParameters& WithParameterGroupFamily(ParameterGroupFamilyT&& value) {
    SetParameterGroupFamily(std::forward<ParameterGroupFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates the starting point for the next set of response
   * records in a subsequent request. If a value is returned in a response, you can
   * retrieve the next set of records by providing this returned marker value in the
   * <code>Marker</code> parameter and retrying the command. If the
   * <code>Marker</code> field is empty, all response records have been retrieved for
   * the request. </p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DefaultClusterParameters& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of cluster default parameters.</p>
   */
  inline const Aws::Vector<Parameter>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<Parameter>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<Parameter>>
  DefaultClusterParameters& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = Parameter>
  DefaultClusterParameters& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_parameterGroupFamily;

  Aws::String m_marker;

  Aws::Vector<Parameter> m_parameters;
  bool m_parameterGroupFamilyHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
