﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class RunStatementRequest : public GlueRequest {
 public:
  AWS_GLUE_API RunStatementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RunStatement"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Session Id of the statement to be run.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  RunStatementRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The statement code to be run.</p>
   */
  inline const Aws::String& GetCode() const { return m_code; }
  inline bool CodeHasBeenSet() const { return m_codeHasBeenSet; }
  template <typename CodeT = Aws::String>
  void SetCode(CodeT&& value) {
    m_codeHasBeenSet = true;
    m_code = std::forward<CodeT>(value);
  }
  template <typename CodeT = Aws::String>
  RunStatementRequest& WithCode(CodeT&& value) {
    SetCode(std::forward<CodeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The origin of the request.</p>
   */
  inline const Aws::String& GetRequestOrigin() const { return m_requestOrigin; }
  inline bool RequestOriginHasBeenSet() const { return m_requestOriginHasBeenSet; }
  template <typename RequestOriginT = Aws::String>
  void SetRequestOrigin(RequestOriginT&& value) {
    m_requestOriginHasBeenSet = true;
    m_requestOrigin = std::forward<RequestOriginT>(value);
  }
  template <typename RequestOriginT = Aws::String>
  RunStatementRequest& WithRequestOrigin(RequestOriginT&& value) {
    SetRequestOrigin(std::forward<RequestOriginT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sessionId;

  Aws::String m_code;

  Aws::String m_requestOrigin;
  bool m_sessionIdHasBeenSet = false;
  bool m_codeHasBeenSet = false;
  bool m_requestOriginHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
