﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 */
class UpdateGroupRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API UpdateGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateGroup"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the group that you want to update.</p>
   */
  inline const Aws::String& GetGroupName() const { return m_groupName; }
  inline bool GroupNameHasBeenSet() const { return m_groupNameHasBeenSet; }
  template <typename GroupNameT = Aws::String>
  void SetGroupName(GroupNameT&& value) {
    m_groupNameHasBeenSet = true;
    m_groupName = std::forward<GroupNameT>(value);
  }
  template <typename GroupNameT = Aws::String>
  UpdateGroupRequest& WithGroupName(GroupNameT&& value) {
    SetGroupName(std::forward<GroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the user pool that contains the group you want to update.</p>
   */
  inline const Aws::String& GetUserPoolId() const { return m_userPoolId; }
  inline bool UserPoolIdHasBeenSet() const { return m_userPoolIdHasBeenSet; }
  template <typename UserPoolIdT = Aws::String>
  void SetUserPoolId(UserPoolIdT&& value) {
    m_userPoolIdHasBeenSet = true;
    m_userPoolId = std::forward<UserPoolIdT>(value);
  }
  template <typename UserPoolIdT = Aws::String>
  UpdateGroupRequest& WithUserPoolId(UserPoolIdT&& value) {
    SetUserPoolId(std::forward<UserPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A new description of the existing group.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role that you want to associate with
   * the group. The role assignment contributes to the <code>cognito:roles</code> and
   * <code>cognito:preferred_role</code> claims in group members' tokens.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  UpdateGroupRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A non-negative integer value that specifies the precedence of this group
   * relative to the other groups that a user can belong to in the user pool. Zero is
   * the highest precedence value. Groups with lower <code>Precedence</code> values
   * take precedence over groups with higher or null <code>Precedence</code> values.
   * If a user belongs to two or more groups, it is the group with the lowest
   * precedence value whose role ARN is given in the user's tokens for the
   * <code>cognito:roles</code> and <code>cognito:preferred_role</code> claims.</p>
   * <p>Two groups can have the same <code>Precedence</code> value. If this happens,
   * neither group takes precedence over the other. If two groups with the same
   * <code>Precedence</code> have the same role ARN, that role is used in the
   * <code>cognito:preferred_role</code> claim in tokens for users in each group. If
   * the two groups have different role ARNs, the <code>cognito:preferred_role</code>
   * claim isn't set in users' tokens.</p> <p>The default <code>Precedence</code>
   * value is null. The maximum <code>Precedence</code> value is
   * <code>2^31-1</code>.</p>
   */
  inline int GetPrecedence() const { return m_precedence; }
  inline bool PrecedenceHasBeenSet() const { return m_precedenceHasBeenSet; }
  inline void SetPrecedence(int value) {
    m_precedenceHasBeenSet = true;
    m_precedence = value;
  }
  inline UpdateGroupRequest& WithPrecedence(int value) {
    SetPrecedence(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_groupName;

  Aws::String m_userPoolId;

  Aws::String m_description;

  Aws::String m_roleArn;

  int m_precedence{0};
  bool m_groupNameHasBeenSet = false;
  bool m_userPoolIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_precedenceHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
