/******************************************************************************
 * (c) Copyright 2002-2004, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: HTTPBridgeAccessor.java,v $
 * Version:       $Name:  $ $Revision: 1.35 $
 * Last Modified: $Date: 2008-10-09 19:48:36 $
 *****************************************************************************/
package org.ten60.transport.http.bridge;

import com.ten60.netkernel.scheduler.Scheduler;
import com.ten60.netkernel.urii.IURAspect;
import com.ten60.netkernel.urii.IURMeta;
import com.ten60.netkernel.urii.IURRepresentation;
import com.ten60.netkernel.urii.URIdentifier;
import com.ten60.netkernel.urii.aspect.IAspectBinaryStream;
import com.ten60.netkernel.urii.aspect.IAspectReadableBinaryStream;
import com.ten60.netkernel.urii.aspect.NetKernelExceptionAspect;
import com.ten60.netkernel.urrequest.URRequest;
import com.ten60.netkernel.urrequest.URResult;
import com.ten60.netkernel.util.IXMLException;
import com.ten60.netkernel.util.NetKernelException;
import com.ten60.netkernel.util.NetKernelError;
import com.ten60.netkernel.util.IXMLException;
import com.ten60.netkernel.util.PairList;
import com.ten60.netkernel.util.Utils;

import org.ten60.netkernel.layer1.accessor.AccessorImpl;
import org.ten60.netkernel.layer1.meta.ActiveAccessorMeta;
import org.ten60.netkernel.layer1.meta.AlwaysExpiredMeta;
import org.ten60.netkernel.layer1.representation.*;
import org.ten60.netkernel.layer1.util.CompoundURIdentifier;

import org.ten60.netkernel.xml.representation.IXAspect;
import org.ten60.netkernel.xml.xda.DOMXDA;

import org.ten60.transport.http.aspect.IAspectHTTPBridgeConfig;
import org.ten60.transport.http.aspect.IAspectHTTPRequestResponse;
import org.ten60.transport.http.cookie.representation.CookieAspect;
import org.ten60.transport.http.cookie.representation.IAspectCookie;
import org.ten60.transport.http.parameter.representation.ParameterNVPAspect;
import org.ten60.transport.http.parameter.representation.ParameterUploadAspect;
import org.ten60.transport.http.response.representation.IAspectResponseCode;
import org.ten60.transport.http.util.MultiPartRequest;
import org.ten60.transport.jetty.HttpHandler;

import org.mortbay.http.HttpFields;
import org.mortbay.http.HttpRequest;
import org.mortbay.http.HttpResponse;

import java.io.*;
import java.util.*;
import java.util.regex.*;
import javax.servlet.http.Cookie;
import java.text.SimpleDateFormat;
/**
 * 	<p>
 * 	An HTTP Bridge implementing the Mapper pattern.
 * 	</p>
 * 	<p>
 * 	Processes Raw HTTP requests from the HTTP Transport.  Uses
 * 	an HTTP BRidge Config Zone resource to specify how to process the
 * 	external HTTPRequest.
 * 	</p>
 * 	<p>
 * 	The processed request is issued to the internal NetKernel URI address space.
 * 	</p>
 *	<p>
 *	A SOAP mode provides support for processing SOAP Messages over HTTP.  Both SOAP 1.1 and 1.2
 *	are implemented.  HTTP response codes are set according to the SOAP standards.
 *	</p>
 * 	<p>
 * 	Returned results are received, processed and written
 * 	into the external HTTPResponse.
 * 	</p>
 * 	<p>
 * 	Finally a VoidAspect is sent to the HTTPTRansport to indicate that the Bridge is done
 * 	and the HTTP connection can be closed.
 * 	</p>
 * @author pjr
 */
public class HTTPBridgeAccessor extends AccessorImpl
{
	private static final URIdentifier URI_PARAM1 = new URIdentifier("literal:param1");
	private static final URIdentifier URI_PARAM2 = new URIdentifier("literal:param2");
	private static final URIdentifier URI_PARAM3 = new URIdentifier("literal:param3");
	private static final URIdentifier URI_COOKIE = new URIdentifier("literal:cookie");
	private static final URIdentifier URI_REMOTE_HOST = new URIdentifier("literal:remoteHost");
	private static final URIdentifier URI_METHOD = new URIdentifier("literal:method");
	private static final URIdentifier URI_REQUEST_URL = new URIdentifier("literal:requestURL");
	private static final String URI_UPLOAD = new String("literal:upload");
	
	/**
	 * The maximum size of any HTTP parameters to accept - upper bound to prevent denial of service
	 */
	public static int MAX_PARAMETER_SIZE=50000;
	
	private IURMeta mLiteralTextMeta = new AlwaysExpiredMeta("text/plain", 0);
	
	/** Creates a new instance of HTTPBridgeAccessor */
	public HTTPBridgeAccessor()
	{	super(new ActiveAccessorMeta(4, true));
	}
	
	/**
	 * 	Handle HTTP Bridge Requests
	 * @param aRequest The Request to be Bridged - this will always originate from the HTTPTransport
	 */
	public void requestAsync(URRequest aRequest)
	{	try
		{	if(aRequest.getType()!=URRequest.RQT_SOURCE)
			{	throw new NetKernelException("HTTPBridge only supports Source");
			}
			else
			{	source(aRequest);
			}
		}
		catch(Throwable e)
		{	NetKernelException nke=new NetKernelException("Problem with source in HTTPBridge");
			nke.addCause(e);
			IURRepresentation result=NetKernelExceptionAspect.create(nke);
			URResult res=new URResult(aRequest,result);
			getScheduler().receiveAsyncException(res);
		}
	}
	
	/*
	private HTTPBridgeZone getZone(URRequest aRequest) throws NetKernelException
	{	CompoundURIdentifier curi=new CompoundURIdentifier(aRequest.getURI());
		String configURI=curi.get("config");
		IAspectHTTPBridgeConfig config=(IAspectHTTPBridgeConfig)getResource(new URIdentifier(configURI), IAspectHTTPBridgeConfig.class, aRequest).getAspect(IAspectHTTPBridgeConfig.class);
		String requestURI = curi.get("url");
		return config.getHTTPBridgeConfig().getZone(requestURI);
	}
	*/
	
	/**
	 * Use the settings specified in an HTTPBridgeConfig Zone to process the
	 * HTTPRequest into an Internal NetKernel URRequest.
	 * @param aRequest The request containing the HTTP request to be processed
	 * @throws Throwable Pass any exceptions to the Kernel to handle
	 */
	protected void source(URRequest aRequest) throws Throwable
	{	CompoundURIdentifier curi=new CompoundURIdentifier(aRequest.getURI());
		String configURI=curi.get("config");
		IAspectHTTPBridgeConfig config=(IAspectHTTPBridgeConfig)getResource(new URIdentifier(configURI), IAspectHTTPBridgeConfig.class, aRequest).getAspect(IAspectHTTPBridgeConfig.class);
		String rawURL = curi.get("url");
		HTTPBridgeZone zone=config.getHTTPBridgeConfig().getZone(rawURL);
		boolean passMethod=zone.shouldPassMethod();
		boolean processQueries=zone.shouldProcessQueries();
		boolean passCookies=zone.shouldPassCookies();
		boolean passRequestURL=zone.shouldPassRequestURL();
		boolean passRemoteHost=zone.shouldPassRemoteHost();
		
		//Get the raw HTTPRequestResponse
		IAspectHTTPRequestResponse arr=(IAspectHTTPRequestResponse)aRequest.getArg(HttpHandler.HTTP_LITERAL_URI).getAspect(IAspectHTTPRequestResponse.class);
		HttpRequest httpRequest=arr.getHttpRequest();
		HttpResponse httpResponse=arr.getHttpResponse();
		
		//Start working through the options
		PairList requestArguments=new PairList(4);
		
		try
		{	// check content length
			if(zone.getMaximumEntitySize()>0)
			{	if(!httpRequest.containsField("Content-Length")) throw new ContentSizeRequiredException();
				if(httpRequest.getContentLength()>zone.getMaximumEntitySize()) throw new ContentTooBigException();
			}
			/*Switch on SOAP mode*/
			if(zone.isSOAPMode())
			{   SOAPMode(aRequest, httpRequest, httpResponse, zone);
			}
			else /*Normal Mode*/
			{
				/*Start building URI*/
				CompoundURIdentifier requestURI = null; 
				//requestURI.append("jetty");
				int colonIndex = rawURL.indexOf(':');
				int qmIndex = rawURL.indexOf('?');
				if(qmIndex>0)
				{	requestURI=new CompoundURIdentifier(new URIdentifier("jetty:"+rawURL.substring(colonIndex+1, qmIndex)));
				}
				else
				{	requestURI=new CompoundURIdentifier(new URIdentifier("jetty:"+rawURL.substring(colonIndex+1)));
				}
				
				// multipart content
				String mimetype=httpRequest.getMimeType();
				HashMap multipartNVP=null;
				if (mimetype!=null)
				{	if(mimetype.equals("multipart/form-data"))
					{	MultiPartRequest mpr=new MultiPartRequest(httpRequest);
						String[] names=mpr.getPartNames();
						multipartNVP=new HashMap(names.length);
						for(int i=0; i<names.length;i++)
						{	String filename=mpr.getFilename(names[i]);
							if(filename!=null)
							{	// uploaded file
								URIdentifier partURI = new URIdentifier(URI_UPLOAD+Integer.toString(i));
								requestURI.addArg(names[i],partURI.toString());
								requestArguments.put( partURI, ParameterUploadAspect.create(new AlwaysExpiredMeta("type/unknown",0), mpr,names[i], filename) );
								multipartNVP.put(names[i], new ByteArrayInputStream(filename.getBytes()));
							}
							else //Must be form data so add to NVP
							{	multipartNVP.put(names[i], mpr.getInputStream(names[i]));
							}
						}
					}
				}
				
				// Handle any non-special body content- e.g. a POSTed XML doc
				// If query data is available and also POST/PUT data
				// then query data goes in param2 argument
				if (mimetype!=null && !mimetype.equals("application/x-www-form-urlencoded") && !mimetype.equals("multipart/form-data"))
				{	    if (httpRequest.getContentLength() > 0)
						{
							IURMeta meta = new AlwaysExpiredMeta(mimetype,0);
							ByteArrayOutputStream baos = new ByteArrayOutputStream(httpRequest.getContentLength());
							Utils.pipe(httpRequest.getInputStream(), baos);
							IURAspect aspect = new ByteArrayAspect(baos, httpRequest.getCharacterEncoding());
							IURRepresentation param = new MonoRepresentationImpl(meta,aspect);
							URIdentifier paramURI = URI_PARAM1;
							requestURI.addArg("param",paramURI.toString());
							requestArguments.put(paramURI, param);
						}
						if(httpRequest.getParameterNames().size()>0)
						{	    IURRepresentation temp=ParameterNVPAspect.create( new AlwaysExpiredMeta("text/nvp", 0),httpRequest);
								URIdentifier paramURI = URI_PARAM2;
								requestURI.addArg("param2",paramURI.toString());
								requestArguments.put(paramURI, temp);
						}
				}
				
				/*	handle the various modes in which data can be supplied
				 *	All data will be put in param argument.
				 */
				else if(httpRequest.getMethod().equals("POST") || (httpRequest.getMethod().equals("GET") && processQueries) || httpRequest.getMethod().equals("PUT") || httpRequest.getMethod().equals("DELETE"))
				{	IURRepresentation param=null;
					//Let Jetty handle Queries/Urlencoded form data.
					IURRepresentation temp=null;
					if(httpRequest.getParameterNames().size()>0)
					{	param=ParameterNVPAspect.create( new AlwaysExpiredMeta("text/nvp", 0),httpRequest);
					}
					else if(multipartNVP!=null)		//We have multipart form data that Jetty doesn't handle
					{	Set keys=multipartNVP.keySet();
						Iterator it=keys.iterator();
						StringBuffer sb=new StringBuffer(2048);
						sb.append("<nvp>");
						while(it.hasNext())
						{	String key=(String)it.next();
							InputStream is=(InputStream)multipartNVP.get(key);
							StringWriter sw=new StringWriter();
							while(is.available()>0)
							{	sw.write(is.read());
							}
							sb.append("<"+key+">");
							sb.append(sw.getBuffer());
							sb.append("</"+key+">");
						}
						sb.append("</nvp>");
						IURAspect aspect = new StringAspect(sb.toString());
						param=new MonoRepresentationImpl(new AlwaysExpiredMeta("text/xml", 0), aspect);
					}
					if (param!=null)
					{	URIdentifier paramURI = URI_PARAM3;
						requestURI.addArg("param",paramURI.toString());
						requestArguments.put(paramURI, param);
					}
				}
				
				
				//Process Cookies
				if(passCookies)
				{	IURRepresentation cjp=null;
					URIdentifier cookieURI=null;
					Cookie[] cookies=httpRequest.getCookies();
					if(cookies.length>0)
					{	HashMap proxies = new HashMap(cookies.length);
						for(int i=0; i<cookies.length; i++)
						{	proxies.put("cookie"+i, CookieAspect.create(cookies[i]));
						}
						IURRepresentation vp = VoidAspect.create();
						cjp=MultiPartAspect.create(vp,proxies);
					}
					if (cjp!=null)
					{	cookieURI = URI_COOKIE;
						requestURI.addArg("cookie",cookieURI.toString());
						requestArguments.put(cookieURI,cjp);
					}
				}
				//Process Method
				if(passMethod)
				{	if(zone.shouldPassByURI())
					{	requestURI.addArg("method","data:text/plain,"+httpRequest.getMethod());
					}
					else
					{	URIdentifier paramURI = URI_METHOD;
						requestURI.addArg("method",paramURI.toString());
						requestArguments.put(
							paramURI,
							createLiteralStringRep(httpRequest.getMethod())
						);
					}
				}
				//Pass RemoateHost
				if(passRemoteHost)
				{	if(zone.shouldPassByURI())
					{	requestURI.addArg("Remote-Host","data:text/plain,"+CompoundURIdentifier.encode(httpRequest.getRemoteHost()));					
					}
					else
					{	URIdentifier paramURI = URI_REMOTE_HOST;
						requestURI.addArg("Remote-Host",paramURI.toString());
						requestArguments.put(
						paramURI,
						createLiteralStringRep(httpRequest.getRemoteHost())
						);
					}
				}
				
				//Process Headers
				for (Iterator i = zone.getHeaders().iterator(); i.hasNext(); )
				{	
					String h=(String)i.next();
					try
					{	String value=httpRequest.getField(h);
						if(zone.shouldPassByURI())
						{	requestURI.addArg(h,"data:text/plain,"+CompoundURIdentifier.encode(value));
						}
						else
						{	URIdentifier paramURI = new URIdentifier("literal:"+h);
							requestURI.addArg(h,paramURI.toString());
							requestArguments.put(
							paramURI,
							createLiteralStringRep(value)
							);
						}
					}
					catch(Exception e)
					{	/*Header doesn't exist - ignore */
					}
				}
				//Process Request URL
				if(passRequestURL)
				{	requestURI.addArg("requestURL",httpRequest.getRootURL().toString()+httpRequest.getURI());
				}
				//Build the request
				URRequest request = new URRequest(
				requestURI.toURI(),
				this,
				aRequest.getSession(),
				this.getModule(),
				URRequest.RQT_SOURCE,
				null,
				aRequest,
				IAspectBinaryStream.class
				);
				
				// add pass-by-value arguments to request
				for (int i=0; i<requestArguments.size(); i++)
				{	URIdentifier uri = (URIdentifier)requestArguments.getValue1(i);
					IURRepresentation rep = (IURRepresentation)requestArguments.getValue2(i);
					request.addArg(uri,rep);
				}
				
				//Issue request (changed to be synchronous)
				URResult result =getScheduler().requestSynch(request);
				processResult(aRequest, result,zone);
			}
		}
		catch(ContentTooBigException e)
		{	httpResponse.setStatus(httpResponse.__413_Request_Entity_Too_Large);
			httpResponse.setReason("Request Entity length exceeds application maximum");
			issueVoidResponse(aRequest,0);
		}
		catch(ContentSizeRequiredException e)
		{	httpResponse.setStatus(httpResponse.__411_Length_Required);
			httpResponse.setReason("Application requires Content-Length");
			issueVoidResponse(aRequest,0);
		}
		catch (NetKernelException e)
		{	IURRepresentation response=NetKernelExceptionAspect.create(e);
			handleFailure( response, httpResponse, httpRequest, aRequest, zone);
			issueVoidResponse(aRequest,0);
		}
		catch (NetKernelError e)
		{	IURRepresentation response=NetKernelExceptionAspect.create(e);
			handleFailure( response, httpResponse, httpRequest, aRequest, zone);
			issueVoidResponse(aRequest,0);
		}
	}
	
	private IURRepresentation createLiteralStringRep(String aValue)
	{	IURAspect aspect = new StringAspect(aValue);
		return new MonoRepresentationImpl(mLiteralTextMeta,aspect);
	}
	
	/** send back a dummy void response so that transport can complete
	 */
	private void issueVoidResponse(URRequest aRequest, int aCost)
	{	URResult res=new URResult(aRequest,com.ten60.netkernel.urii.aspect.VoidAspect.create(aCost));
		getScheduler().receiveAsyncResult(res);
	}
	
	public void processResult(URRequest originalRequest, URResult aResult, HTTPBridgeZone zone)
	{	int cost=0;
		IAspectHTTPRequestResponse arr=(IAspectHTTPRequestResponse)originalRequest.getArg(HttpHandler.HTTP_LITERAL_URI).getAspect(IAspectHTTPRequestResponse.class);
		HttpRequest httpRequest=arr.getHttpRequest();
		HttpResponse httpResponse=arr.getHttpResponse();
		IURRepresentation response=aResult.getResource();
		
		try
		{   //HTTPBridgeZone zone = getZone(aResult.getRequest().getParent());
			if(zone.isSOAPMode())
			{	outputSOAP(aResult.getRequest(), response, httpRequest, httpResponse );
			}
			else /*Normal Mode*/
			{    outputOk(aResult.getRequest(), response, httpRequest, httpResponse, true );
			}
			cost = response.getMeta().getCreationCost()+response.getMeta().getUsageCost();
		}
		catch (NetKernelException e)
		{	IURRepresentation error = NetKernelExceptionAspect.create(e);
			handleFailure(error, httpResponse, httpRequest, originalRequest, zone);
		}
		catch (Throwable e)
		{	NetKernelException nke = new NetKernelException("Unhandled Exception in HTTPBridge");
			nke.addCause(e);
			IURRepresentation error = NetKernelExceptionAspect.create(nke);
			handleFailure(error, httpResponse, httpRequest, originalRequest, zone);
		}
		finally
		{	issueVoidResponse(originalRequest,cost);
		}
	}
	
	/**
	 * Try to locate an exception handler
	 * @param aResult An exception returned from the URI Address Space
	 */
	/*
	public void receiveAsyncException(URResult aResult)
	{
		URRequest originalRequest = aResult.getRequest().getParent();
		IAspectHTTPRequestResponse arr=(IAspectHTTPRequestResponse)originalRequest.getArg(HttpHandler.HTTP_LITERAL_URI).getAspect(IAspectHTTPRequestResponse.class);
		HttpRequest httpRequest=arr.getHttpRequest();
		HttpResponse httpResponse=arr.getHttpResponse();
		IURRepresentation response=aResult.getResource();
		handleFailure( response, httpResponse, httpRequest, originalRequest);
		issueVoidResponse(originalRequest,0);
	}
	*/
	
	/**
	 *	General purpose error handler - tries to invoke a registered Error Process URI
	 *	Falls back to wrting the serialized NetKernelException to the HTTP response.
	 */
	private void handleFailure(IURRepresentation response, HttpResponse httpResponse, HttpRequest httpRequest, URRequest aRequest, HTTPBridgeZone zone)
	{	IXMLException e =((NetKernelExceptionAspect)response.getAspect(NetKernelExceptionAspect.class)).getXMLException();
		boolean rawOutputNeeded=false;
		try
		{   
			//HTTPBridgeZone zone = getZone(aRequest);
			if(zone.isSOAPMode())
			{   /*the SOAP Server should never allow an Exception to get through to us - they should all be
			 *returned as SOAP Faults.  However it's happened so lets do out best...
			 */
				boolean SOAP11=httpRequest.getField("SOAPAction")!=null;
				issueSOAPServerFault(httpResponse, SOAP11, e.toString());
			}
			else  /*Normal Mode*/
			{   if (zone.getExceptionURI()!=null && zone.getExceptionURI().length()>0)
				{	// use exception handler
					httpResponse.setStatus(HttpResponse.__500_Internal_Server_Error);
					String id=httpRequest.getRemoteHost();
					String requestURI=zone.getExceptionURI();
					URIdentifier paramURI=null;
					if (response!=null)
					{	paramURI = URI_PARAM1;
						String suffix = "+param@";
						requestURI+=suffix+paramURI.toString();
					}
					URRequest request = new URRequest(new URIdentifier(requestURI),	this,aRequest.getSession(),	this.getModule(),URRequest.RQT_SOURCE,	null,aRequest,IAspectBinaryStream.class	);
					if (response!=null)
					{	request.addArg(paramURI,response);
					}
					//Issue request for exception process
					Scheduler sch=(Scheduler)getContainer().getComponent(Scheduler.URI);
					IURRepresentation result=sch.requestSynch(request).getResource();
					if (!result.hasAspect(NetKernelExceptionAspect.class))
					{	outputOk(aRequest, result, httpRequest, httpResponse, false);
						rawOutputNeeded=false;
					}
					else
					{	// exception handler failed so output to console to help developer
						System.out.println("hhtp-bridge: "+((NetKernelExceptionAspect)result.getAspect(NetKernelExceptionAspect.class)).getXMLException().toString());
						rawOutputNeeded=true;
					}
				}

				else
				{	// no exception handler
					String id = e.getDeepestId();
					int code;
					if (id.equals("com.ten60.netkernel.scheduler.NoAccessorFoundException"))
					{	// accessor not found for requested URI
						code = HttpResponse.__404_Not_Found;
					}
					else if (id.equals("Error sourcing resource"))
					{	// resource not found in module
						code = HttpResponse.__404_Not_Found;
					}
					else
					{	code = HttpResponse.__500_Internal_Server_Error;
					}
					httpResponse.setStatus(code);
					rawOutputNeeded=true;
				}
			}
		}
		catch(IOException ex)
		{	// ignore as this is usually when a client doesn't want response
		}
		catch(Throwable ex)
		{   System.out.println("http:bridge: "+ex.toString());
			ex.printStackTrace();
			rawOutputNeeded=true;
		}
		finally
		{
		}				
				
		if (rawOutputNeeded)
		{	try
			{	NetKernelExceptionAspect nkep=(NetKernelExceptionAspect)response.getAspect(NetKernelExceptionAspect.class);
				httpResponse.setContentType("text/xml");
				httpResponse.setDateField("Expires",0);
				OutputStream os=httpResponse.getOutputStream();
				nkep.write(os);
				os.flush();
			}
			catch (Throwable ex2)
			{ /* nothing more we can do */ }
		}
	}
	
	/**
	 * Process application specified response codes
	 */
	private void processResponseCode(IAspectResponseCode aResponseCode, HttpResponse httpResponse)
	{	httpResponse.setStatus(aResponseCode.getCode());
		if (aResponseCode.getReason()!=null)
		{	httpResponse.setReason(aResponseCode.getReason());
		}
		PairList headers = aResponseCode.getHeaders();
		for (int j=headers.size()-1; j>=0; j--)
		{	String name=(String)headers.getValue1(j);
			String value=(String)headers.getValue2(j);
			httpResponse.setField(name,value);
		}
		//Do special case processing - TODO ensure each code is HTTP/1.1  complient for headers, entity etc.
		
	}
	
	/**
	 *	Process the result into the HTTP response stream.
	 *	Takes care of Cookies and handle HTTP Response
	 *  codes set by the application.
	 */
	private void outputOk(URRequest aRequest, IURRepresentation response, HttpRequest httpRequest, HttpResponse httpResponse, boolean ifmodifiedtest) throws IOException, NetKernelException
	{	boolean writeStream=true;
		// handle multipart stuff (cookies or response code overrides)
		if (response.hasAspect(IAspectMultipart.class))
		{	IAspectMultipart mp=(IAspectMultipart)response.getAspect(IAspectMultipart.class);
			IURRepresentation[] parts=mp.getParts();
			for(int i=0; i<parts.length;i++)
			{	// process any cookies
				if(parts[i].hasAspect(IAspectCookie.class))
				{	IAspectCookie cp=(IAspectCookie)parts[i].getAspect(IAspectCookie.class);
					httpResponse.addSetCookie(cp.getCookie());
					ifmodifiedtest=false;
				}
				// process any response code overrides
				else if(parts[i].hasAspect(IAspectResponseCode.class))
				{	processResponseCode((IAspectResponseCode)parts[i].getAspect(IAspectResponseCode.class), httpResponse);
					ifmodifiedtest=false;
				}
			}
		}
		//Etag Processing
		if(ifmodifiedtest && ( httpRequest.getMethod().equals(httpRequest.__GET)|| httpRequest.getMethod().equals(httpRequest.__HEAD)) )
		{	String etag=httpRequest.getField(HttpFields.__IfNoneMatch);  //Doesn't matter if null
			String tagString=Integer.toHexString(response.hashCode());
			if(tagString.equals(etag))
			{	//Not modified
				writeStream=false;
				httpResponse.setStatus(httpResponse.__304_Not_Modified);
			}
			//Always write the Etag
			httpResponse.setField(HttpFields.__ETag,tagString);
		}
		// write main representation to body
		if(writeStream)
		{	long expires = response.getMeta().getPessimisticExpiryTime();
			httpResponse.setDateField(HttpFields.__Expires,expires);
			String mimeType=response.getMeta().getMimeType();
			if (mimeType!=null)
			{	if (mimeType.equals("text/html"))
				{	mimeType+=";charset=UTF-8";
				}
				httpResponse.setContentType(mimeType);
			}
			
			if (response.hasAspect(IAspectReadableBinaryStream.class))
			{	IAspectReadableBinaryStream rbs = (IAspectReadableBinaryStream)response.getAspect(IAspectReadableBinaryStream.class);
				int length = rbs.getContentLength();
				if (length>=0)
				{	httpResponse.setIntField("Content-Length",length);
				}
				InputStream is = rbs.getInputStream();
				Utils.pipe(is,httpResponse.getOutputStream());
			}
			else
			{	ByteArrayOutputStream baos = new ByteArrayOutputStream(4096);
				IAspectBinaryStream pbs = (IAspectBinaryStream)response.getAspect(IAspectBinaryStream.class);
				pbs.write(baos);
				baos.flush();
				OutputStream os = httpResponse.getOutputStream();
				os.write(baos.toByteArray());
				os.flush();
			}
		}
	}
	
	/*Perform SOAP 1.1 and 1.2 binding compatible processing of HTTP Request - Issue processed input message
	 *to service:wsSOAPServer or service:wsdlGenerator with endpoint and action uris.
	 *
	 *Note attachments are not yet supported.
	 */
	private void SOAPMode(URRequest aRequest, HttpRequest aHttpRequest, HttpResponse aHttpResponse, HTTPBridgeZone zone) throws Throwable
	{   boolean SOAP11=aHttpRequest.getField("SOAPAction")!=null;
		URResult result=null;
		try
		{   try
			{
				String endpoint=aHttpRequest.getRequestURL().toString();
				String query=aHttpRequest.getQuery();
				URRequest request=null;
				if(query!=null)
				{   boolean iswsdl=query.toUpperCase().equals("WSDL");
					boolean isdoc=query.toUpperCase().startsWith("DOC");
					if(iswsdl || isdoc)
					{	String service;
						if(iswsdl) service="service:wsdlGenerator";
						else
						{   service="service:wsDocumentation";
							int idx=query.indexOf('+');
							if(idx>0) service+=query.substring(idx);
						}
						CompoundURIdentifier curi=new CompoundURIdentifier(new URIdentifier(service));
						curi.addArg("endpoint", endpoint);
						request = new URRequest(
						curi.toURI(),
						this,
						aRequest.getSession(),
						this.getModule(),
						URRequest.RQT_SOURCE,
						null,
						aRequest,
						IURAspect.class
						);
					}
					else throw new Exception("Query not supported");
				}
				else
				{
					String action=aHttpRequest.getField("SOAPAction");
					if(action!=null)action=action.substring(1, action.length()-1); //Remove quotation marks
					String method=aHttpRequest.getMethod();
					boolean methodGET=method.equals("GET");
					if(methodGET && action!=null) throw new SOAPModeClientException("SOAP 1.1 does not support HTTP GET verb");
					String contentType=aHttpRequest.getField("Content-Type");
					if(action==null && contentType!=null) /*Must be SOAP 1.2*/
					{   int j=contentType.indexOf("action=");
						if(j>0)
						{   action=contentType.substring(j+8);  //Get action and remove quotation marks.
							j=action.indexOf("\"");
							action=action.substring(0,j);
						}
					}
					//HTTPBridgeZone zone=this.getZone(aRequest);
					String config=zone.getSOAPConfig();
					CompoundURIdentifier curi=new CompoundURIdentifier(new URIdentifier("service:wsSOAPServer"));
					curi.addArg("endpoint", endpoint);
					if(action!=null) curi.addArg("action", action);
					URIdentifier messageURI=URI_PARAM1;
					if(!methodGET) curi.addArg("message", messageURI.toString());
					if(zone.getSOAPConfig()!=null) curi.addArg("config", zone.getSOAPConfig());
					//Build the request
					request = new URRequest(
					curi.toURI(),
					this,
					aRequest.getSession(),
					this.getModule(),
					URRequest.RQT_SOURCE,
					null,
					aRequest,
					IXAspect.class
					);
					if(!methodGET)
					{	IURMeta meta = new AlwaysExpiredMeta(contentType,0);
						ByteArrayOutputStream baos = new ByteArrayOutputStream(aHttpRequest.getContentLength());
						Utils.pipe(aHttpRequest.getInputStream(), baos);
						IURAspect aspect = new ByteArrayAspect(baos);
						IURRepresentation message = new MonoRepresentationImpl(meta,aspect);
						request.addArg(messageURI, message);
					}
					
				}
				/*Issue Request to SOAP Server*/
				result=getScheduler().requestSynch(request);
			}
			catch(NetKernelException nke)
			{   nke.printStackTrace();
				throw new SOAPModeServerException(nke.getMessage());
			}
			catch(IOException e)
			{   e.printStackTrace();
				throw new SOAPModeServerException(e.getMessage());
			}
		}
		catch(SOAPModeClientException e)
		{	/*Issue SOAP Client Fault*/
			e.printStackTrace();
			try
			{   StringBuffer sb=new StringBuffer(1024);
				if(SOAP11) /*SOAP1.1 Fault*/
				{	sb.append("<env:Envelope   xmlns:env=\"http://schemas.xmlsoap.org/soap/envelope/\">  <env:Body> <env:Fault>");
					sb.append("<faultcode>env:Client</faultcode>");
					sb.append("<faultstring>"+e.getMessage()+"</faultstring>");
					sb.append("</env:Fault></env:Body></env:Envelope>");
					aHttpResponse.setContentType("text/xml");
				}
				else /*SOAP1.2*/
				{	sb.append("<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" ");
					sb.append("xmlns:xml=\"http://www.w3.org/XML/1998/namespace\">");
					sb.append("<env:Body>");
					sb.append("<env:Fault>");
					sb.append("<env:Code>");
					sb.append("<env:Value>env:Sender</env:Value>");
					sb.append("</env:Code>");
					sb.append("<env:Reason>");
					sb.append("<env:Text>"+e.getMessage()+"</env:Text>");
					sb.append("</env:Reason>");
					sb.append("</env:Fault>");
					sb.append("</env:Body>");
					sb.append("</env:Envelope>");
					aHttpResponse.setContentType("application/soap+xml");
				}
				aHttpResponse.setContentLength(sb.length());
				OutputStream os = aHttpResponse.getOutputStream();
				os.write(sb.toString().getBytes());
				os.flush();
			}
			catch(IOException ioe)
			{   ioe.printStackTrace();
			}
		}
		catch(SOAPModeServerException e)
		{	/*Issue SOAP Client Fault*/
			e.printStackTrace();
			issueSOAPServerFault(aHttpResponse, SOAP11, e.getMessage());
		}
		catch (Throwable e)
		{	e.printStackTrace();			
		}
		finally
		{	if(result!=null)
			{	processResult(aRequest,result,zone);
			}
			else
			{	issueVoidResponse(aRequest, 8);
			}
		}
	}
	
	/*Handle Output of SOAP message - test for SOAP Fault and set appropriate HTTP response code*/
	private void outputSOAP(URRequest aRequest, IURRepresentation aResponse, HttpRequest aHttpRequest, HttpResponse aHttpResponse )
	{   boolean SOAP11=aHttpRequest.getField("SOAPAction")!=null;
		String query=aHttpRequest.getQuery();
		boolean wsdlflag=(query!=null && aHttpRequest.getQuery().equals("WSDL"));
		boolean docflag=(query!=null && aHttpRequest.getQuery().startsWith("DOC"));
		if(SOAP11 || wsdlflag)
		{	aHttpResponse.setContentType("text/xml");
		}
		else if (docflag)
		{	aHttpResponse.setContentType("text/html");
		}
		else aHttpResponse.setContentType("application/soap+xml");
		try
		{	/*Test for SOAP Fault*/
			if(!wsdlflag && !docflag)
			{	IXAspect xa=(IXAspect)aResponse.getAspect(IXAspect.class);
				DOMXDA dx=new DOMXDA(xa.getReadOnlyDocument(),false);
				String envPrefix=null;
				if(SOAP11) envPrefix=dx.getPrefix("http://schemas.xmlsoap.org/soap/envelope/");
				else envPrefix=dx.getPrefix("http://www.w3.org/2003/05/soap-envelope");
				if(envPrefix!=null && dx.isTrue("/"+envPrefix+"Envelope/"+envPrefix+"Body/"+envPrefix+"Fault"))
				{   /*We have a SOAP Fault*/
					aHttpResponse.setStatus(500);
				}
			}
			/*Write Message*/
			OutputStream os = aHttpResponse.getOutputStream();
			if(aResponse.hasAspect(IAspectReadableBinaryStream.class))
			{   IAspectReadableBinaryStream bsaspect=(IAspectReadableBinaryStream)aResponse.getAspect(IAspectReadableBinaryStream.class);
				aHttpResponse.setContentLength(bsaspect.getContentLength());
				bsaspect.write(os);
			}
			else
			{   IAspectBinaryStream bsaspect=null;
				if(!aResponse.hasAspect(IAspectBinaryStream.class))
				{   bsaspect=(IAspectBinaryStream)transrepresent(aRequest.getURI(), aResponse, IAspectBinaryStream.class, aRequest).getAspect(IAspectBinaryStream.class);
				}
				else bsaspect=(IAspectBinaryStream)aResponse.getAspect(IAspectBinaryStream.class);
				bsaspect.write(os);
			}
			os.flush();
		}
		catch(Exception e)
		{   e.printStackTrace();
		}
	}
	
	/*Issue a Server SOAP Fault*/
	private void issueSOAPServerFault(HttpResponse aHttpResponse, boolean SOAP11, String message)
	{   try
		{   StringBuffer sb=new StringBuffer(1024);
			if(SOAP11) /*SOAP1.1 Fault*/
			{	sb.append("<env:Envelope   xmlns:env=\"http://schemas.xmlsoap.org/soap/envelope/\">  <env:Body> <env:Fault>");
				sb.append("<faultcode>env:Server</faultcode>");
				sb.append("<faultstring>NetKernelException</faultstring>");
				sb.append("<detail>"+message+"</detail>");
				sb.append("</env:Fault></env:Body></env:Envelope>");
				aHttpResponse.setContentType("text/xml");
			}
			else /*SOAP1.2*/
			{	sb.append("<env:Envelope xmlns:env=\"http://www.w3.org/2003/05/soap-envelope\" ");
				sb.append("xmlns:xml=\"http://www.w3.org/XML/1998/namespace\">");
				sb.append("<env:Body>");
				sb.append("<env:Fault>");
				sb.append("<env:Code>");
				sb.append("<env:Value>env:Receiver</env:Value>");
				sb.append("</env:Code>");
				sb.append("<env:Reason>");
				sb.append("<env:Text>"+message+"</env:Text>");
				sb.append("</env:Reason>");
				sb.append("</env:Fault>");
				sb.append("</env:Body>");
				sb.append("</env:Envelope>");
				aHttpResponse.setContentType("application/soap+xml");
			}
			aHttpResponse.setContentLength(sb.length());
			OutputStream os = aHttpResponse.getOutputStream();
			os.write(sb.toString().getBytes());
			os.flush();
		}
		catch(IOException ioe)
		{   ioe.printStackTrace();
		}
	}
	
	//Special Exceptions
	class ContentTooBigException extends Exception
	{}
	class ContentSizeRequiredException extends Exception
	{}
	class SOAPModeClientException extends Exception
	{   public SOAPModeClientException(String aMessage)
		{	super(aMessage);
		}
	}
	class SOAPModeServerException extends Exception
	{   public SOAPModeServerException(String aMessage)
		{	super(aMessage);
		}
	}
}
