/******************************************************************************
 * (c) Copyright 2002-2004, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: DynamicBridgeConfigAccessor.java,v $
 * Version:       $Name:  $ $Revision: 1.12 $
 * Last Modified: $Date: 2008/03/26 13:14:05 $
 *****************************************************************************/
package org.ten60.transport.http.bridge;

import com.ten60.netkernel.urii.IURAspect;
import org.ten60.netkernel.layer1.nkf.INKFConvenienceHelper;
import org.ten60.netkernel.layer1.nkf.INKFRequest;
import org.ten60.netkernel.layer1.nkf.INKFRequestReadOnly;
import org.ten60.netkernel.layer1.nkf.INKFResponse;
import org.ten60.netkernel.layer1.nkf.impl.NKFAccessorImpl;
import org.ten60.netkernel.xml.representation.DOMXDAAspect;
import org.ten60.netkernel.xml.representation.IXAspect;
import org.ten60.netkernel.xml.util.XMLUtils;
import org.ten60.netkernel.xml.xda.DOMXDA;
import org.ten60.netkernel.xml.xda.IXDAReadOnly;
import org.ten60.netkernel.xml.xda.IXDAReadOnlyIterator;
import org.ten60.netkernel.xml.xda.IXDAIterator;
import org.ten60.netkernel.layer1.representation.StringAspect;

import java.io.StringReader;
import java.util.Iterator;

/**
 * This accessor returns a configuration resource for the HTTP bridge service.
 *
 * The configuration is created from constinuent parts found in the resource
 * ffcpl:/etc/HTTPBridgeConfig.xml located in first-level imported children of
 * the module in which this accessor is called (the parent module, usually a fulcrum).
 *
 * This accessor requires the parameter "module" which is the URI identifier of
 * the parent module.
 *
 * Optionally, the version of the parent module is provided as the value of the
 * parameter "version".
 *
 * @author  pjr
 */
public class DynamicBridgeConfigAccessor extends NKFAccessorImpl
{
	
	/** Creates a new instance of DynamicBridgeConfigAccessor */
	public DynamicBridgeConfigAccessor()
	{ super(4, true , INKFRequestReadOnly.RQT_SOURCE);
	}
	
	public void processRequest(INKFConvenienceHelper context) throws Exception
	{ 
		String module=context.getThisRequest().getArgument("module");
		String version=null;
		if(context.getThisRequest().argumentExists("version"))
		{	version=context.getThisRequest().getArgument("version");
			version=version.substring(8);
		}

    //log that we are generating a configuration
		INKFRequest req=context.createSubRequest();
		req.setURI("active:log");
		IURAspect literal=new StringAspect("<http-bridge-config>Generating config for module:" + module+" version "+version+"</http-bridge-config>");
		req.addArgument("operand", literal);
		context.issueAsyncSubRequest(req);
		
		DOMXDA zone=null;
		IXAspect defaultConfigXA=null;

    DOMXDA config=null;
		try
		{	// Locate the default configuration and use this as the first "default" zone
      // if it exists and if the zone match is not ".*".
			defaultConfigXA=(IXAspect)context.sourceAspect("ffcpl:/etc/DefaultHTTPBridgeConfig.xml", IXAspect.class);

      config = new DOMXDA(defaultConfigXA.getReadOnlyDocument());

      // Look at all zones in the default configuration to make sure none have ".*" as their match
      IXDAReadOnlyIterator iter = config.readOnlyIterator("/HTTPBridgeConfig/zone");
      boolean keepConfig = true;
      while (keepConfig && iter.hasNext())
      { iter.next();
        String match = iter.getText("match",false);
        if (".*".equals(match))
        { keepConfig = false;
        }
      }
    
      if (!keepConfig)
        { config = null;
          INKFRequest req2=context.createSubRequest("active:log");
          IURAspect literal2=new org.ten60.netkernel.layer1.representation.StringAspect("<http-bridge-config>Warning - did not add any zone from \"ffcpl:/etc/DefaultHTTPBridgeConfig.xml\" because at least one has a match with \".*\". Please make the match more narrow.</http-bridge-config>");
          req2.addArgument("operand", literal2);
          req2.addArgument("operator", new org.ten60.netkernel.layer1.representation.StringAspect("<log><warning/></log>"));
          context.issueAsyncSubRequest(req2);
        }
    }
		catch(Exception e) {/*Ignore - there's no default in the super stack*/ }

    if(null == config)
		{ // Create a new empty configuration structure
      config=new DOMXDA(XMLUtils.getInstance().newDocument(),false);
		  config.appendPath("/", "HTTPBridgeConfig", null);
    }

    //Find all imported modules and and look for ffcpl:/etc/HTTPBridgeConfig.xml
		IXAspect modulesXA=(IXAspect)context.sourceAspect("netkernel:module", IXAspect.class);
		IXDAReadOnly mro=modulesXA.getXDA();
		StringBuffer xpath=new StringBuffer();
		xpath.append("/descendant::module[identity/uri = '"+module+"'");
		if(version!=null)
		{	xpath.append(" and identity/version='"+version+"' ");
		}
		xpath.append("]/dependencies/module");
		IXDAReadOnlyIterator it=mro.readOnlyIterator(xpath.toString());
		while(it.hasNext())
		{	it.next();
			try
			{			
				String m=it.getText("uri", false);
				String v=it.getText("version", false);

				INKFRequest sr=context.createSubRequest();
				sr.setURI("active:module");
				sr.addArgument("module", m);
				sr.addArgument("version", "version:"+v);
				sr.addArgument("resource", "ffcpl:/etc/HTTPBridgeConfig.xml");
				sr.setAspectClass(IXAspect.class);
				IXAspect bc=(IXAspect)context.issueSubRequest(sr).getAspect(IXAspect.class);
				IXDAReadOnlyIterator it2=bc.getXDA().readOnlyIterator("/HTTPBridgeConfig/zone");
				while(it2.hasNext())
				{	it2.next();
					String match=it2.getText("match", false);
					if(!match.equals(".*"))
					{	//Add zone provided it's not attempting to match the URI default pattern
						config.append(it2, ".", "/HTTPBridgeConfig");
						//log config message
						INKFRequest req2=context.createSubRequest("active:log");
						IURAspect literal2=new org.ten60.netkernel.layer1.representation.StringAspect("<http-bridge-config>HTTPBridgeConfig zone "+match+" found in "+m+" version "+v+" </http-bridge-config>");
						req2.addArgument("operand", literal2);
						context.issueAsyncSubRequest(req2);
					}
					else
					{	//log warning config message
						INKFRequest req2=context.createSubRequest("active:log");
						IURAspect literal2=new org.ten60.netkernel.layer1.representation.StringAspect("<http-bridge-config>Warning - did not add zone .* for " +m+" version "+v+" - .* is reserved, please narrow the match</http-bridge-config>");
						req2.addArgument("operand", literal2);
						req2.addArgument("operator", new org.ten60.netkernel.layer1.representation.StringAspect("<log><warning/></log>"));
						context.issueAsyncSubRequest(req2);
					}
				}
			}
			catch(Exception e)
			{ /*No valid config found in this module so ignore*/
			  //System.err.println("No HTTPBridgeConfig found for "+m+" version "+v);
			}
		}
				
		//Add the default zone
		String defaultZone="<zone><match>.*</match><processQueries/><passCookies/></zone>";
		zone=new DOMXDA(XMLUtils.getInstance().parse(new StringReader(defaultZone)));
		config.append(zone, "/", "/HTTPBridgeConfig");
		
		//Show final config
		//config.serialize(new OutputStreamWriter(System.err), true);

		DOMXDAAspect configDXA=new DOMXDAAspect(config);
		
		//Log the config - issue this asynch so that the log doesn't become a dependency
		req=context.createSubRequest();
		req.setURI("active:log");
		req.addArgument("operand", configDXA);
		context.issueAsyncSubRequest(req);
		 
		INKFResponse resp=context.createResponseFrom(configDXA);
		resp.setMimeType("text/xml");
		resp.setCacheable();
		
		context.setResponse(resp);
	  
	}	
}