﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/ConnectionInput.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class CreateConnectionRequest : public GlueRequest {
 public:
  AWS_GLUE_API CreateConnectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConnection"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the Data Catalog in which to create the connection. If none is
   * provided, the Amazon Web Services account ID is used by default.</p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  CreateConnectionRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A <code>ConnectionInput</code> object defining the connection to create.</p>
   */
  inline const ConnectionInput& GetConnectionInput() const { return m_connectionInput; }
  inline bool ConnectionInputHasBeenSet() const { return m_connectionInputHasBeenSet; }
  template <typename ConnectionInputT = ConnectionInput>
  void SetConnectionInput(ConnectionInputT&& value) {
    m_connectionInputHasBeenSet = true;
    m_connectionInput = std::forward<ConnectionInputT>(value);
  }
  template <typename ConnectionInputT = ConnectionInput>
  CreateConnectionRequest& WithConnectionInput(ConnectionInputT&& value) {
    SetConnectionInput(std::forward<ConnectionInputT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags you assign to the connection.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateConnectionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateConnectionRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;

  ConnectionInput m_connectionInput;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_catalogIdHasBeenSet = false;
  bool m_connectionInputHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
