/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.velocity.runtime.resource.loader;

import java.io.InputStream;
import java.io.BufferedInputStream;

import javax.sql.DataSource;

import org.apache.velocity.runtime.RuntimeSingleton;
import org.apache.velocity.runtime.resource.Resource;
import org.apache.velocity.runtime.resource.loader.ResourceLoader;
import org.apache.velocity.exception.ResourceNotFoundException;

import org.apache.commons.collections.ExtendedProperties;
import org.apache.commons.dbcp.ConnectionFactory;
import org.apache.commons.dbcp.DriverManagerConnectionFactory;
import org.apache.commons.dbcp.PoolableConnectionFactory;
import org.apache.commons.dbcp.PoolingDataSource;
import org.apache.commons.pool.ObjectPool;
import org.apache.commons.pool.impl.StackObjectPool;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.sql.SQLException;

public class BasicDataSourceResourceLoader extends ResourceLoader {
	
	private String driverClassName;
	
	private String url;
	
	private String username;
	
	private String password;
	
	private String tableName;

	private String keyColumn;

	private String templateColumn;

	private String timestampColumn;

	private DataSource dataSource;

	
	public void init( ExtendedProperties configuration) {
		driverClassName = configuration.getString("resource.driverClassName");
		url             = configuration.getString("resource.url");
		username        = configuration.getString("resource.username");
		password        = configuration.getString("resource.password");
		tableName       = configuration.getString("resource.table");
		keyColumn       = configuration.getString("resource.keycolumn");
		templateColumn  = configuration.getString("resource.templatecolumn");
		timestampColumn = configuration.getString("resource.timestampcolumn");

		RuntimeSingleton.info(
				"Resources Loaded From: " + url + "/" + tableName);
		RuntimeSingleton.info(
				"Resource Loader using columns: " + keyColumn + ", "
				+ templateColumn + " and " + timestampColumn);
		RuntimeSingleton.info("Resource Loader Initalized.");
	}

	
	public boolean isSourceModified(Resource resource) {
		return (resource.getLastModified() !=
				readLastModified(resource, "checking timestamp"));
	}

	
	public long getLastModified(Resource resource) {
		return readLastModified(resource, "getting timestamp");
	}

	
	/**
	 * Get an InputStream so that the Runtime can build a
	 * template with it.
	 *
	 *  @param name name of template
	 *  @return InputStream containing template
	 */
	public synchronized InputStream getResourceStream(String name)
		throws ResourceNotFoundException {
		
		if (name == null || name.length() == 0) {
			throw new ResourceNotFoundException (
					"Need to specify a template name!");
		}

		try {
			Connection conn = openDbConnection();

			try {
				ResultSet rs = readData(conn, templateColumn, name);

				try {
					if (rs.next()) {
						return new	BufferedInputStream(
								rs.getAsciiStream(templateColumn));
					}
					else {
						String msg = "DataSourceResourceLoader Error: "
							+ "cannot find resource " + name;
						RuntimeSingleton.error(msg);

						throw new ResourceNotFoundException(msg);
					}
				}
				finally {
					rs.close();
				}
			}
			finally {
				closeDbConnection(conn);
			}
		}
		catch(Exception e) {
			String msg =  "DataSourceResourceLoader Error: "
				+ "database problem trying to load resource "
				+ name + ": " + e.toString();

			RuntimeSingleton.error(msg);

			throw new ResourceNotFoundException(msg);
		}
	}

	/**
	 *  Fetches the last modification time of the resource
	 *
	 *  @param resource Resource object we are finding timestamp of
	 *  @param i_operation string for logging, indicating caller's intention
	 *
	 *  @return timestamp as long
	 */
	private long readLastModified(Resource resource, String i_operation) {
		String name = resource.getName();
		try {
			Connection conn = openDbConnection();

			try {
				ResultSet rs = readData(conn, timestampColumn, name);
				
				try {
					if (rs.next()) {
						return rs.getTimestamp(timestampColumn).getTime();
					}
					else {
						RuntimeSingleton.error("DataSourceResourceLoader Error: while "
									  + i_operation
									  + " could not find resource " + name);
					}
				}
				finally {
					rs.close();
				}
			}
			finally {
				closeDbConnection(conn);
			}
		}
		catch(Exception e) {
			RuntimeSingleton.error( "DataSourceResourceLoader Error: error while "
				+ i_operation + " when trying to load resource "
				+ name + ": " + e.toString() );
		}
		
		return 0;
	}

	
	/**
	 * @return
	 * @throws Exception
	 */
	private Connection openDbConnection()
		throws Exception {

		if(dataSource == null) {
			Class.forName(driverClassName);
			ObjectPool pool = new StackObjectPool();
			ConnectionFactory connFactory
				= new DriverManagerConnectionFactory(url, username, password);

			new PoolableConnectionFactory(
				connFactory, pool, null, null, false, true);

			dataSource = new PoolingDataSource(pool);
		}
		
		return dataSource.getConnection();
	}

	
	/**
	 * @param conn
	 */
	private void closeDbConnection(Connection conn) {
		try	{
			conn.close();
		}
		catch (Exception e) {
			RuntimeSingleton.info(
				"DataSourceResourceLoader Quirk: "
				+ "problem when closing connection: " + e.toString());
		}
	}

	
	/**
	 *  Reads the data from the datasource.  It simply does the following query :
	 *  <br>
	 *   SELECT <i>columnNames</i> FROM <i>tableName</i> WHERE <i>keyColumn</i>
	 *	  = '<i>templateName</i>'
	 *  <br>
	 *  where <i>keyColumn</i> is a class member set in init()
	 *
	 *  @param conn connection to datasource
	 *  @param columnNames columns to fetch from datasource
	 *  @param templateName name of template to fetch
	 *  @return result set from query
	 */
	private ResultSet readData(
			Connection conn,
			String columnNames,
			String templateName)
		throws SQLException {
		
		Statement stmt = conn.createStatement();

		String sql = 
			"SELECT " + columnNames
			+ " FROM " + tableName
			+ " WHERE " + keyColumn + " = '" + templateName + "'";

		return stmt.executeQuery(sql);
	}

}
