/***************************************************************************
**                                                                        **
**  Polyphone, a soundfont editor                                         **
**  Copyright (C) 2013-2020 Davy Triponney                                **
**                                                                        **
**  This program is free software: you can redistribute it and/or modify  **
**  it under the terms of the GNU General Public License as published by  **
**  the Free Software Foundation, either version 3 of the License, or     **
**  (at your option) any later version.                                   **
**                                                                        **
**  This program is distributed in the hope that it will be useful,       **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of        **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          **
**  GNU General Public License for more details.                          **
**                                                                        **
**  You should have received a copy of the GNU General Public License     **
**  along with this program. If not, see http://www.gnu.org/licenses/.    **
**                                                                        **
****************************************************************************
**           Author: Davy Triponney                                       **
**  Website/Contact: https://www.polyphone-soundfonts.com                 **
**             Date: 01.01.2013                                           **
***************************************************************************/

#include "graphiquefourier.h"
#include "sound.h"
#include "sampleutils.h"
#include "soundfontmanager.h"
#include "contextmanager.h"
#include "frequency_peaks/toolfrequencypeaks.h"
#include "utils.h"
#include <QMenu>
#include <QFileDialog>
#include <QPainter>

GraphiqueFourier::GraphiqueFourier(QWidget * parent) : QCustomPlot(parent),
    _fixedTickerX(new QCPAxisTickerFixed()),
    _fixedTickerY(new QCPAxisTickerFixed()),
    _menu(nullptr),
    _toolFrequencyPeak(new ToolFrequencyPeaks())
{
    // Configuration du graphe
    this->addGraph();
    QPen graphPen;
    QColor color = ContextManager::theme()->getColor(ThemeManager::WINDOW_TEXT);
    color.setAlpha(150);
    graphPen.setColor(color);
    graphPen.setWidthF(1);
    this->graph(0)->setPen(graphPen);

    // Axes
    QPen penGrid = this->xAxis->grid()->pen();
    QPen penSubGrid = this->xAxis->grid()->subGridPen();
    penGrid.setColor(QColor(140, 140, 140));
    penSubGrid.setColor(QColor(220, 220, 220));

    this->xAxis->setRange(0, 20000);
    this->xAxis->setVisible(false);
    this->xAxis->setTicks(false);
    this->xAxis->setLabel(tr("Frequency (Hz)"));
    this->xAxis->grid()->setSubGridVisible(true);
    this->xAxis->grid()->setPen(penGrid);
    this->xAxis->grid()->setSubGridPen(penSubGrid);
    this->xAxis->setTickLength(0);
    this->xAxis->setSubTickLength(0);
    this->xAxis->setBasePen(penGrid);
    _fixedTickerX->setScaleStrategy(QCPAxisTickerFixed::ssNone);
    _fixedTickerX->setTickStep(2000);
    this->xAxis->setTicker(_fixedTickerX);

    this->yAxis->setRange(0, 1.05);
    this->yAxis->setVisible(false);
    this->yAxis->setTicks(false);
    this->yAxis->setLabel(tr("Intensity"));
    this->yAxis->grid()->setSubGridVisible(true);
    this->yAxis->grid()->setPen(penGrid);
    this->yAxis->grid()->setSubGridPen(penSubGrid);
    this->yAxis->setTickLength(0);
    this->yAxis->setSubTickLength(0);
    this->yAxis->setBasePen(penGrid);
    _fixedTickerY->setScaleStrategy(QCPAxisTickerFixed::ssNone);
    _fixedTickerY->setTickStep(0.2);
    this->yAxis->setTicker(_fixedTickerY);

    // Marges
    this->axisRect()->setAutoMargins(QCP::msNone);
    this->axisRect()->setMargins(QMargins(0, 0, 0, 0));

    // Préparation du menu contextuel
    _menu = new QMenu(this);
    _menu->setStyleSheet(ContextManager::theme()->getMenuTheme());
    QAction * action1 = _menu->addAction(tr("Export graph") + "...");
    connect(action1, SIGNAL(triggered()), this, SLOT(exportPng()));
    QAction * action2 = _menu->addAction(tr("Show peak frequencies") + "...");
    connect(action2, SIGNAL(triggered()), this, SLOT(exportPeaks()));

    this->plotLayout()->insertRow(0);
    this->plotLayout()->setRowStretchFactors(QList<double>() << 1 << 100);
    this->plotLayout()->setRowSpacing(0);
}

GraphiqueFourier::~GraphiqueFourier()
{
    _fixedTickerX.clear();
    _fixedTickerY.clear();
    delete _toolFrequencyPeak;
}

void GraphiqueFourier::setBackgroundColor(QColor color)
{
    // Modification de la couleur de fond
    this->setBackground(color);
}

void GraphiqueFourier::setCurrentIds(IdList ids)
{
    _currentIds = ids;

    if (_currentIds.count() == 1)
    {
        // Sample rate
        this->_dwSmplRate = SoundfontManager::getInstance()->get(_currentIds[0], champ_dwSampleRate).dwValue;

        // Load data
        QByteArray baData = SoundfontManager::getInstance()->getData(_currentIds[0], champ_sampleData16);
        int length = baData.size() / 2;
        _fData.resize(length);
        qint16 * data = reinterpret_cast<qint16*>(baData.data());
        for (int i = 0; i < length; i++)
            _fData[i] = static_cast<float>(data[i]);

        // Menu
        _menu->actions()[0]->setEnabled(true);
    }
    else
    {
        this->_dwSmplRate = -1;
        _fData.clear();
        _peaks.clear();
        graph(0)->data()->clear();
        this->replot();

        // Menu
        _menu->actions()[0]->setEnabled(false);
    }
}

void GraphiqueFourier::setPos(quint32 posStart, quint32 posEnd, bool withReplot)
{
    if (_currentIds.count() != 1)
        return;

    _peaks.clear();
    _key = -1;
    _delta = 0;

    if (_fData.isEmpty())
        graph(0)->data()->clear();
    else
    {
        QVector<float> vectFourier;
        int posMaxFourier;
        _peaks = computePeaks(_fData, _dwSmplRate, posStart, posEnd, vectFourier, posMaxFourier, &_key, &_delta);

        // Display the Fourier transform?
        if (posMaxFourier == -1)
            graph(0)->data()->clear();
        else
            this->dispFourier(vectFourier, posMaxFourier);
    }

    if (withReplot)
        this->replot();
}

QList<Peak> GraphiqueFourier::computePeaks(QVector<float> fData, quint32 dwSmplRate,
                                           quint32 posStart, quint32 posEnd,
                                           QVector<float> &vectFourier, int &posMaxFourier,
                                           int *key, int *correction)
{
    // If there is no loop, determine a stable portion for the Fourier transforme and the autocorrelation
    if (posStart == posEnd)
        SampleUtils::regimePermanent(fData, dwSmplRate, posStart, posEnd);

    // Limit or increase the portion to 0.5 second
    if (posEnd > dwSmplRate / 2 + posStart)
    {
        quint32 offset = (posEnd - dwSmplRate / 2 - posStart) / 2;
        posStart += offset;
        posEnd -= offset;
    }
    else if (posEnd < dwSmplRate / 2 + posStart)
    {
        quint32 offset = (posStart + dwSmplRate / 2 - posEnd) / 2;
        if (posStart < offset)
            posStart = 0;
        else
            posStart -= offset;
        posEnd += offset;
        if (posEnd > static_cast<quint32>(fData.size() - 1))
            posEnd = fData.size() - 1;
    }

    // Extraction des données du régime permanent
    QVector<float> baData2 = fData.mid(static_cast<int>(posStart), static_cast<int>(posEnd - posStart));

    // Corrélation du signal de 20 à 20000Hz
    quint32 dMin;
    QVector<float> vectCorrel = SampleUtils::correlation(baData2.constData(),
                                                         qMin(static_cast<quint32>(baData2.size()), 4000u),
                                                         dwSmplRate, 20, 20000, dMin);

    // Transformée de Fourier du signal
    vectFourier = SampleUtils::getFourierTransform(baData2);
    quint32 size = static_cast<quint32>(vectFourier.size()) * 2;

    // Recherche des corrélations minimales (= plus grandes similitudes) et intensités fréquencielles maximales
    QList<quint32> posMinCor = SampleUtils::findMins(vectCorrel, 20, 0.7f);
    QList<quint32> posMaxFFT = SampleUtils::findMax(vectFourier, 50, 0.05f);
    if (posMaxFFT.isEmpty())
    {
        posMaxFourier = -1;
        return QList<Peak>();
    }
    else
        posMaxFourier = posMaxFFT[0];

    if (posMinCor.isEmpty())
        return QList<Peak>();

    // Pour chaque minimum de corrélation (considéré comme la base), on cherche un pic de fréquence
    float freq = 0;
    float score = -1;
    for (int i = 0; i < posMinCor.size(); i++)
    {
        // Portion de Fourier contenant la corrélation
        quint32 iMin, iMax;
        int coefFourier = 1;
        iMin = (size - 1) / (posMinCor[i] + dMin + 1) - 1;
        if (iMin < 1) iMin = 1;
        if (posMinCor[0] + dMin - 1 <= 0)
            iMax = 0;
        else
            iMax = (size - 1) / (posMinCor[i] + dMin - 1) + 1;
        if (iMax > size / 2) iMax = size / 2;

        // Un pic s'y trouve-t-il ?
        bool rep = false;
        int numeroPic = 0;
        while (rep == false && numeroPic < posMaxFFT.size())
        {
            rep = iMin < posMaxFFT[numeroPic] && posMaxFFT[numeroPic] < iMax;
            if (!rep) numeroPic++;
        }

        if (!rep)
        {
            // Recherche d'un pic à l'octave du dessus
            coefFourier = 2;
            iMin *= 2;
            iMax *= 2;
            if (iMin >= size / 2) iMin = size / 2;
            if (iMax >= size / 2) iMax = size / 2;

            // Un pic s'y trouve-t-il ?
            bool rep = false;
            numeroPic = 0;
            while (rep == false && numeroPic < posMaxFFT.size())
            {
                rep = iMin < posMaxFFT[numeroPic] && posMaxFFT[numeroPic] < iMax;
                if (!rep) numeroPic++;
            }
        }

        if (rep)
        {
            // Fréquence et score corrélation
            float freqCorrel = (static_cast<float>(size - 1) / (posMinCor[i] + dMin) * dwSmplRate) / (size - 1);
            float scoreCorrel;
            if (vectCorrel[static_cast<int>(posMinCor[i])] <= 0)
                scoreCorrel = 1;
            else
                scoreCorrel = vectCorrel[static_cast<int>(posMinCor[0])] / vectCorrel[static_cast<int>(posMinCor[i])];

            // Fréquence et score Fourier
            float freqFourier = (static_cast<float>(posMaxFFT[numeroPic]) * dwSmplRate) / (size - 1) / coefFourier;
            float scoreFourier = vectFourier[static_cast<int>(posMaxFFT[numeroPic])] /
                    vectFourier[static_cast<int>(posMaxFFT[0])];

            // Score global
            float scoreGlobal = scoreCorrel * scoreFourier;
            if (scoreGlobal > score)
            {
                score = scoreGlobal;

                // Ajustement des scores en fonction de la hauteur de note
                float noteTmp = 12.0f * static_cast<float>(qLn(static_cast<double>(freqCorrel))) / 0.69314718056f - 36.3763f;
                if (noteTmp < 40 || posEnd - posStart < 4096)
                    scoreFourier = 0;
                else if (noteTmp < 90)
                {
                    noteTmp = (noteTmp - 40) / 50;
                    scoreFourier *= noteTmp;
                    scoreCorrel *=  1.f - noteTmp;
                }
                else
                    scoreCorrel = 0;

                // Détermination de la fréquence globale et enregistrement
                freq = (scoreCorrel * freqCorrel + scoreFourier * freqFourier) / (scoreCorrel + scoreFourier);
            }
        }
    }

    // Si aucune note n'a été trouvée, on prend la note la plus probable d'après Fourier
    if (score < 0)
        freq = static_cast<float>(posMaxFFT[0]) * dwSmplRate / (size - 1);

    // Numéro de la note correspondant à cette fréquence
    double note3 = 12 * qLn(static_cast<double>(freq)) / 0.69314718056 - 36.3763;

    // Note la plus proche
    int note = qRound(note3);

    // Affichage
    QList<Peak> result;
    if (note >= 0 && note <= 128)
    {
        // Closest key with the associated correction
        if (key != nullptr)
            *key = note;
        if (correction != nullptr)
            *correction = Utils::round32((note3 - static_cast<double>(note)) * 100.);

        // Peaks
        for (int i = 0; i < posMaxFFT.size(); i++)
        {
            // intensité
            double factor = static_cast<double>(vectFourier[static_cast<int>(posMaxFFT[i])] /
                            vectFourier[static_cast<int>(posMaxFFT[0])]);

            // fréquence
            double freq = static_cast<double>(posMaxFFT[i] * dwSmplRate) / (size - 1);

            // note la plus proche
            double note = 12. * qLn(freq) / 0.69314718056 - 36.3763;
            if (note < 0)
                note = 0;
            else if (note > 128)
                note = 128;
            int note2 = Utils::round32(note);
            int delta = Utils::round32((static_cast<double>(note2) - note) * 100.);

            // Prepare text
            result << Peak(factor, freq, note2, delta);
        }
    }

    return result;
}

void GraphiqueFourier::dispFourier(QVector<float> vectFourier, float posMaxFourier)
{
    this->graph(0)->data()->clear();
    qint32 size_x = (vectFourier.size() * 40000) / static_cast<signed>(this->_dwSmplRate);
    QVector<double> x(size_x), y(size_x);
    for (qint32 i = 0; i < size_x; i++)
    {
        x[i] = (20000. * i) / (size_x - 1); // Conversion Hz
        if (i < vectFourier.size())
            y[i] = static_cast<double>(vectFourier[i]) /
                   static_cast<double>(vectFourier[static_cast<int>(posMaxFourier)]); // normalisation entre 0 et 1
        else
            y[i] = 0;
    }
    this->graph(0)->setData(x, y, true);
}

void GraphiqueFourier::mousePressEvent(QMouseEvent *event)
{
    if (event->button() == Qt::RightButton)
    {
        // Affichage menu contextuel
        _menu->exec(QCursor::pos());
    }
}

void GraphiqueFourier::paintEvent(QPaintEvent * event)
{
    // Draw the graph
    QCustomPlot::paintEvent(event);
    if (_peaks.empty())
        return;

    // Helpful elements
    QSize size = this->size();
    int marginRight = 1;
    int marginTop = 1;
    int tuneWidth = qMin(160, size.width() - marginRight);
    int tuneCellPadding = 5;
    int tickHalfHeight = 3;

    QColor highlightColor = ContextManager::theme()->getColor(ThemeManager::HIGHLIGHTED_BACKGROUND);
    QColor highlightTextColor = ContextManager::theme()->getColor(ThemeManager::HIGHLIGHTED_TEXT);
    QColor backgroundColor = ContextManager::theme()->getColor(ThemeManager::WINDOW_BACKGROUND);
    QColor textColor = ContextManager::theme()->getColor(ThemeManager::WINDOW_TEXT);

    QFont fontInfo = QFont(font().family(), 7);
    QFont fontInfoSmall = QFont(font().family(), 6);
    QFont fontMain = fontInfo;
    fontMain.setBold(true);
    QFontMetrics fm(fontMain);
    int fontHeight = fm.height();

    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing);

    // Text to display
    QString keyName = ContextManager::keyName()->getKeyName(static_cast<unsigned int>(_key));
    if (ContextManager::keyName()->getNameMiddleC() == KeyNameManager::MIDDLE_C_60)
        keyName += " (" + ContextManager::keyName()->getKeyName(static_cast<unsigned int>(_key), true, false, false, true) + ")";
    int textWidth = fm.horizontalAdvance(keyName);

    // TUNING

    // Left and right scales, delta
    textColor.setAlpha(100);
    painter.setPen(textColor);
    painter.setFont(fontMain);
    {
        int tuneCellMargin = fontHeight * 3 / 2;
        int x1 = size.width() - marginRight - tuneWidth + fontHeight / 2 + tuneCellPadding;
        int x2 = size.width() - marginRight - tuneWidth / 2 - textWidth / 2 - tuneCellMargin - tuneCellPadding;
        int x3 = size.width() - marginRight - tuneWidth / 2 + textWidth / 2 + tuneCellMargin + tuneCellPadding;
        int x4 = size.width() - marginRight - fontHeight / 2 - tuneCellPadding;

        // Horizontal line
        painter.drawLine(x1, marginTop + tuneCellPadding + fontHeight / 2, x2,
                         marginTop + tuneCellPadding + fontHeight / 2);
        painter.drawLine(x3, marginTop + tuneCellPadding + fontHeight / 2, x4,
                         marginTop + tuneCellPadding + fontHeight / 2);

        // Ticks
        for (int i = 0; i < 6; i++)
        {
            int x = x1 + (x2 - x1) * i / 5;
            painter.drawLine(x, marginTop + tuneCellPadding + fontHeight / 2 - tickHalfHeight,
                             x, marginTop + tuneCellPadding + fontHeight / 2 + tickHalfHeight);
            x = x3 + (x4 - x3) * i / 5;
            painter.drawLine(x, marginTop + tuneCellPadding + fontHeight / 2 - tickHalfHeight,
                             x, marginTop + tuneCellPadding + fontHeight / 2 + tickHalfHeight);
        }

        // Delta
        if (_delta != 0)
        {
            // Text position
            int pos = _delta > 0 ?
                        (50 * x3 + (x4 - x3) * _delta) / 50 :
                        (50 * x2 + (x2 - x1) * _delta) / 50;
            QString txt = QString::number(_delta < 0 ? -_delta : _delta);
            int textWidth = fm.horizontalAdvance(txt);

            painter.setBrush(highlightColor);
            painter.setPen(highlightColor);
            painter.drawEllipse(QPoint(pos, marginTop + tuneCellPadding + fontHeight / 2),
                                fontHeight / 2 + tuneCellPadding, fontHeight / 2 + tuneCellPadding);
            painter.fillRect(pos, marginTop + tuneCellPadding + fontHeight / 2 - tickHalfHeight,
                             size.width() - marginRight - tuneWidth / 2 - pos, 2 * tickHalfHeight,
                             highlightColor);
            painter.setPen(highlightTextColor);
            painter.drawText(QRect(pos - textWidth / 2, marginTop + tuneCellPadding, textWidth, fontHeight),
                             Qt::AlignCenter, txt);
        }
    }

    // Write key name
    {
        QPainterPath path;
        textColor.setAlpha(255);
        path.addRoundedRect(QRect(size.width() - marginRight - tuneWidth / 2 - textWidth / 2 - tuneCellPadding,
                                   marginTop, textWidth + 2 * tuneCellPadding, fontHeight + 2 * tuneCellPadding), 3, 3);
        painter.fillPath(path, _delta == 0 ? highlightColor : textColor);
        painter.setPen(_delta == 0 ? highlightTextColor : backgroundColor);
        painter.drawText(QRect(size.width() - marginRight - tuneWidth / 2 - textWidth / 2 - tuneCellPadding, marginTop + tuneCellPadding,
                               textWidth + 2 * tuneCellPadding, fontHeight), Qt::AlignCenter, keyName);
    }

    // MAIN FREQUENCY PEAKS

    textColor.setAlpha(100);
    painter.setPen(textColor);
    {
        // Compute the text
        QList<QStringList> lines;
        int peakNumber = 0;
        int posY = 2 * tuneCellPadding + fontHeight * 3 / 2;
        while (posY + fontHeight < size.height() && peakNumber < _peaks.count())
        {
            QStringList line;
            line << QLocale::system().toString(_peaks[peakNumber]._factor, 'f', 2)
                 << QLocale::system().toString(_peaks[peakNumber]._frequency, 'f', 2) + " " + tr("Hz", "unit for Herz")
                 << ContextManager::keyName()->getKeyName(static_cast<unsigned int>(_peaks[peakNumber]._key))
                 << (_peaks[peakNumber]._correction > 0 ? "+" : "") + QString::number(_peaks[peakNumber]._correction);
            lines << line;

            peakNumber++;
            posY += fontHeight;
        }
        QString lastTextElement = "/ 100";

        // Compute the column widths
        int columnWidth[5];
        QFontMetrics fm(fontInfo);
        for (int col = 0; col < 4; col++)
        {
            columnWidth[col] = 0;
            for (peakNumber = 0; peakNumber < lines.count(); peakNumber++)
            {
                int w = fm.horizontalAdvance(lines[peakNumber][col]);
                if (w > columnWidth[col])
                    columnWidth[col] = w;
            }
        }
        columnWidth[4] = QFontMetrics(fontInfoSmall).horizontalAdvance(lastTextElement);

        // Margins
        columnWidth[1] += 10;
        columnWidth[2] += 10;
        columnWidth[3] += 10;
        columnWidth[4] += 2;

        // Write the text
        posY = 2 * tuneCellPadding + fontHeight * 3 / 2;
        for (peakNumber = 0; peakNumber < lines.count(); peakNumber++)
        {
            painter.setFont(fontInfo);
            painter.drawText(QRect(0, posY, size.width() - columnWidth[1] - columnWidth[2] - columnWidth[3] - columnWidth[4] - marginRight, fontHeight),
                             Qt::AlignRight, lines[peakNumber][0]);
            painter.drawText(QRect(0, posY, size.width() - columnWidth[2] - columnWidth[3] - columnWidth[4] - marginRight, fontHeight),
                             Qt::AlignRight, lines[peakNumber][1]);
            painter.drawText(QRect(0, posY, size.width() - columnWidth[3] - columnWidth[4] - marginRight, fontHeight),
                             Qt::AlignRight, lines[peakNumber][2]);
            painter.drawText(QRect(0, posY, size.width() - columnWidth[4] - marginRight, fontHeight),
                             Qt::AlignRight, lines[peakNumber][3]);
            painter.setFont(fontInfoSmall);
            painter.drawText(QRect(0, posY, size.width() - marginRight, fontHeight),
                             Qt::AlignRight, lastTextElement);
            posY += fontHeight;
        }
    }
}

void GraphiqueFourier::exportPng()
{
    QString defaultFile = ContextManager::recentFile()->getLastDirectory(RecentFileManager::FILE_TYPE_FREQUENCIES) + "/" +
            _name.replace(QRegularExpression(QString::fromUtf8("[`~*|:<>«»?/{}\"\\\\]")), "_") + ".png";
    QString fileName = QFileDialog::getSaveFileName(this, tr("Export a graph"),
                                                    defaultFile, tr("Png file") + " (*.png)");
    if (!fileName.isEmpty())
    {
        ContextManager::recentFile()->addRecentFile(RecentFileManager::FILE_TYPE_FREQUENCIES, fileName);
        exportPng(fileName);
    }
}

void GraphiqueFourier::exportPng(QString fileName)
{
    // Préparation
    this->yAxis->setRange(0, 1);
    QColor backgroundColor = this->mBackgroundBrush.color();
    this->setBackgroundColor(Qt::white);
    this->xAxis->setVisible(true);
    this->xAxis->setTicks(true);
    this->yAxis->setVisible(true);
    this->yAxis->setTicks(true);
    this->axisRect()->setAutoMargins(QCP::msAll);
    QCPTextElement * title = new QCPTextElement(this, _name);
    this->plotLayout()->addElement(0, 0, title);
    QPen graphPen;
    graphPen.setColor(QColor(240, 0, 0));
    graphPen.setWidthF(1);
    this->graph(0)->setPen(graphPen);

    // Redimensionnement
    QSize minSize = this->minimumSize();
    this->setMinimumSize(1200, 400);

    // Création fichier png
    QFile file(fileName);
    if (file.open(QIODevice::WriteOnly))
    {
        QPixmap pix = this->grab();
        pix.save(&file, "PNG");
        file.close();
    }

    // Restauration du style du départ
    this->yAxis->setRange(0, 1.05);
    this->setBackgroundColor(backgroundColor);
    this->xAxis->setVisible(false);
    this->xAxis->setTicks(false);
    this->yAxis->setVisible(false);
    this->yAxis->setTicks(false);
    this->axisRect()->setAutoMargins(QCP::msNone);
    this->axisRect()->setMargins(QMargins(0, 0, 0, 0));
    this->plotLayout()->remove(title);
    graphPen.setColor(ContextManager::theme()->getColor(ThemeManager::WINDOW_TEXT));
    graphPen.setWidthF(1);
    this->graph(0)->setPen(graphPen);

    // Redimensionnement
    this->setMinimumSize(minSize);
}

void GraphiqueFourier::exportPeaks()
{
    _toolFrequencyPeak->setIds(_currentIds);
    _toolFrequencyPeak->run();
}

void GraphiqueFourier::getEstimation(int &pitch, int &correction)
{
    pitch = _key;
    correction = -_delta;
}
