

--
-- Copyright (C) 2021  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--



--
--*****************************************************************
--
-- Description:
--
-- This package implements an extremely efficient self-adjusting
-- binary search tree called a splay tree with very little overhead
-- to maintain the balance.  The ordering by IdType is maintained
-- to permit fast access by Id and fast checks for duplicates.
-- Linear access and traversal of the tree elements according to
-- insertion order [fifo queue] is also supported using
-- the "head" and "next" accessors.
--
-- Reference:
-- See the Journal of the A.C.M., July 1985,
-- Vol. 32, No. 3, pg. 652, library call # QA 76 A77
--
--*******************************************************************

--with splaytypes;


generic

   type IdType is private;
   type DataType is private;

   with function "<" ( k1, k2: in IdType ) return boolean is <>;
   with function ">" ( k1, k2: in IdType ) return boolean is <>;

   -- These infix functions (operators) together with the type IdType
   -- are used to define the search tree ordering.  

   -- the following example instantiation is particularly simple
   -- because the idtype (integer) has an intrinsic ordering:
   --
   -- package mylist is new splaylist( idtype   => integer,
   --                             datatype => float,
   --                             "<"      =>  "<",
   --                             ">"      =>  ">" );
   --
	-- If the idtype has an Ada-implicit ordering,
   -- then the instantiations require only the addition
   -- of { "<", ">" } as the last two generic parameters.

	-- This legacy version of splaylist keeps a simple list 
	-- structure sorted by priority for legacy BFS
	-- sokoban solvers.

package splaylist is -- clients: ibox3r, iplr3r


  type StatusType is
       (Ok, Found, Empty, NotFound, Error, DupId, NilPtr);




   type ListType is private;


   --===================================================
   -- temporary routine for debugging purposes only;
   -- allows users to deduce the tree's structure
   --procedure GetParentKey( k: IdType;
   --                        list: in out listtype;
   --                        kp: out IdType );
   --===================================================



   --------------------------------------
   -- Adds a node;
   -- repositions iterator to new node
   --------------------------------------
   procedure AddNode( Id     : in IdType;
                      Data   : in DataType;
                      --dupData   : out DataType;
                      List   : in out ListType;
                      Status :    out StatusType);






   ------------------------------------------------------------
   -- returns the nodal data at the iterator's current position
   ------------------------------------------------------------
   procedure data( List   : in ListType;
                   Id     : out IdType;
                   Data   : out DataType;
                   Status :    out StatusType);



   ---------------------------------------
   -- deletes the node with specified Id
   -- repositions iterator to predecessor
   -- ( successor when deleting the head )
   ---------------------------------------
   procedure DelNode( Id     : in IdType;
                      List   : in out ListType;
                      Status :    out StatusType);




   ------------------------------------------------
   -- gets the nodal data belonging to specified Id
   ------------------------------------------------
   procedure search( Id     : in     IdType;
                      List   : in     ListType;
                      Data   :    out DataType;
                      Status :    out StatusType);




   ------------------------------------------
   -- returns the number of nodes in the tree
   ------------------------------------------
   function length( List : in ListType ) return integer;




   ------------------------------------------
   -- modifies the nodal data at specified Id
   ------------------------------------------
   procedure ModifyNode( Id     : in     IdType;
                         Data   : in     DataType;
                         List   : in out ListType;
                         Status :    out StatusType);



   ------------------------------------------
   -- moves current node pointer to next node
   ------------------------------------------
   procedure next ( List   : in out ListType;
                    Status :    out StatusType);

   ----------------------------------------------
   -- moves current node pointer to previous node
   ----------------------------------------------
   procedure prev ( List   : in out ListType;
                    Status :    out StatusType);



   --------------------------------------------
   -- sets the iterator to the tail of the list
   --------------------------------------------
   procedure tail ( List   : in out ListType;
                    Status :    out StatusType);


	--------------------------------------------
	-- sets the iterator to the head of the list
	--------------------------------------------
	procedure head ( List : in out ListType;
                   Status :    out StatusType);

	-- acts like head if first call
   procedure nextOrHead ( List   : in out ListType;
                    Status :    out StatusType);


	procedure make_empty ( List : in out ListType;
                   Status :    out StatusType);


	function getsize return integer;



private

   type splayrec;
   type splayptr is access splayrec;
   type splayrec is
      record
         Id : IdType;
         Data: DataType;
         parent,
         left_child,
         right_child,
         next, prev   : splayptr := null;
      end record; --376 bytes

   type listheader is
      record
         root, head, tail : splayptr := null;
         size : integer := 0;
      end record;

   type listptr is access listheader;

   type listtype is
      record
         header : listptr := null;
         curr   : splayptr := null;
      end record;

	--for splayptr'storage_pool use splaytypes.bounded_pool;


end splaylist;
