/*
 *    Qizx/open 4.1
 *
 * This code is the open-source version of Qizx.
 * Copyright (C) 2004-2009 Axyana Software -- All rights reserved.
 *
 * The contents of this file are subject to the Mozilla Public License 
 *  Version 1.1 (the "License"); you may not use this file except in 
 *  compliance with the License. You may obtain a copy of the License at
 *  http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 *  for the specific language governing rights and limitations under the
 *  License.
 *
 * The Initial Developer of the Original Code is Xavier Franc - Axyana Software.
 *
 */
package com.qizx.api;

/**
 * Streaming output for XML contents, working in 'push' mode.
 * <p>
 * This interface is mainly implemented by XMLSerializer, but it can also be
 * used for any conversion of the XML data model to another representation: in
 * particular there is an adapter to SAX2 ({@link
 * com.qizx.api.util.PushStreamToSAX}) and an adapter to W3C DOM ({@link
 * com.qizx.api.util.PushStreamToDOM}).
 * <p>
 * Note: this interface is similar in essence to
 * javax.xml.stream.XMLStreamWriter. It is however simpler and supports the
 * copy-namespaces feature of XQuery.
 * <p>
 * In this representation, a well-formed fragment of XML contents is considered
 * as a stream of abstract "events" reflecting the physical structure:
 * <ul>
 * <li>Beginning of a document: putDocumentStart
 * <li>DTD information: putDTD, optional
 * <li>Beginning of an Element: putElementStart. Corresponds to the start-tag,
 * can be followed by putAttribute and putNamespace. The contents of the
 * element is then defined by following events putElementStart (nested
 * elements), putText, putComment, putProcessingInstruction. An element is
 * closed by putElementEnd.
 * <li>Attribute: putAttribute defines an attribute on the current element.
 * Must come after a putElementStart, and before any content event.
 * <li>Namespace: putNamespace associates a namespace prefix with a namespace
 * URI. Similar to the xmlns pseudo-attributes. The prefix/namespace-uri
 * association defined remains valid until the end of current element (but can
 * be overloaded by another prefix defined on an enclosed element). Must come
 * after a putElementStart and before any content event.
 * <li>Text fragment: putText. Consecutive putText are coalesced into a single
 * text node. A putText with length 0 creates no text node.
 * <li>Comment: putComment creates a comment node.
 * <li>Processing instruction: putProcessingInstruction.
 * <li>End of element: putElementEnd. Should match the opening
 * putElementStart.
 * <li>End of document: putDocumentEnd.
 * </ul>
 * <p>
 * Example: the following XML document
 * 
 * <pre>
 * &lt;x:doc id='1' xmlns:x=&quot;some.namespace.uri&quot;&gt;some text&lt;empty value='a'/&gt;
 *  comment:&lt;!-- commentary --&gt;, a PI: &lt;?pi some stuff?&gt;&lt;/x:doc&gt;
 * </pre>
 * 
 * <p>
 * can be generated by this stream of "events": <table border="1">
 * <tr>
 * <th bgcolor="#E0E0E0">Event method
 * <th bgcolor="#E0E0E0">arguments
 * <tr>
 * <td>putDocumentStart
 * <td>-
 * <tr>
 * <td>putElementStart
 * <td>QName {some.namespace.uri}doc (prefix irrelevant)
 * <tr>
 * <td>putAttribute
 * <td>QName id, value '1', type null
 * <tr>
 * <td>putNamespace
 * <td>prefix 'x', URI "some.namespace.uri"
 * <tr>
 * <td>putText
 * <td>String 'some text'
 * <tr>
 * <td>putElementStart
 * <td>QName empty
 * <tr>
 * <td>putAttribute
 * <td>QName value, value 'a', type null
 * <tr>
 * <td>putElementEnd
 * <td>QName empty
 * <tr>
 * <td>putText
 * <td>String 'a comment:'
 * <tr>
 * <td>putComment
 * <td>String 'commentary'
 * <tr>
 * <td>putText
 * <td>String ', a PI: '
 * <tr>
 * <td>putProcessingInstruction
 * <td>target 'pi', contents 'some stuff'
 * <tr>
 * <td>putElementEnd
 * <td>QName x:doc
 * <tr>
 * <td>putDocumentEnd
 * <td>- </table>
 * <p>
 * Note: this interface has all facilities needed to handle Namespaces
 * completely (in particular missing prefixes can be generated automatically).
 * However prefixes defined on QName's passed as arguments are ignored: one has
 * to call putNamespace explicitly.
 * <p id='cex'><b>Exceptions thrown by methods:</b> depends on the actual
 * stream implementation. Should be thrown for an error preventing proper
 * completion of the node streaming.
 */
public interface XMLPushStream
{
    /**
     * Puts a Document Start event. This method should be called first if the
     * result is meant to be a well-formed document. However it it possible to
     * omit it (and putDocumentEnd) if the desired result is an element.
     * @return false if the current stream position is already inside a node
     * (this event is then ignored)
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    boolean putDocumentStart()
        throws DataModelException;

    /**
     * Puts a Document End event.
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putDocumentEnd()
        throws DataModelException;

    /**
     * Optionally puts a DTD event.
     * @param name name of the DTD. Optional, can be null
     * @param publicId public-id of the DTD. Optional, can be null
     * @param systemId system-id of the DTD. Optional, can be null
     * @param internalSubset source form of the internal subset. Optional, can
     *        be null
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putDTD(String name, String publicId, String systemId,
                String internalSubset)
        throws DataModelException;

    /**
     * Puts an Element Start event.
     * <p>
     * Should precede namespace and attributes event for the element.
     * @param name qualified name of the element. The local-part and
     *        namespace-URI properties of the name are used, but the prefix is
     *        ignored: to define a namespace (i.e a prefix/namespace-URI
     *        association), use the {@link #putNamespace} method.
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putElementStart(QName name)
        throws DataModelException;

    /**
     * Puts an Element End event. Should balance putElementStart exactly, else
     * the result could be incorrect.
     * @param name qualified name of the element
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putElementEnd(QName name)
        throws DataModelException;

    /**
     * Puts a Namespace declaration event for the current element.
     * <p>
     * Must follow putElementStart and precede the contents of the element. The
     * order is irrelevant. Duplicate namespace declarations are normally an
     * error.
     * <p>
     * A Namespace declaration associates a prefix with a namespace-URI; it is
     * equivalent to the <code>xmlns:</code> pseudo-attributes of XML markup.
     * The empty prefix may be used for the default element namespace.
     * <p>
     * Note that such declarations are not mandatory: prefixes can be
     * synthesized automatically by a XMLSerializer if missing.
     * @param prefix prefix of the namespace, can be the empty string, but not
     *        null
     * @param namespaceURI URI of the namespace: may not be null. The empty
     *        string means that the namespace-URI associated with the current
     *        prefix is erased (a XML 1.1 feature)
     * @return false if the declaration was redundant
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    boolean putNamespace(String prefix, String namespaceURI)
        throws DataModelException;

    /**
     * Puts an Attribute event for the current element.
     * <p>
     * Must follow putElementStart and precede the contents of the element. The
     * order of attributes is kept. A duplicate attribute is normally an error.
     * @param name name of the attribute
     * @param value string value of the attribute
     * @param attrType attribute type: optional, may be null, or one of the
     *        strings "CDATA", "ID", "IDREF", "IDREFS", "NMTOKEN", "NMTOKENS",
     *        "ENTITY", "ENTITIES", or "NOTATION"
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putAttribute(QName name, String value, String attrType)
        throws DataModelException;

    /**
     * Puts a Text Fragment event.
     * <p>
     * A text fragment should appear inside an element. Consecutive text events
     * are coalesced into one. An empty text fragment is discarded.
     * @param text a fragment of text
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putText(String text)
        throws DataModelException;

    /**
     * Puts a Text Fragment event.
     * <p>
     * A text fragment should appear inside an element. Consecutive text events
     * are coalesced into one. An empty text fragment is discarded.
     * @param text an array of characters
     * @param start start offset in the character array
     * @param textLength length of the fragment
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putChars(char[] text, int start, int textLength)
        throws DataModelException;

    /**
     * Puts a Comment event.
     * @param text contents of a comment.
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putComment(String text)
        throws DataModelException;

    /**
     * Puts a Processing-instruction event.
     * @param target target (name) of the PI
     * @param contents contents of the PI
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putProcessingInstruction(String target, String contents)
        throws DataModelException;

    /**
     * Optional flush of the output flow. This is normally called by
     * putDocumentEnd, but it can be used at any time. The actual effect
     * depends on the implementation used (for example on XMLSerializer, the
     * effect is to flush the output byte stream).
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void flush()
        throws DataModelException;

    /**
     * copy-namespace mode used in {@link #putNodeCopy(Node, int)}.
     */
    public static final int NSCOPY_PRESERVE_INHERIT = 1;

    /**
     * copy-namespace mode used in {@link #putNodeCopy(Node, int)}.
     */
    public static final int NSCOPY_NOPRESERVE_INHERIT = 2;

    /**
     * copy-namespace mode used in {@link #putNodeCopy(Node, int)}.
     */
    public static final int NSCOPY_PRESERVE_NOINHERIT = 3;

    /**
     * copy-namespace mode used in {@link #putNodeCopy(Node, int)}.
     */
    public static final int NSCOPY_NOPRESERVE_NOINHERIT = 4;

    /**
     * Copies a Node to the output flow by traversing it recursively.
     * <p>
     * Namespace declarations are handled according to the specified mode. This
     * mode implements the copy-namespaces declaration of XQuery.
     * @param node Node to include in the output flow.
     * @param copyNamespaceMode one of the 4 combinations of PRESERVE and
     *        INHERIT. The normal mode is NSCOPY_PRESERVE_INHERIT.
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putNodeCopy(Node node, int copyNamespaceMode)
        throws DataModelException;

    /**
     * Puts Namespace declarations of the element to the stream according to
     * the specified mode. This mode implements the copy-namespaces declaration
     * of XQuery. Auxiliary method used by putNodeCopy.
     * @param element Element whose namespace declarations are put to the
     *        output flow.
     * @param copyNamespaceMode one of the 4 combinations of PRESERVE and
     *        INHERIT. The normal mode is NSCOPY_PRESERVE_INHERIT.
     * @exception DataModelException <a href='#cex'>reasons</a>
     */
    void putNamespaces(Node element, int copyNamespaceMode)
        throws DataModelException;

    /**
     * Finds the namespace prefix bound to this NS URI in the current context.
     * @param namespaceURI
     * @return a prefix or null
     */
    String getNSPrefix(String namespaceURI);
    
    /**
     * Finds the namespace URI bound to this NS prefix in the current context.
     * @param prefix
     * @return an URI or null
     */
    String getNSURI(String nsPrefix);
}
