<?php 
/**
 * CpmFetch
 * 
 * This is a support library to help grab images from a coppermine database and display them on normal php
 * pages outside the coppermine program. 
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 * Runs on: Tested on coppermine 1.3.2, 
 * HTML: Generates XHTML (as far as I can tell)
 * PHP: Tested on version 4
 * RELEASE VERSION 2.0.0
 *
 * @version $Revision: 1.12 $
 * @package cpmfetch
 */
/**
* cpmfetch_dao is the data access object for cpmfetch.
* @access private
*/
require_once "cpmfetch_dao.php";

/**
* CpmFetch allows you to easily access and display information from  your
* Coppermine Photo Gallery (CPG) in a variety of ways.   CpmFetch respects the
* privacy settings in your gallery and will not display images the public would
* not be allowed to see normally.
* 
* By using CpmFetch you can access:
* - images
* - comments
* - descriptions
* - statistics,
* - last added media
* - random media
*
* then output them to HTML, RSS, Text, Avatars, and more.
*
* CpmFetch will also return just data so that you can use it to query the database,
* then format the output to your specific needs.
* 
* CpmFetch is NOT a modification to CPG - you do not need to worry about it
* interfering with future CPG upgrades, or breaking your existing gallery.
*
*
* @package cpmfetch
*/
class cpm extends cpm_dao {

/**
* This is the version number of this release of CpmFetch
* @access private
*/
var $VERSION = "2.0.0";

/**
* This contains the option array that is submitted by the user.
* It should be cleared out between calls that use it.
* @access private
*/
var $optionHash = array();	

/**
* This contains the style information from the option array
* that is submitted by the user.
* It is cleared out between calls that use it.
* @access private
*/
var $styleHash = array();

/**
* Tracks the current returntype.  Default is "print"
* @access private
*/
var $returntype = "print";		// print, html, resultset


/**
* Creates the CpmFetch object
*
* The function creates and initializes the CpmFetch system.  You need to pass it the
* filename and path to your configuration file.  Typically this is cpmfetch_config.php, but
* you are allowed to have multiple config files, so it may be different.
*
* @tutorial cpmfetch/gettingStarted.pkg
* @example startingcpmfetch.php Example of code used to start CpmFetch
* @param string $config_file Path and filename (path to cpmfetch_config.php) 
* 
*/
function cpm ($config_file = "") {
		$this->cpm_dao($config_file);
}


/**
* Closes down open database connections and cleans up
*
* This will insure that any open database connections are closed if CpmFetch opened them.
* In a forum environment, you may need to ommit this call.  This should be safe as PHP will
* clean up open connections on its own.  But you should use it unless you have a problem with it.
* 
*/
function cpm_close () {
	$this->destroy();
}


/**
* Used for printing debug information.  If will output the parameter directly, preceeding it 
* with a newline, then inside HTML blockquote tag, followed by a newline.
*
* @param string $string Text to be output
* @internal
*/
function debugPrint ($string) {
	print "\n<blockquote>" . $string . "</blockquote>\n";
}


/**
* Return the version of CpmFetch you are running.
*
* @return string representing the version of CpmFetch that is running
*/
function cpm_getVersion() {
	return ($this->VERSION);
}



/**
* Sets how most functions will handle returning information
*
* This sets the output from any calls made after this.  Output can be selected by setting
* this function with:
*
* - print (default)  - This causes most functions to dump the results in HTML directly to the output
* - html - returns a string containing all the HTML for the results
* - resultset - returns an array of associative arrays representing the real data
*
* Note: this will change the behavior of CpmFetch from this point forward, not for just
* the next call.  You can call this as many times as you need to.  If you call it with no parameters
* it will be set back to the default (print).
*
* @param string $_type - set to "html", "print", or "resultset"
* @returns mixed - string if set to html, array if set to resultset
*/
function cpm_setReturnType($_type="print") {
	$_type = strtolower($_type);

	switch ($_type) {
			case('resultset'):
			$this->returntype = "resultset";
			break;
		case('html'):
			$this->returntype = "html";
			break;
		case('print'):
			$this->returntype = "print";
			break;
		default:
			if ($this->cfg['cfDebugMode']) { print "setReturnType defaulted"; }
			$this->returntype = "print";
			break;
	}
}



/**
* Gets the number of items in the gallery.
*
* Prints the value directly to output, but now Supports returnType for print and html.  This
* function will include all media, even media protected as private.
*
* @access private
* @deprecated Will be removed in CpmFetch 4.0
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_listMediaCount ( ) {
	$output = $this->getMediaCount();

	$retval = "";

	switch ($this->returntype) {
		case ('resultset'):
			break;
		case ('html'):
			$retval = $output;
			break;
		case ('print'):
		default:
			print $output;
	}
	return ($retval);
}

/**
* Returns number of items in the gallery, including private entries
*
* Returns value of items in the gallery as a return value.  Does not 
* output directly to the screen.  This is the same as cpm_listMediaCount() with the
* returntype set to html.
*
* @access private
* @return number Shows the number of items in the gallert
* @deprecated Will be removed in CpmFetch 4.  @see cpm_listMediaCount() instead
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_getMediaCount ( ) {
	$output = $this->getMediaCount( );
	return ($output);
}

/**
* Returns number of items in the gallery, including private entries added
* since the date provided by the timestamp parameter
*
* Returns number of items in the gallery added since the timestamp.
* @link http://www.php.net/manual/en/function.strtotime.php PHP information on making the timestamp
*
* @param string $timestamp A valid php timestamp
* @return number Shows the number of items in the gallert
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_getMediaCountAddedSince($timestamp) {
	$output = $this->getMediaCountAddedSince($timestamp);
	$retval = "";

	switch ($this->returntype) {
		case ('resultset'):
			break;
		case ('html'):
			$retval = $output;
			break;
		case ('print'):
		default:
			print $output;
	}

	return ($retval);

}

/**
* Returns number of items in the gallery for a time range
*
* The results include private entries added
* in the number of days specified by the $numberOfDays
*
*
* @param int $numberOfDays The number of days going back from now to include
* @return number Shows the number of items in the gallert
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_getMediaCountAddedInLast($numberOfDays) {

		$lwd = getdate(strtotime("-" . $numberOfDays . " day"));
		//$verbosedate = date("l, F j Y",strtotime("-" . $numberOfDays . " day"));
		$timestamp = $lwd[0];

		$output = $this->getMediaCountAddedSince($timestamp);
		$retval = "";

		switch ($this->returntype) {
			case ('resultset'):
				break;
			case ('html'):
				$retval = $output;
				break;
			case ('print'):
			default:
				print $output;
		}

	return ($retval);

}


/**
* Returns number of items in the gallery, including private entries, as defined by source or source = "" for everything
*
* Returns value of items in the gallery
* By Default prints the number directly to output.  Supports returnType for print and html
*
* @tutorial explainations.sourcetag.pkg
* 
* @param string $source A valid CpmFetch source string
* @return number Shows the number of items in the gallery
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_listMediaCountFrom ($source ) {
	$output = $this->getMediaCountFrom($source );

	$retval = "";

	switch ($this->returntype) {
		case ('resultset'):
			break;
		case ('html'):
			$retval = $output;
			break;
		case ('print'):
		default:
			print $output;
	}

	return ($retval);
}


/**
* Displays number representing the number of items in an album, including private entries.
*
* By Default prints the number directly to output.  Supports returnType for print and html
*
* @access private
* @param int $albumid Number representing the album number
* @since Version 0.5
* @deprecated Will be removed in CpmFetch 4.0
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_listMediaCountForAlbum ($albumid) {
	$output = $this->getMediaCountForAlbum($albumid);
	$retval = "";

	switch ($this->returntype) {
		case ('resultset'):
			break;
		case ('html'):
			$retval = $output;
			break;
		case ('print'):
		default:
			print $output;
	}

	return ($retval);
}

/**
* Returns number representing the number of items in an album, including private entries
*
* Does not Supports returnType,
*
* @param int $albumid Number representing the album number
* @return number Representing the count for a specific album
* @since Version 0.5
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_getMediaCountForAlbum ($albumid) {
	$output = $this->getMediaCountForAlbum($albumid);
	return ($output);
}
		
/**
* Outputs stats based on a format submitted
*
* The format is specified by placeholders indicated by percent signs '%'.  If you actually want a percent sign
* you will need to put two percent signs in a row '%%'.  The most up to date list for supported placeholders
* can be found in the coppermine_dao formatStats function.  But the main ones are:
* - %f - file count
* - %a - album count
* - %c - category count
* - %v - view count
* - %n - comment count (note count)
* 
* For example: "There are %f files in %c categories containing %a albums, having served %v views and getting %n comments"
* This supports the returnType setting except for type resultset.  Returns nothing in that case.
*
* @param string $format The format template to output in
* @since Version 0.7
* 
*/
function cpm_formatStats($format) {

	$output =  $this->formatStats($format);

	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			break;
		case ('html'):
			$retval = $output;
			break;
		case ('print'):
		default:
			print $output;
	}

	return ($retval);
}
	
/**
* Displays images of last added media
*
* This will generate a table and populate it with the last added media and style
* it based on entries in the style hash. The number of photos returned matches the number
* of rows multiplied by the number of columns.    Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
* @return string XHTML code
* @since Version 0.5
* @tutorial explainations.optionsarray.pkg
*
* @access private
* @deprecated Will be removed in CpmFetch 4.0
* @see cpm_viewLastAddedMediaFrom()
*/
function cpm_viewLastAddedMedia($rows, $columns, $options="") {
	$this->loadOptions($options);
	//$resultset = $this->getLastAddedMedia($rows*$columns);
	$resultset = $this->getLastAddedMediaFrom("",$rows*$columns);
	$this->addDescriptionsToResultSet($resultset);
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();
	return ($retval);
}

/**
* Displays images of last added media for a specific album
*
* This will generate a table and populate it with the last added media and style
* it based on entries in the style hash. The number of photos returned matches the number
* of rows multiplied by the number of columns.  Supports the setReturnType option.
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param int $albumid The album id of the album you want to display
* @param array $options Optional, Configuration options
* @tutorial cpmfetch/explainations.optionsarray.pkg
*
* @access private
* @deprecated Will be removed in CpmFetch 1.4
* @see cpm_viewLastAddedMediaFrom()
* 
*/
function cpm_viewLastAddedMediaFromAlbum($rows, $columns, $albumid, $options="") {
	$this->loadOptions($options);
	//$resultset = $this->getLastAddedMediaFromAlbum($albumid, $rows*$columns);
	$resultset = $this->getLastAddedMediaFrom("album=$albumid", $rows*$columns);
	$this->addDescriptionsToResultSet($resultset);
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();
	return ($retval);
}

/**
* Displays images of last added media for a specific category
*
* This will generate a table and populate it with the last added media and style
* it based on entries in the style hash. The number of photos returned matches the number
* of rows multiplied by the number of columns.  Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param int $catid The category id of the album you want to display
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* @access private
* @deprecated Will be removed in CpmFetch 1.4
* @see cpm_viewLastAddedMediaFrom()
* 
*/
function cpm_viewLastAddedMediaFromCategory($rows, $columns, $catid, $options="") {
	$this->loadOptions($options);
	//$resultset = $this->getLastAddedMediaFromCategory($catid,$rows*$columns);
	$resultset = $this->getLastAddedMediaFrom("cat=$catid",$rows*$columns);
	$this->addDescriptionsToResultSet($resultset);
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();
	return ($retval);
}

/**
* Displays images of last added media for a specific category
*
* This will generate a table and populate it with the last added media and style
* it based on entries in the style hash. The number of photos returned matches the number
* of rows multiplied by the number of columns.  Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param int $catid The category id of the album you want to display
* @param array $options Optional, Configuration options
* @tutorial cpmfetch/explainations.optionsarray.pkg
* @tutorial cpmfetch/explainations.sourcetag.pkg
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_viewLastAddedMediaFrom($source, $rows, $columns, $options="") {
	$this->loadOptions($options);
	$resultset = $this->getLastAddedMediaFrom($source,$rows*$columns);
	$this->addDescriptionsToResultSet($resultset);

	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();
	return ($retval);
}
		
/**
* Displays random images from specific album or category
*
* This will generate a table and populate it with random media from the combination of
* categories and albums specified and style it based on entries in the style hash.
* The number of photos returned matches the number of rows multiplied by the number of columns.
* Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param int $sources The sources you want to draw from ("cat=1,4,5:album=6,7,8")
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_viewRandomMediaFrom($source,$rows, $columns, $options="") {
	$this->loadOptions($options);
	$resultset = $this->getRandomImageFrom ($source, $rows*$columns);
	$this->addDescriptionsToResultSet($resultset);

	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();
	return ($retval);
}



/**
* Generates a list of Albums based
*
*
*
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
*
* @param string $source 
* @param int $rows
* @param int $colums
* @param mixed $options
* 
*/
function cpm_getAlbumListFrom ($source, $rows, $columns, $options="") {
		$this->loadOptions($options);
		$resultset = $this->getAlbumListFrom($source);
		$this->addDescriptionsToResultSet($resultset);
		
		$retval = "";
		switch ($this->returntype) {
			case ('resultset'):
				$retval = $resultset;
				break;
			case ('html'):
				$retval = $this->createTable($resultset,$rows,$columns);
				break;
			case ('print'):
			default:
				print $this->createTable($resultset,$rows,$columns);
		}

		$this->clearOptions();		
		return ($retval);
}

/**
*
* @tutorial explainations.optionsarray.pkg
* 
* @param mixed $options A valid CpmFetch options array
*/
function cpm_getCategoryList ($options="") {
		$this->loadOptions($options);
		$resultset = $this->getCategoryListFrom();
		$this->addDescriptionsToResultSet($resultset);
		
		$retval = "";
		switch ($this->returntype) {
			case ('resultset'):
				$retval = $resultset;
				break;
			case ('html'):
				//$retval = $this->createTable($resultset,$rows,$columns);
				break;
			case ('print'):
			default:
				//print $this->createTable($resultset,$rows,$columns);
		}

		$this->clearOptions();		
		return ($retval);
}


/**
*
* This (in thoery returns the last updated albums and the last photo added to them
*
* @param int $source The category id of the album you want to display
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_viewLastUpdatedAlbumsFrom($source, $rows, $columns, $options="") {
	$this->loadOptions($options);
	$resultset = $this->getLastUpdatedAlbumsFrom($source,$rows * $columns);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);
}


			
/**
* Displays random images from specific album
*
* This will generate a table and populate it with random media from one album and style
* it based on entries in the style hash.  The number of photos returned matches the number 
* of rows multiplied by the number of columns.  Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param int $albumid The album id of the album you want to display
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
*
* @deprecated Will be removed in CpmFetch 4.0
* @access private
* @see cpm_viewRandomMediaFrom()
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_viewRandomMediaFromAlbum ($rows, $columns, $albumid, $options="") {
	$this->loadOptions($options);
	//$resultset = $this->getRandomImageFromAlbum ($albumid, $rows*$columns);
	$resultset = $this->getRandomImageFrom ("album=$albumid", $rows*$columns);
	$this->addDescriptionsToResultSet($resultset);
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);
}
		
/**
* Displays random images from specific category
*
* This will generate a table and populate it with random media from one category and style
* it based on entries in the style hash.  The number of photos returned matches the number 
* of rows multiplied by the number of columns.  Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param int $catid The category id of the album you want to display
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
*
* @access private
* @deprecated This will be removed in CpmFetch 4.0
* @see cpm_viewRandomMediaFrom()
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_viewRandomMediaFromCategory ($rows, $columns, $catid, $options="") {
	$this->loadOptions($options); 
	//$resultset = $this->getRandomImageFromCategory ($catid, $rows*$columns);
	$resultset = $this->getRandomImageFrom ("cat=$catid", $rows*$columns);	
	$this->addDescriptionsToResultSet($resultset);
	$retval = "";

	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);
}
			
/**
* Displays random images from database
*
* This will generate a table and populate it with the random media and style
* it based on entries in the style hash.  The number of photos returned matches the number 
* of rows multiplied by the number of columns.  Supports the setReturnType option.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
* @since Version 0.5
* @deprecated Will be removed in CpmFetch 4.0
* @access private
* @tutorial explainations.optionsarray.pkg
* @see cpm_viewRandomMediaFrom()
* 
* @todo allow for media to be specified / filtered - jpg, mp3, etc
*/
function cpm_viewRandomMedia($rows, $columns, $options="") {
	$this->loadOptions($options); 
	//$resultset = $this->getRandomImage($rows*$columns);
	$resultset = $this->getRandomImageFrom("",$rows*$columns);
	$this->addDescriptionsToResultSet($resultset);
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);
}


/**
* Selects one media item by the CPG pid (Picture Id)
* @tutorial explainations.optionsarray.pkg
*
* @param number $pid A valid picture id number from your gallery
* @param mixed $options A valid CpmFetch options array 
*/
function cpm_viewMediaByPid($pid, $options="") {
	$this->loadOptions($options); 
	$resultset = $this->getMediaByPid($pid);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$rows = 1;
			$columns = 1;
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			$rows = 1;
			$columns = 1;
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);
}

/**
* Displays last comments with images
*
* This will generate a table and populate it with the last commented images
* The number of photos returned matches the number of rows multiplied by the number of columns.
* Supports the setReturnType option.  This acts like many others excet you now have two new subtitle 
* and title options %C - Which is the comment itself, and %a - which is the author of the comment.
*
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* @since Version 1.1.3
* @todo allow for media to be specified / filtered - jpg, mp3, etc
* 
*/	
function cpm_viewLastCommentedImages($rows, $columns, $options = "") {
	$this->loadOptions($options); 
	$resultset = $this->getLastImagesWithComments($rows*$columns);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}
	$this->clearOptions();		
	return ($retval);
}
	
/**
*
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
* 
* @param string $source A valid CpmFetch source string
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
*/
function cpm_viewTopRatedMediaFrom ($source, $rows, $columns, $options="") {
	$this->loadOptions($options); 
	
	$resultset = $this->getTopRatedMediaFrom ($source, $rows * $columns);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);

}


/**
*
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
* 
* @param string $source A valid CpmFetch source string
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
*/
function cpm_viewMostVotedMediaFrom ($source,  $rows, $columns, $options="") {

	$this->loadOptions($options); 
	$resultset = $this->getMostVotedMediaFrom ($source, $rows * $columns);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);

}

/**
*
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
* 
* @param string $source A valid CpmFetch source string
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
*/
function cpm_viewRandomMostViewedMediaFrom ($source,  $rows, $columns, $options="") {
	$this->loadOptions($options); 
	$resultset = $this->getRandomMostViewedMediaFrom($source, $rows * $columns);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);
}


/**
*
* @tutorial explainations.optionsarray.pkg
* @tutorial explainations.sourcetag.pkg
* 
* @param string $source A valid CpmFetch source string
* @param int $rows The number of rows desired in the output
* @param int $columns The number of columns desired in the output
* @param array $options Optional, Configuration options
*/
function cpm_viewRandomTopRatedMediaFrom ($source,  $rows, $columns, $options="") {

	$this->loadOptions($options); 
	$resultset = $this->getRandomTopRatedMediaFrom ($source, $rows * $columns);
	$this->addDescriptionsToResultSet($resultset);
	
	$retval = "";
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $this->createTable($resultset,$rows,$columns);
			break;
		case ('print'):
		default:
			print $this->createTable($resultset,$rows,$columns);
	}

	$this->clearOptions();		
	return ($retval);

}


/**
* Displays a list of albums with icons and count over past days
*
* This function does not currently use the stylehash functionality.   Supports the setReturnType option.
*
* @param int $numberOfDays The number of days going back to be included.
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* @todo add style has functionality
* @todo allow for media to be specified / filtered - jpg, mp3, etc
* 
*/
function cpm_showCategoriesUpdatedOverLastDays($numberOfDays,$options="") {

	$this->loadOptions($options);
	
	$htmlOut = "";
	$retval = "";		
	
	$lwd = getdate(strtotime("-" . $numberOfDays . " day"));
	$verbosedate = date("l, F j Y",strtotime("-" . $numberOfDays . " day"));
	$timestamp = $lwd[0];
				
	$mediacount = $this->getMediaCountAddedSince($timestamp);

	if ($mediacount == 0) {
			$htmlOut .= "No categories have been updated since {$verbosedate}";
	}
	else {

		$resultset = $this->getMediaAddedToCategoriesSince($timestamp);
		$this->addDescriptionsToResultSet($resultset);
		
		if ($this->returntype != 'resultset') {
			$htmlOut .= "\n" . $this->htmlTagTABLE() . "\n";
									
			$htmlOut .= $this->htmlTagTR() . "\n" . 
							$this->htmlTagTH("<strong>Since {$verbosedate} we've added {$mediacount} images</strong>",'colspan="2"');
			$htmlOut .= $this->htmlTagTR_END();				
						
			foreach ($resultset as $row) {							
				$htmlOut .= $this->htmlTagTR();
				if (array_key_exists('alttag',$this->optionHash)) {
					$alttag = $this->createDescription($this->optionHash['alttag'],$row);
				}else{
					$alttag = $this->createDescription("{{pFilename}}",$row);
				}
																
				$attributes = 'alt="' . $alttag . '" title="' . $alttag . '"';
				$imgfile = $this->cfg['cpg_url'] .$this->urlEncodeImagePath($this->getImageToUse($row['pFilepath'],$row['pFilename'], $this->getPrefixToUse()));


				$link = $this->createlink($row['pFilepath'],rawurlencode($row['pFilename']),$row['pAid'],$row['pPid'],$this->getoption('imagelink'));		
				
				if ($link != "") {
					$htmlOut .= $this->htmlTagTD($this->htmlTagAHREF($link,$this->htmlTagIMG($imgfile,$attributes)));
				} else {
					$htmlOut .= $this->htmlTagTD($this->htmlTagIMG($imgfile,$attributes));
				}
								
				$htmlOut .= $this->htmlTagTD("<strong>{$row['count']} new in " . 
					$this->htmlTagAHREF("{$this->cfg['cpg_url']}index.php?cat={$row['cCid']}",$row['cName']) . "</strong><br>{$row['count']}",'valign="top"');
											
				$htmlOut .= "\n" . $this->htmlTagTR_END() . "\n";				
			} 	
					
			$htmlOut .= "\n" . $this->htmlTagTABLE_END() . "\n";
		}	
	
	}
	
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $htmlOut;
			break;
		case ('print'):
		default:
			print $htmlOut;
	}
			
	$this->clearOptions();
	return ($retval);
}

	
/**
* Displays a list of albums with thumbnails and count over past days
*
* Supports the setReturnType option.
*
* @param int $numberOfDays The number of days going back to be included.
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* @todo add style has functionality
* @todo allow for media to be specified / filtered - jpg, mp3, etc
* 
*/
function cpm_showAlbumsUpdatedOverLastDays($numberOfDays,$options="") {
	
	$this->loadOptions($options);
	
	$htmlOut = "";
	$retval = "";		
	
	$lwd = getdate(strtotime("-" . $numberOfDays . " day"));
	$verbosedate = date("l, F j Y",strtotime("-" . $numberOfDays . " day"));
	$timestamp = $lwd[0];
				
	$mediacount = $this->getMediaCountAddedSince($timestamp);

	if ($mediacount == 0) {
			$htmlOut .= "No albums have been updated since {$verbosedate}";
	}
	else {

		$resultset = $this->getMediaAddedSince($timestamp);
		$this->addDescriptionsToResultSet($resultset);
		
		if ($this->returntype != 'resultset') {
			$htmlOut .= "\n" . $this->htmlTagTABLE() . "\n";
									
			$htmlOut .= $this->htmlTagTR() . "\n" . 
							$this->htmlTagTH("<strong>Since {$verbosedate} we've added {$mediacount} images</strong>",'colspan="2"');
			$htmlOut .= $this->htmlTagTR_END();				
						
			foreach ($resultset as $row) {				
				$htmlOut .= $this->htmlTagTR();
				
				if (array_key_exists('alttag',$this->optionHash)) {
					$alttag = $this->createDescription($this->optionHash['alttag'],$row);
				}else{
					$alttag = $this->createDescription("{{pFilename}}",$row);
				}
																
				$attributes = 'alt="' . $alttag . '" title="' . $alttag . '"';
				$imgfile = $this->cfg['cpg_url'] .$this->urlEncodeImagePath($this->getImageToUse($row['pFilepath'],$row['pFilename'], $this->getPrefixToUse()));

				$link = $this->createlink($row['pFilepath'],rawurlencode($row['pFilename']),$row['pAid'],$row['pPid'],$this->getoption('imagelink'));		
				
				if ($link != "") {
					$htmlOut .= $this->htmlTagTD($this->htmlTagAHREF($link,$this->htmlTagIMG($imgfile,$attributes)));
				} else {
					$htmlOut .= $this->htmlTagTD($this->htmlTagIMG($imgfile,$attributes));
				}
								
				$htmlOut .= $this->htmlTagTD("<strong>{$row['count']} new in " . 
					$this->htmlTagAHREF("{$this->cfg['cpg_url']}thumbnails.php?album={$row['pAid']}",$row['aTitle']) .
					" of {$row['cName']}</strong><br>{$row['count']}",'valign="top"');
											
				$htmlOut .= "\n" . $this->htmlTagTR_END() . "\n";				
			} 	
					
			$htmlOut .= "\n" . $this->htmlTagTABLE_END() . "\n";
		}	
	
	}
	
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $htmlOut;
			break;
		case ('print'):
		default:
			print $htmlOut;
	}
			
	$this->clearOptions();
	return ($retval);
}
			
		
/**
* Displays a list of category, album and count over past days.    Supports the setReturnType option.
*
* @param int $numberOfDays The number of days going back to be included.
* @param array $options Optional, Configuration options
* @tutorial explainations.optionsarray.pkg
* 
* @todo add style has functionality
* @todo allow for media to be specified / filtered - jpg, mp3, etc
* 
*/
function cpm_listMediaAddedOverLastDays($numberOfDays,$options="") {

	$this->loadOptions($options);
	$htmlOut = "";
	$retval = "";
			
	$lwd = getdate(strtotime("-" . $numberOfDays . " day"));
	$verbosedate = date("l, F j Y",strtotime("-" . $numberOfDays . " day"));
	$timestamp = $lwd[0];
				
	$mediacount = $this->getMediaCountAddedSince($timestamp);

	if ($mediacount == 0) {
		$htmlOut .= "No images have been added since {$verbosedate}";
	}
	else {
	
		$resultset = $this->getMediaAddedSince($timestamp);
		$this->addDescriptionsToResultSet($resultset);
	
		if ($this->returntype != 'resultset') {	
			
			$htmlOut .= $this->htmlTagTABLE('width="80%"');
						
			$htmlOut .= $this->htmlTagTR() . "\n" . 
							$this->htmlTagTH("<strong>Since {$verbosedate} we've added {$mediacount} images</strong>", 'colspan="3" align="left"');
			$htmlOut .= $this->htmlTagTR_END();				
			
							
			$htmlOut .= $this->htmlTagTR() . "\n" . 
							$this->htmlTagTH("<strong>Category name</strong>",'align="left"') .
							$this->htmlTagTH("<strong>Album name</strong>",'align="left"') .
							$this->htmlTagTH("<strong>New Count</strong>",'align="left"');
							
			$htmlOut .= $this->htmlTagTR_END();				
			
			$lastcname="";

			foreach ($resultset as $row) {				
				$htmlOut .= $this->htmlTagTR() . "\n";
				if ($row['cName'] == $lastcname) {
					$htmlOut .= $this->htmlTagTD("&nbsp;");
				}
				else {
					$htmlOut .= $this->htmlTagTD("<strong>{$row['cName']}</strong>");
					$lastcname = $row['cName'];	
				}
						
				$htmlOut.=$this->htmlTagTD($this->htmlTagAHREF("{$this->cfg['cpg_url']}thumbnails.php?album={$row['pAid']}",$row['aTitle']),'align="left"');
							
				$htmlOut .= $this->htmlTagTD("{$row['count']} new",'align="left"');
	
				$htmlOut .= "\n" . $this->htmlTagTR_END() . "\n";				
								
			} 
	
			$htmlOut .= "\n" . $this->htmlTagTABLE_END() . "\n";
		}
			
	}
	
	switch ($this->returntype) {
		case ('resultset'):
			$retval = $resultset;
			break;
		case ('html'):
			$retval = $htmlOut;
			break;
		case ('print'):
		default:
			print $htmlOut;
	}
			
	$this->clearOptions();
	return ($retval);
}	// END OF FUNCTION cpm_viewMediaAddedOverLastDays


/**
* Creates a table to display images
*
* This is an internal helper function to construct tables to hold the images.
*
* @returns string (XHTML code)
* @access private
*/
function createTable ($resultset,$rows,$columns) {

$optionshash = $this->optionHash;
$htmlOut = "";
$alttag = "";
$subtitle = "";

//$fileprefix=$this->defaultimagesize;
$maximages = $rows * $columns;
$numimages = count ($resultset);
$arrayptr = 0;

if ($numimages < $maximages) {
	$gap = $maximages - $numimages;

	if ($rows > $columns) {
		$numofrowstolose = floor($gap / $columns);
		$rows = $rows - $numofrowstolose;	
	} 
	elseif ($columns > $rows) {
		$numofcolstolose = floor($gap / $rows);
		$columns = $columns - $numofcolstolose;
	}
	else {
		$totalloss = floor($gap / $rows);
		$first = floor($totalloss / 2);
		$columns = $columns - $first;
		$rows = $rows - ($totalloss-$first);
	}
}
	
$htmlOut .= $this->htmlTagTABLE() . "\n";
		
for($rw=0; $rw < $rows ; $rw++) {
$htmlOut .= $this->htmlTagTR() . "\n";
	
	for ($cl=0; $cl < $columns; $cl++) {
		
		if ($arrayptr == $numimages) {		// num images starts a 1 while the array starts at 0
			$row = "";
		} else {
			$row = $resultset[$arrayptr];
			$arrayptr++;	
		}

		if ($row != "") {
			if (array_key_exists('alttag',$this->optionHash)) {
				$alttag = $this->createDescription($this->optionHash['alttag'],$row);
			}else{
				$alttag = $this->createDescription("{{pFilename}}",$row);
			}
															
			$attributes = 'alt="' . $alttag . '" title="' . $alttag . '"';

			$imgfile = $this->getImageToUse($row['pFilepath'],$row['pFilename'], $this->getPrefixToUse());
			$imgfile = $this->cfg['cpg_url'] .    $this->urlEncodeImagePath($imgfile);

// New link code
			if (array_key_exists('linktemplate',$this->optionHash)) {
				$link =  $this->createDescription($this->optionHash['linktemplate'],$row);
			} else {
				$link = $this->createlink($row['pFilepath'], rawurlencode($row['pFilename']),$row['pAid'],$row['pPid'],$this->getoption('imagelink'));
			}

//				$link = $this->createlink($row['pFilepath'], rawurlencode($row['pFilename']),$row['pAid'],$row['pPid'],$this->getoption('imagelink'));

			$subtitle = "";				
			if (! array_key_exists('noimage',$this->optionHash)) {
				$subtitle = '<br />';
			}
		
			if(array_key_exists('subtitle',$this->optionHash) && array_key_exists('subtitlelink',$this->optionHash)) {
				$subtitle .= $this->htmlTagAHREF($link, $this->createDescription($this->optionHash['subtitle'],$row));
			}elseif (array_key_exists('subtitle',$this->optionHash)) {
				$subtitle .= $this->createDescription($this->optionHash['subtitle'],$row);	
			} else {
				$subtitle = "";
			}
							
			if ($link != "" && !(array_key_exists('noimage',$this->optionHash))) {
				$htmlOut .= $this->htmlTagTD($this->htmlTagAHREF($link,$this->htmlTagIMG($imgfile,$attributes)) . "<br />" . $subtitle);
			} else {
				$htmlOut .= $this->htmlTagTD($this->htmlTagIMG($imgfile,$attributes) . $subtitle);
			}
		}
		else
		{
		$htmlOut .= $this->htmlTagTD('&nbsp;');
		}
	} // end column for
$htmlOut .= "\n" . $this->htmlTagTR_END() . "\n";

} // end row for
	
$htmlOut .= "\n" . $this->htmlTagTABLE_END() . "\n";

return $htmlOut;

} // end function

/**
*	
* @access private
*/
function htmlTagTABLE($extraAttributes="") { 

	$htmlOut = '<table ';
	if (array_key_exists('tablestyle',$this->styleHash)) $htmlOut .= " {$this->styleHash['tablestyle']}";	
	if ($extraAttributes != "") $htmlOut .= " $extraAttributes";
	$htmlOut .= " " . $this->getOptionAttributes("tableattributes");
	$htmlOut .= '>' . "\n";	
	return $htmlOut;	
}
		
/**
*	
* @access private
*/
function htmlTagTABLE_END() { return "</table>\n"; }


/**
*	
* @access private
*/
function htmlTagTR($extraAttributes="") { 
	$htmlOut = '<tr';
	if (array_key_exists('rowstyle',$this->styleHash)) $htmlOut .= " {$this->styleHash['rowstyle']}";	
	if ($extraAttributes != "") $htmlOut .= " $extraAttributes";
	$htmlOut .= $this->getOptionAttributes("rowattributes");
	$htmlOut .= '>' . "\n";	
	return $htmlOut;	
}

/**
*	
* @access private
*/
function htmlTagTR_END() { return "</tr>\n"; }


/**
*	
* @access private
*/
function htmlTagTH($contents,$extraAttributes="") {
	$htmlOut = '<th';
	if (array_key_exists('tableheadstyle',$this->styleHash)) $htmlOut .= " {$this->styleHash['tableheadstyle']}";	
	if ($extraAttributes != "") $htmlOut .= " $extraAttributes";
	$htmlOut .= $this->getOptionAttributes("rowheaderattributes");
	$htmlOut .= '>' . $contents . '</th>' . "\n";
	return $htmlOut;	
}

/**
*	
* @access private
*/
function htmlTagTD($contents, $extraAttributes="") {
	$htmlOut = '<td';
	if (array_key_exists('cellstyle',$this->styleHash)) $htmlOut .= " {$this->styleHash['cellstyle']}";	
	if ($extraAttributes != "") $htmlOut .= " " . $extraAttributes;
	$htmlOut .= $this->getOptionAttributes("cellattributes");
	$htmlOut .= '>' . $contents . '</td>' . "\n";
		return $htmlOut;	
}

/**
*	
* @access private
*/

function htmlTagAHREF($target, $contents, $extraAttributes="") {
	
	$htmlOut = '<a';

	if ( ( (array_key_exists('imagelink',$this->optionHash)) 
					&& $this->optionHash['imagelink'] == 'large') 
					&& array_key_exists('windowtarget',$this->optionHash ) ) 
	{

			if ($this->optionHash['windowtarget'] == '_blank-js' ) {
				$htmlOut .= " href=\"javascript:cmfshowpic('$target','')\" ";

			} elseif ($this->optionHash['windowtarget'] == '_blank-cfshow') {
				
				$htmlOut .= " href=\"javascript:void(window.open('cfshow.php?pic=$target','cpmFetch','width=825,height=625,toolbars=0,scrollbars=1,resizable=1'));\""; 


			} elseif ($this->optionHash['windowtarget'] == '_blank') {
				$htmlOut .= " href=\"$target\" target=\"_blank\" ";
			
			} else {
				$htmlOut .= " href=\"$target\" target=\"{$this->optionHash['windowtarget']}\"  ";
			}
	
	} // handling less than full size photos
	else {

		if (array_key_exists('windowtarget',$this->optionHash)) $htmlOut .= ' target="' . $this->optionHash['windowtarget'] . '"';
		$htmlOut .= " href=\"$target\" ";

	}		

	if (array_key_exists('linkstyle',$this->styleHash)) $htmlOut .= " {$this->styleHash['linkstyle']}";	
	if ($extraAttributes != "") $htmlOut .= " " . $extraAttributes;
	$htmlOut .= $this->getOptionAttributes("linkattributes");

	$htmlOut .= " >$contents</a>" ;
	
	return $htmlOut;	
}


/**
*	
* @access private
*/
function htmlTagIMG($imagefile,$extraAttributes) {
	
	$htmlOut = "";
	if (! array_key_exists('noimage',$this->optionHash)) {
		$htmlOut .= '<img  src="' . $imagefile . '"';
		if (array_key_exists('imagestyle',$this->styleHash)) $htmlOut .= " {$this->styleHash['imagestyle']}";	
		if ($extraAttributes != "") $htmlOut .= " " . $extraAttributes;
		$htmlOut .= $this->getOptionAttributes("imageattributes");
		$htmlOut .= '  />';
	}		

	return ($htmlOut);			
} 


/**
*	
* @access private
*/
function getPrefixToUse() {

	//$fileprefix=$this->defaultimagesize;
	$fileprefix = "";		

	if (array_key_exists('imagesize',$this->optionHash)) {
		if ($this->optionHash['imagesize'] == 'thumb') {
			$fileprefix = $this->cfg['thumb_pfx']; 
		} elseif ($this->optionHash['imagesize'] =='int') {
			$fileprefix = $this->cfg['normal_pfx'];		
		} elseif ($this->optionHash['imagesize'] == 'large') {
			$fileprefix="";	
		} 
	}
	elseif ($this->cfg['cfDefaultImageSize'] == "thumb") {
		$fileprefix = $this->cfg['thumb_pfx']; 
	}
	elseif ($this->cfg['cfDefaultImageSize'] == "normal") {
		$fileprefix = $this->cfg['normal_pfx'];		
	}
	elseif ($this->cfg['cfDefaultImageSize'] == "largel") {
		$fileprefix="";	
	}

	return $fileprefix;
}

/**
*	
* @access private
*/
function addDescriptionsToResultSet(&$resultset) {
	if (array_key_exists('subtitle',$this->optionHash)) {
		$newarray = array();
		foreach ($resultset as $row) {
			$row['cpmSubtitle'] = $this->createDescription($this->optionHash['subtitle'],$row);
			array_push($newarray,$row);
		}
		$resultset = $newarray;
	}

	if (array_key_exists('alttag',$this->optionHash)) {
		$newarray = array();
		foreach ($resultset as $row) {
			$row['cpmAlttag'] = $this->createDescription($this->optionHash['alttag'],$row);
			array_push($newarray,$row);
		}
		$resultset = $newarray;
	}

}


/**
*
* Sets the member $optionHash with user submission		
* @access private
*/
function loadOptions($options="") {

	$this->optionHash = array();
	$this->styleHash = array();
	
	if (is_array($options)) {

		$tmparray = array();
		foreach(array_keys($options) as $optionkey)
		{
				$tmparray[strtolower($optionkey)] = $options[$optionkey];
		}

		$options = $tmparray;
		
		if (array_key_exists('tablestyle',$options)) $this->styleHash['tablestyle'] = 'class="' . $options['tablestyle'] . '"';		
		if (array_key_exists('tableheadstyle',$options)) $this->styleHash['tableheadstyle'] = 'class="' . $options['tableheadstyle'].'"';
		if (array_key_exists('rowstyle',$options)) $this->styleHash['rowstyle'] = 'class="' . $options['rowstyle'] . '"';		
		if (array_key_exists('cellstyle',$options)) $this->styleHash['cellstyle'] = 'class="' . $options['cellstyle'] . '"';		
		if (array_key_exists('imagestyle',$options)) $this->styleHash['imagestyle'] = 'class="' . $options['imagestyle'] . '"';		
		if (array_key_exists('linkstyle',$options)) $this->styleHash['linkstyle'] = 'class="' . $options['linkstyle'] . '"';		

		$this->optionHash = $options;	

		if (array_key_exists('imagewidth',$options)) {
				$this->insertOptionAttributes('imageattributes','width',$options['imagewidth']);
				
		}

		if (array_key_exists('imageheight',$options)) {
				$this->insertOptionAttributes('imageattributes','height',$options['imageheight']);
		}

	}
	
} // end function loadOptions
	
/**
*	
* @access private
*/
function getOption($optionname) {
	if (array_key_exists($optionname,$this->optionHash)) {
		return $this->optionHash[$optionname];
	} else {
		return "";
	}
}


/**
*	
* @access private
*/
function insertOptionAttributes($elementname,$key,$value) {
	if (array_key_exists($elementname, $this->optionHash)) {
		$attributearray = $this->optionHash[$elementname];
	}
	else
	{
		$attributearray = array( );
	}

	$attributearray[$key] = $value;

	$this->optionHash[$elementname] = $attributearray;
		
}


/**
*	
* @access private
*/
function getOptionAttributes($elementname) {
	$retval = "";
	
	if (array_key_exists($elementname, $this->optionHash)) {
		$attributearray = $this->optionHash[$elementname];
		foreach ($attributearray as $attr => $value) {
			$retval .= " " . $attr . '="' . $value . '" ';
		}	
	}
	return ($retval);
}


/**
*	
* @access private
*/
function clearOptions( ) {
	$this->optionHash = array();	
	$this->styleHash = array();
}	





			
} // end class

?>
