#ifndef GGAF_CORE_CURTAIN_H_
#define GGAF_CORE_CURTAIN_H_
#include "GgafCommonHeader.h"
#include "jp/ggaf/core/Object.h"

namespace GgafCore {

enum CurtainState {
    OPEN,
    OPENING,
    OPENED,
    CLOSE,
    CLOSING,
    CLOSED
};

/**
 * J[e .
 * J[eJA̊Ǘ܂B<BR>
 * <pre>
 *
 *           |             |         <-- J[e[~߂hĂʒu
 *           v             v
 *     +-----+------+------+----+    <-- J[e[
 *           +------+
 *           |  J  |
 *           |  b  |
 *           |  e  |
 *           |    |
 *           |      |
 *           +------+
 *
 *
 *     <---------- 1 ----------->
 *     <- 2 ->
 *     <------- 3 --------->
 *     <---- 4 ----->
 *           <----- 5 ----->
 *
 *  1. _curtain_rail_length iJ[e[Ŝ̒j
 *  2. _min_position  (J[e100% JƔ肳ʒuj
 *  3. _max_position  (J[e100% 肳ʒuj
 *  4. _curtain_position ĩ݂J[eʒuj
 *  5. _moveable_length   (J[e͈() = 3 - 2 )
 * </pre>
 * @version 1.00
 * @since 2010/04/21
 * @author Masatoshi Tsuge
 */
template<class T>
class Curtain : public Object {

public:
    /** [r]݂̃J[ȅ */
    CurtainState _state;
    /** [r]J[e[̒ */
    const T _curtain_rail_length;
    /** [r]J[e100% JƔ肳ʒu */
    const T _min_position;
    /** [r]J[e100% 肳ʒu */
    const T _max_position;
    /** [r]J[e͈() */
    const T _moveable_length;
    /** [r]݂̃J[e̕܂Ă钷 */
    T _curtain_position;
    /** [r]J[eJۂ̑x */
    T _opening_velocity;
    /** [r]J[e߂ۂ̑x */
    T _closing_velocity;

public:
    /**
     * RXgN^ .
     * @param prm_curtain_rail_length J[e[̒
     * @param prm_min_position J[e100% JƔ肳ʒu
     * @param prm_max_position J[e100% 肳ʒu
     */
    Curtain(T prm_curtain_rail_length,
                T prm_min_position,
                T prm_max_position) : Object(),
    _curtain_rail_length(prm_curtain_rail_length),
    _min_position(prm_min_position),
    _max_position(prm_max_position),
    _moveable_length(_max_position - _min_position)
    {
        _curtain_position = _min_position; //ŏ100%JĂ܂
        _state = OPENED;
        _opening_velocity = 0.0f;
        _closing_velocity = 0.0f;
    }

    /**
     * J[eXɊJ .
     * {\bhsƁAbehave() Ăяo邲Ƃ
     * prm_opening_velocity ÂJ[eJ܂iZȂ܂jB
     * J[ê͒ 0.0 ܂łB 0.0  ɒBƁAprocessCloseDone() 
     * R[obN open ͏I܂B
     * @param prm_opening_velocity J[eJ鑬x
     */
    virtual void openV(T prm_opening_velocity) {
        _opening_velocity = prm_opening_velocity;
        _state = OPEN;
    }
    virtual void openT(frame prm_opening_frames) {
        T length = _curtain_position - _min_position;
        openV((T)(length / prm_opening_frames));
    }

    /**
     * J[e𑦍ɍőɊJ .
     * {\bhs J[e[̒ 0.0 ɂȂ܂B
     * processOpenDone() PR[obN܂B
     */
    virtual void open() {
        _curtain_position = _min_position;
        _state = OPENED;
        processOpenDone();
    }

    /**
     * J[eXɕ߂ .
     * {\bhsƁAbehave() Ăяo邲Ƃ
     * prm_closing_velocity ÂJ[eĂ䂫܂iȂ܂jB<BR>
     * J[e[̒̍ő̓ftHg 1.0 łB
     * ftHgł͒ 1.0 ɒBƁAprocessCloseDone() P
     * R[obNAclose ͏I܂B
     * @param prm_closing_velocity J[e߂鑬x
     */
    virtual void closeV(T prm_closing_velocity) {
        _closing_velocity = prm_closing_velocity;
        _state = CLOSE;
    }
    virtual void closeT(frame prm_closing_frames) {
        T length = _moveable_length - (_curtain_position - _min_position);
        closeV((T)(length / prm_closing_frames));
    }

    /**
     * J[e𑦍ɕ߂ .
     * {\bhs J[e[̒ 1.0 ɂȂ܂B
     * processOpenDone() PR[obN܂B
     */
    virtual void close() {
        _curtain_position = _max_position;
        _state = CLOSED;
        processCloseDone();
    }

    /**
     * J[e𓮂 .
     * {NX@\ɂɂ́At[̃\bhsĂB
     * t[ behave() 邱ƂŎ̃\bhKXĂяo悤ɂȂ܂B
     * <code>
     * processOpenBegin()
     * processOpening()
     * processOpenDone()
     * processCloseBegin()
     * processClosing()
     * processCloseDone()
     * </code>
     */
    virtual void behave() {
        if (_state == OPEN) {
            processOpenBegin();
            _state = OPENING;
        } else if (_state == OPENING) {
            _curtain_position -= _opening_velocity;
            if (_curtain_position < _min_position) {
                _curtain_position = _min_position;
                _state = OPENED;
                processOpenDone();
            } else {
                processOpening();
            }

        } else if (_state == CLOSE) {
            processCloseBegin();
            _state = CLOSING;
        } else if (_state == CLOSING) {
            _curtain_position += _closing_velocity;
            if (_curtain_position > _max_position) {
                _curtain_position = _max_position;
                _state = CLOSED;
                processCloseDone();
            } else {
                processClosing();
            }
        }
    }

    /**
     * J[ëʒui܂Ăj擾 .
     * @return J[ëʒu
     */
    virtual T getCurtainPosition() {
        return _curtain_position;
    }

    /**
     * J[e̕܂Ă銄擾 .
     * @return J[e̕܂Ă銄(0.0F100%JĂ ` 1.0F100%܂Ăj
     */
    virtual T getClosingRatio() {
        return (_curtain_position - _min_position) / _moveable_length;
    }

    /**
     * J[eJn߂uԂ̏ .
     * open() ɂPxĂяo܂B <BR>
     * ʂŎۂ̏ĂB<BR>
     */
    virtual void processOpenBegin() {};

    /**
     * J[eJĂȑ .
     * ʂŎۂ̏ĂB<BR>
     */
    virtual void processOpening() {};

    /**
     * J[eJ̏ .
     * _curtain_position  0.0 ɂȂۂPxĂяo܂B <BR>
     * ʂŎۂ̏ĂB<BR>
     */
    virtual void processOpenDone() {};

    /**
     * J[e܂n߂uԂ̏ .
     * close() ɂPxĂяo܂B <BR>
     * ʂŎۂ̏ĂB<BR>
     */
    virtual void processCloseBegin() {};

    /**
     * J[e܂Ăȑ .
     * ʂŎۂ̏ĂB<BR>
     */
    virtual void processClosing() {};

    /**
     * J[e܂̏ .
     * _curtain_position  _curtain_rail_length ɂȂۂPxĂяo܂B <BR>
     * ʂŎۂ̏ĂB<BR>
     */
    virtual void processCloseDone() {};




    virtual ~Curtain() {

    }
};

}
#endif /*GGAF_CORE_CURTAIN_H_*/
