package slothLib.web.search;

import java.net.Proxy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.*;

import slothLib.portUtil.PortUtil;

/**
 * Yahoo!ウェブ画像Webサービス
 * http://developer.yahoo.co.jp/search/image/V1/imageSearch.html
 * <remarks>
 * <newpara>[2007-05-24][kondo]作成</newpara>
 * </remarks>
 */
public class YahooJpImageSearch implements IImageSearch
{
	
	// プライベートフィールド
	
	// アプリケーションID
	private String applicationID;
	
	// query と results(resultNumber) 以外の引数
	private SearchType type;
	private int start;
	private SearchFormat format;
	private boolean adultOk;
	private Color coloration;
	private String[] site;
	
	// proxy
	private Proxy proxy;
	
	// 結果が繰り返さないようにするために必要
	Map<String, Boolean> alreadyGotURL = new HashMap<String, Boolean>();
	
	

	
	// コンストラクタ
	
	// / <summary>
	// / コンストラクタ
	// / アプリケーションIDを指定する
	// / </summary>
	// / <param name="applicationID">プログラムで用いるアプリケーションID</param>
	public YahooJpImageSearch(String applicationID)
	{
		this.applicationID = applicationID;
		
		// query と resultNumber 以外の引数
		this.type= SearchType.all;
		this.start= 1;
		this.format = SearchFormat.any;
		this.adultOk = false;
		this.coloration = Color.any;
		this.site = null;
	}
	

	
	// public メソッド
	
	// / <summary>
	// / Yahoo!画像検索を行う
	// / </summary>
	// / <param name="query">検索クエリーです。文字を含む場合は '+'、含まない場合は '-' を指定します。フレーズの場合は
	// "検索 クエリー" のように " でくくります。</param>
	// / <param name="maxNumber">返却結果の数です。</param>
	// / <returns>YahooJpImageSearchResult型の検索結果</returns>
	public YahooJpImageSearchResult doSearch(String query, int maxNumber) throws WebSearchException
	{
		return doSearchOver(query, maxNumber);
	}
	
	
	// / <summary>
	// / 検索HIT数のみを取得するメソッド
	// / </summary>
	// / <param name="query">検索クエリ</param>
	// / <returns>HIT数</returns>
	public long getTotalNumber(String query) throws WebSearchException 		
	{
		return doSearch(query, 1).getTotalResultsAvailable();
	}
	

	
	// プロパティ
	
		
	/**
	 * 指定検索の種類 デフォルトはall
	 */
	public SearchType getType()
	{
		return this.type;
	}
	public void setType(SearchType value)
	{
		this.type = value;
	}
	
	/**
	 * 検索するファイルの種類　デフォルトはany
	 */
	public SearchFormat getFormat()
	{
		return this.format;
	}
	public void setFormat(SearchFormat value)
	{
		this.format = value;
	}
	
	/**
	 * アダルトコンテンツの検索結果を含めるかどうかを指定します。デフォルトはfalse
	 */
	public boolean getAdultOk()
	{
		return this.adultOk;
	}
	public void setAdultOk(boolean value)
	{
		this.adultOk = value;
	}
	
	/**
	 * カラー画像だけを検索結果とするかどうかを指定します（カラーか白黒）。
	 */
	public Color getColoration()
	{
		return this.coloration;
	}
	public	void setColoration(Color value)
	{
		this.coloration = value;
	}
	
	/**
	 * 検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます。デフォルトはnull
	 */
	public String[] getSite()
	{
		return this.site;
	}
	public	void setSite(String[] value)
	{
		this.site = value;
	}
	
	

	
	// Proxy設定
	
	/**
	 * プロクシを設定する。
	 * @param proxyAddress 
	 */
	public void setProxy(String proxyAddress)
	{
		if (proxyAddress == null)
		{
			this.proxy = null;
		}
		else
		{
			this.proxy = PortUtil.createProxy(proxyAddress);
		}
	}
	

	
	// フル機能の検索メソッド
	
	/**
	 * 50件までのYahoo!画像検索を行う
	 * @param query 検索クエリーです。文字を含む場合は '+'、含まない場合は '-' を指定します。フレーズの場合は "検索 クエリー" のように " でくくります。
	 * @param results 返却結果の数です。
	 * @param start 返却結果の先頭位置です。最終位置（start + results - 1）は、1000を超えられません。
	 * @return YahooJpImageSearchResult型の検索結果
	 */
	private YahooJpImageSearchResult doSearchOriginal(String query, int results, int start) throws WebSearchException
	{
		String requestURL = makeRequestURL(query, this.type, results, start, this.format, this.adultOk, this.coloration, this.site);

		Document xmlDoc = PortUtil.getDocumentFromURL(requestURL, this.proxy);
		// ルートの要素を取得
		Element xmlRoot = xmlDoc.getDocumentElement();
		
		
		// <Result>要素のtotalResultsAvailableの属性値を取得
		long totalResultsAvailable = Long.parseLong(xmlRoot.getAttribute("totalResultsAvailable"));
		int totalResultsReturned = Integer.parseInt(xmlRoot.getAttribute("totalResultsReturned"));
		int firstResultPosition = Integer.parseInt(xmlRoot.getAttribute("firstResultPosition"));
		
		// 検索結果要素を入れていく
		List<YahooJpImageElement> resultElementList = new ArrayList<YahooJpImageElement>();
		
		NodeList xmlResultList = xmlRoot.getElementsByTagName("Result");
		
		int rank = start - 1;
		
		String firstURL = getElementString(((Element)xmlResultList.item(0)).getElementsByTagName("Url"));
		if (alreadyGotURL.containsKey(firstURL))
		{
			return new YahooJpImageSearchResult(query, 0, 0, -1, new YahooJpImageElement[0]);
		}
		else
		{
			alreadyGotURL.put(firstURL, true);
		}
		for (int i = 0; i < xmlResultList.getLength(); i++)
		{
			Element xmlResult = (Element)xmlResultList.item(i);
			rank++;
			String title = getElementString(xmlResult.getElementsByTagName("Title"));
			String summary = getElementString(xmlResult.getElementsByTagName("Summary"));
			String url = getElementString(xmlResult.getElementsByTagName("Url"));
			String clickUrl = getElementString(xmlResult.getElementsByTagName("ClickUrl"));
			String refererUrl = getElementString(xmlResult.getElementsByTagName("RefererUrl"));
			String fileSize = getElementString(xmlResult.getElementsByTagName("FileSize"));
			String fileFormat = getElementString(xmlResult.getElementsByTagName("FileFormat"));
			double height = Double.parseDouble(getElementString(xmlResult.getElementsByTagName("Height")));
			double width = Double.parseDouble(getElementString(xmlResult.getElementsByTagName("Width")));
			NodeList xmlThumbnailNode = xmlResult.getElementsByTagName("Thumbnail");
			String thumbnailUrl = "";
			double thumbnailHeight = 0.0;
			double thumbnailWidth = 0.0;
			if (xmlThumbnailNode != null)
			{
				for (int j = 0; j < xmlThumbnailNode.getLength(); j++)
				{
					Element xmlThumbnailElement = (Element)xmlThumbnailNode.item(j);
					thumbnailUrl = getElementString(xmlThumbnailElement.getElementsByTagName("Url"));
					thumbnailHeight = Double.parseDouble(getElementString(xmlThumbnailElement.getElementsByTagName("Height")));
					thumbnailWidth = Double.parseDouble(getElementString(xmlThumbnailElement.getElementsByTagName("Width")));
				}
			}            
			String publisher = getElementString(xmlResult.getElementsByTagName("Publisher"));
			boolean frameOk = true;
			boolean inlineOk = true;
			
			if (xmlResult.hasAttribute("noframe"))
			{
				frameOk = false;
			}
			if (xmlResult.hasAttribute("noinline"))
			{
				inlineOk = false;
			}
			
			String copyright = getElementString(xmlResult.getElementsByTagName("Copyright"));
			
			YahooJpImageElement result = new YahooJpImageElement(rank, title, summary, url, clickUrl, refererUrl, fileSize,
					fileFormat, height, width, thumbnailUrl, thumbnailHeight, thumbnailWidth, publisher, frameOk, inlineOk, copyright);
			resultElementList.add(result);
		}
		
		return new YahooJpImageSearchResult(query, totalResultsAvailable, totalResultsReturned, firstResultPosition, resultElementList.toArray(new YahooJpImageElement[0]));
	}
	
	
	/**
	 * Yahoo!画像検索を行う
	 * @param query 検索クエリーです。文字を含む場合は '+'、含まない場合は '-' を指定します。フレーズの場合は "検索 クエリー" のように " でくくります。
	 * @param results 返却結果の数です。
	 * @return YahooJpImageSearchResult型の検索結果
	 */
	private YahooJpImageSearchResult doSearchOver(String query, int results) throws WebSearchException
	{
		int i;
		int loop = (results - 1) / 50;
		List<YahooJpImageElement> resultElementList = new ArrayList<YahooJpImageElement>();
		long totalResultsAvailable = 0;
		YahooJpImageSearchResult r;
		
		for (i = 0; i < loop; i++)
		{
			r = doSearchOriginal(query,50, ((i * 50) + start));
			if (r.getFirstResultPosition() == -1) { break; }
			totalResultsAvailable = r.getTotalResultsAvailable();
			resultElementList.addAll(PortUtil.toList(r.getResultElements()));
		}
		r = doSearchOriginal(query,  results - (loop * 50), loop * 50 + start);
		resultElementList.addAll(PortUtil.toList(r.getResultElements()));
		if (r.getFirstResultPosition() == -1)
		{
		}
		else
		{
			totalResultsAvailable = r.getTotalResultsAvailable();
		}
		
		return new YahooJpImageSearchResult(query, totalResultsAvailable, resultElementList.size(), start, resultElementList.toArray(new YahooJpImageElement[0]));
	}
	

	
	// 雑用メソッド
	
	/**
	 * 例外を投げる
	 */
//	/// <param name="errorCode"></param>
//	/// <param name="errorMessage"></param>
//	private void ThrowException(HttpStatusCode errorCode, String errorMessage)
//	{
//		switch (errorCode)
//		{
//		case HttpStatusCode.BadRequest: // 400
//			throw new YahooJpSearchException(YahooJpSearchException.HttpCode.BadRequest, errorMessage);
//		case HttpStatusCode.Forbidden: // 403
//			throw new YahooJpSearchException(YahooJpSearchException.HttpCode.Forbidden, errorMessage);
//		case HttpStatusCode.ServiceUnavailable: // 503
//			throw new YahooJpSearchException(YahooJpSearchException.HttpCode.ServiceUnavailable, errorMessage);
//		default:
//			throw new Exception("YahooWebSearchで想定外のHTTPエラーが発生しました。（エラーコード: " + (int)errorCode + "）" + errorMessage);
//		}
//	}
//	
	/**
	 * XmlNodeListの初めのノードのテキストを取得する
	 * @param nodeList XmlNodeList
	 * <returns>XmlNodeListの初めのノードのInnerText
	 *          XmlNodeListが空であれば空文字列を返す</returns>
	 */
	private String getElementString(NodeList nodeList)
	{
		if (nodeList.getLength() == 0)
		{
			return "";
		}
		else
		{
			return nodeList.item(0).getTextContent();
		}
	}
	
	/**
	 * リクエストURLを作成する
	 * @param query 検索クエリーです。文字を含む場合は '+'、含まない場合は '-' を指定します。フレーズの場合は "検索 クエリー" のように " でくくります。
	 * @param type 指定検索の種類　デフォルトはall
	 * @param results 検索取得数
	 * @param start 検索開始の先頭位置
	 * @param format 検索するファイルの種類　デフォルトはany
	 * @param adultOk アダルトコンテンツの検索結果を含めるかどうか　デフォルトはfalse
	 * @param coloration カラー画像だけを検索結果とするかどうかを指定します（カラーか白黒）。
	 * @param site 検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます。デフォルトはnull
	 * @return URL
	 */
	private String makeRequestURL(String query, SearchType type, int results, int start, SearchFormat format, boolean adultOk,
			Color coloration, String[] site)
	{
		String strType = "";
		String strFormat = "";
		String strAdult = "";
		String strColoration = ""; 
		String strSite = "";
		
		switch (type)
		{
		case all:
			break;
		case any:
			strType = "&type=any";
			break;
		case phrase:
			strType = "&type=phrase";
			break;
		}
		
		switch (format)
		{
		case any:
			break;
		case bmp:
			strFormat = "&format=bmp";
			break;
		case gif:
			strFormat = "&format=gif";
			break;
		case jpeg:
			strFormat = "&format=jpeg";
			break;
		case png:
			strFormat = "&format=png";
			break;
		}
		
		if (adultOk == true)
		{
			strAdult = "&adulat_ok=1";
		}
		
		switch (coloration)
		{
		case any:
			break;
		case bw:
			strColoration = "&coloration=bw";
			break;
		case color:
			strColoration = "&coloration=color";
			break;
		}
		
		if (site != null)
		{
			if (site.length > 30)
			{
				throw new IllegalArgumentException("siteに指定できるドメインは30個までです " + "site");
			}
			else
			{
				for (String s: site)
				{
					strSite += "&site=" + s;
				}
			}
		}
		
		String uriString = "http://api.search.yahoo.co.jp/ImageSearchService/V1/imageSearch?"
			+ "appid=" + this.applicationID
			//+ "&query=" + System.Web.HttpUtility.UrlEncode(query, Encoding.UTF8)
			+ "&query=" + query
			+ strType
			+ "&results=" + results
			+ "&start=" + start
			+ strFormat
			+ strAdult
			+ strColoration
			+ strSite;
		
		
		//return uriString;
		
//		Uri uri = new Uri(uriString);
		String result = PortUtil.absoluteUri(uriString);
		//System.Diagnostics.Debug.WriteLine(result);
		
		
		return result;
	}
	

	
	// 引数で指定するための列挙型
	
	/**
	 * 指定検索の種類
	 */
	public enum SearchType
	{
		/**
		 * 全クエリー文字を含む検索結果を返します（デフォルト）。
		 */
		all,
		/**
		 * クエリー文字のうちいずれかを含む検索結果を返します。
		 */
		any,
		/**
		 * クエリー文字を文章として含む検索結果を返します。
		 */
		phrase
	}
	
	/**
	 * 検索するファイルの種類 デフォルトはany
	 */
	public enum SearchFormat
	{
		/**
		 * デフォルト
		 */
		any,
		/**
		 * BMP形式
		 */
		bmp,
		/**
		 * GIF形式
		 */
		gif,
		/**
		 * JPEG形式
		 */
		jpeg,
		/**
		 * PNG形式
		 */
		png
	}
	
	/**
	 * 検索結果の画像の種類 デフォルトはany
	 */
	public enum Color
	{
		/**
		 * デフォルト
		 */
		any,
		/**
		 * カラー画像
		 */
		color,
		/**
		 * 白黒画像
		 */
		bw
	}
	

	
	// IImageSearch メンバ
	
	/**
	 * 画像検索エンジンで検索する
	 * @param query 検索クエリ
	 * @param resultNum 取得件数
	 * @return 検索結果
	 */
//	IImageSearchResult IImageSearch.DoSearch(String query, int resultNum)
//	{
	//	return this.DoSearch(query, resultNum);
	//}
	

	
	// ISearch メンバ
	
	/**
	 * サーチエンジンで検索する
	 * @param query 検索クエリ
	 * @param resultNum 取得件数
	 * @return 検索結果
	 */
	//ISearchResult ISearch.DoSearch(String query, int resultNum)
//	{
	//	return this.DoSearch(query, resultNum);
	//}
	

	
}
