//using System;
//using System.IO;
//using System.Collections.Generic;
//using System.Text;
//using System.Web;
//using System.Xml;
//using System.Net;
//using SlothLib.Web.Search;


package slothLib.web.search;

import java.net.Proxy;
import java.util.ArrayList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import slothLib.portUtil.PortUtil;

    /**
     * FlickrAPIサーチを実行するクラス
     * http://www.flickr.com/services/api/
     */
    public class FlickrSearch implements IImageSearch
    {

        // プライベートフィールド
        /**
         * クラスが保持するAPIKey
         */
        private String apiKey;

        // proxy
        private Proxy proxy;




        // 列挙型

        /**
         * 検索クエリの種類
         */
        public enum SearchType
        {
            /**
             * テキストクエリ（デフォルト）
             */
            Text,
            /**
             * タグクエリ
             */
            Tag
        }

        /**
         * タグの種類（ORもしくはAND）
         */
        public enum TagMode
        {
            /**
             * OR結合（デフォルト）
             */
            Any,
            /**
             * AND結合
             */
            All
        }



        // コンストラクタ

        /**
         * コンストラクタ　APIKeyを指定する
         * @param apiKey プログラムで用いるAPIKey
         */

        public FlickrSearch(String apiKey)
        {
            this.apiKey = apiKey;
        }




        // Proxy設定

        /**
         * プロクシを取得・設定する。
         */
        public String getProxy()
        {
        	return PortUtil.getProxyURL(this.proxy);
        }
        public void setProxy(String value){
            if (PortUtil.isNullOrEmpty(value))
            {
                this.proxy = null;
            }
            else 
            {
            	this.proxy = PortUtil.createProxy(value);
            }
        }



        // DoSearch


        /**
         * Flickr検索を行う
         * @param query 検索クエリ
         * @param num 検索取得数 500以上は500に置き換えられます
         * @param searchType 検索クエリの種類　デフォルトはText
         * @param tagMode タグの種類　デフォルトはAny
         * @return FlickrResult型の検索結果
         */
        public FlickrSearchResult doSearch(String query, int num, SearchType searchType, TagMode tagMode)
        throws WebSearchException
        {
            //アプリケーションIDが登録されてない場合エラー
            if (apiKey == "")
            {
                throw new FlickrException("No APIKey");
            }

            //使用中のアプリケーションIDを出力
            System.err.println("API_Key:"+apiKey);

            FlickrSearchResult searchResult = null;

            try
            {
                searchResult = doFlickrSearch(query, num, searchType, tagMode);
            }

            catch (WebSearchException e)
            {
                System.err.println(e.getMessage());
                throw e;

            }

            finally
            {

            }


            return searchResult;

        }



        // DoFlickrSearch


        /**
         * Flickr検索を行う
         * @param query 検索クエリー
         * @param num 検索取得数
         * @param searchType 検索クエリーの種類　デフォルトはText
         * @param tagMode タグの種類　デフォルトはAny
         * @return FlickrResult型の検索結果
         */
        private FlickrSearchResult doFlickrSearch(String query, int num, SearchType searchType, TagMode tagMode)
        throws WebSearchException
        {
            String requestURL = makeFlickrQuery(query, num, searchType, tagMode);
            System.err.println("RequestURL:"+requestURL);
            Document xmlDoc = PortUtil.getDocumentFromURL(requestURL, proxy);

                // ルートの要素を取得
                Element xmlRoot = xmlDoc.getDocumentElement();

                //ルート=<rsp>要素のstat属性がfailの場合
                if (xmlRoot.getAttribute("stat").equals("fail"))
                {
                    String errcode = xmlRoot.getFirstChild().getAttributes().getNamedItem("code").getNodeValue();
                    String errmsg = xmlRoot.getFirstChild().getAttributes().getNamedItem("msg").getNodeValue();
                    //エラー番号とエラーメッセージをスルー
                    throw new FlickrException(errcode+":"+errmsg);
                    
                }

                Node xmlResultsNode = xmlRoot.getFirstChild();

                // 検索結果要素のtotalの属性値を取得
                long resultNum = Long.parseLong(xmlResultsNode.getAttributes().getNamedItem("total").getNodeValue());

                // 検索結果要素を入れていく
                List<FlickrElement> resultElementList = new ArrayList<FlickrElement>();

                NodeList xmlResultList = xmlResultsNode.getChildNodes();

                for (int i = 0; i < xmlResultList.getLength(); i++)
                {
                	Element xmlResult = (Element)xmlResultList.item(i);
                	String id = xmlResult.getAttribute("id");
                    String owner = xmlResult.getAttribute("owner");
                    String secret = xmlResult.getAttribute("secret");
                    String server = xmlResult.getAttribute("server");
                    String farm = xmlResult.getAttribute("farm");
                    String title = xmlResult.getAttribute("title");
                    boolean isPublic = bitToBoolean(xmlResult.getAttribute("ispublic"));
                    boolean isFriend = bitToBoolean(xmlResult.getAttribute("isfriend"));
                    boolean isFamily = bitToBoolean(xmlResult.getAttribute("isfamily"));
                    FlickrElement result = new FlickrElement(id, owner, secret, server, farm, title, isPublic, isFriend, isFamily);
                    resultElementList.add(result);

                }

                return new FlickrSearchResult(query, resultNum, resultElementList.toArray(new FlickrElement[0]));



        }



        // MakeFlickrQuery

        /**
         * リクエストURLを作成する
         * @param query 検索クエリー
         * @param results 検索取得数
         * @param searchType 検索クエリーの種類　デフォルトはText
         * @param tagMode タグの種類　デフォルトはAny
         * @return URL
         */
        private String makeFlickrQuery(String query, int results, SearchType searchType, TagMode tagMode)
        {
            String str = "";
            String mode = "";
            String flickrQuery = "http://api.flickr.com/services/rest/?method=flickr.photos.search";

            // スペースで区切られたクエリをカンマで区切られたクエリに変換
            String csvQuery = makeCSVQuery(query);

            // SearchTypeとTagModeによる検索タイプの決定
            switch (searchType)
            {
                case Text:
                    str = "&text=" + csvQuery;
                    break;
                case Tag:
                    str = "&tags=" + csvQuery;
                    break;
            }

            switch (tagMode)
            {
                case Any:
                    mode = "&tag_mode=any";
                    break;
                case All:
                    mode = "&tag_mode=all";
                    break;
            }

            flickrQuery += "&api_key=" + apiKey + str + "&per_page=" + results + mode;

            return flickrQuery;

        }



        // MakeCSVQuery

        /**
         * ' 'で連結されたクエリを','で連結されたクエリに直す
         * @param query スペースで区切られたクエリ
         * @return String型のカンマで区切られたクエリ
         */
        private String makeCSVQuery(String query)
        {
            String[] array = query.split(" ");
            String q = "";
            for (String s: array)
                q += s + ",";
            return q.substring(0, q.length() - 1);

        }



        // BitToBoolean

        /**
         * "0"か"1"のStringを受け、1ならばtrueを返し、それ以外ならばfalseを返す
         * @param bit "0"か"1"の文字列
         * @return boolean型の値
         */
        private boolean bitToBoolean(String bit)
        {
            if (bit.contains("1"))
            {
                return true;
            }
            else
            {
                return false;
            }
        }



        // GetSearchType,GetFormat

        /**
         * 検索クエリのタイプを返す
         * @return SearchType(列挙型) デフォルトは"Text"
         */
        public static SearchType getSearchType(String type)
        {
            SearchType st;

            String lower = type.toLowerCase();
            if (lower.equals("tag")) {
                st = SearchType.Tag;
            } 
            else if (lower.equals("text")) {
            	st = SearchType.Text;
            }
            else 
            {
            	st = SearchType.Text; //defaultはText
            }

            return st;

        }

        /**
         * Tagの種類を返す
         * @return TagMode(列挙型) デフォルトは"Any"
         */
        public static TagMode getTagMode(String tagMode)
        {
            TagMode tm;

            String lower = tagMode.toLowerCase();
            if (lower.equals("any"))
            {
            	tm = TagMode.Any;
            } 
            else if (lower.equals("all"))
            {
            	tm = TagMode.All;
            }
            else {
            	tm = TagMode.Any; //defaultはphoto
            }

            return tm;

        }



//        //#region IImageSearch メンバ
//
//        /// <summary>
//        /// 検索を実行し結果を格納する
//        /// </summary>
//        /// <param name="query">検索クエリ</param>
//        /// <param name="num">検索結果取得数</param>
//        /// <returns>IImageSearchResult型の検索結果</returns>
//        IImageSearchResult IImageSearch.DoSearch(String query, int num)
//        {
//            return DoSearch(query, num, SearchType.Text, TagMode.Any);
//        }
//
//        //#endregion
//
//        //#region ISearch メンバ
//
//        /// <summary>
//        /// 検索を実行し結果を格納する
//        /// </summary>
//        /// <param name="query">検索クエリ</param>
//        /// <param name="num">検索結果取得数</param>
//        /// <returns>ISearchResult型の検索結果</returns>
        public IImageSearchResult doSearch(String query, int num)
        throws WebSearchException
        {
            return doSearch(query, num, SearchType.Text, TagMode.Any);
        }

//        //#endregion

    }
