/* vim:set ts=4 sts=4 sw=4 noet fenc=utf-8:

   Copyright 2009 senju@users.sourceforge.jp

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */

package jp.sourceforge.rabbitBTS.services;

import java.util.Date;
import java.util.List;

import jp.sourceforge.rabbitBTS.Sht;
import jp.sourceforge.rabbitBTS.dao.AccountDao;
import jp.sourceforge.rabbitBTS.exceptions.HasNotValidIdException;
import jp.sourceforge.rabbitBTS.exceptions.RabbitBTSException;
import jp.sourceforge.rabbitBTS.models.Account;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataRetrievalFailureException;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import com.google.appengine.api.users.User;

/**
 * アカウント/アカウントまわりのサービス
 * 
 * @author senju
 */
public class AccountService {

	/**
	 * システムに最初管理者として登録されるユーザーのアドレス
	 */
	private String firstSuperUser;

	@Autowired
	private AccountDao accountDao;

	/**
	 * 現在ログイン中のgoogleアカウントのアカウントをDBより取得する。
	 * 
	 * <p>
	 * また、"account"の名前でリクエストコンテクストにアカウントを保存する。
	 * 
	 * @return 取得したアカウント。見つからない場合はnull
	 * @throws HasNotValidIdException
	 *             ログインしていないユーザーの場合
	 */
	public Account fetchAccount() throws HasNotValidIdException {
		final User gu = Sht.user();
		if (gu == null) {
			throw new HasNotValidIdException();
		}
		final Account acc = this.accountDao.findAccountByEmail(gu.getEmail());
		if (acc == null) {
			// 見つからない場合
			return null;
		}

		// getCurrentAccount用にリクエストコンテクストに保存しておく。
		final RequestAttributes req = RequestContextHolder
				.currentRequestAttributes();
		req.setAttribute("account", acc, RequestAttributes.SCOPE_REQUEST);
		return acc;
	}

	/**
	 * 現在ログイン中のアカウントを返す。
	 * 
	 * @return 現在ログイン中のアカウント。
	 */
	public Account getCurrentAccount() {
		final RequestAttributes req = RequestContextHolder
				.currentRequestAttributes();
		final Account account = (Account) req.getAttribute("account",
				RequestAttributes.SCOPE_REQUEST);
		return account;
	}

	/**
	 * 指定されたアカウントを登録する。 TODO: トランザクション
	 * 
	 * @param account
	 *            登録するアカウント
	 * @throws RabbitBTSException
	 *             既に登録済みの場合
	 */
	public void registAccount(Account account) throws RabbitBTSException {
		// 登録済みチェック
		if (this.accountDao.findAccountByEmail(account.getEmail()) != null) {
			throw new RabbitBTSException("既に登録されています。");
		}

		// 最初のスーパーユーザーかチェック
		if (account.getEmail().equals(this.firstSuperUser)) {
			account.setAdmin(true);
			Sht.log(this).warn(
					"Admin Account Created. {}", this.firstSuperUser);
		}

		// nickName重複チェック
		if (this.accountDao.findAccountByNickName(account.getNickName()) != null) {
			Sht.log(this).info(
					"nickName {} is already used.", account.getNickName());
			throw new RabbitBTSException("nickNameが重複しています。");
		}

		// 登録処理
		account.setLastAccess(new Date());
		this.accountDao.save(account);

		// TODO: メッセージ追加

	}

	/**
	 * DBからニックネームでアカウントを検索する。
	 * 
	 * @param nickName
	 *            検索するアカウントのニックネーム
	 * @return 見つかったアカウント。無い場合null。
	 */
	public Account getAccountByNickName(String nickName) {
		final Account account = this.accountDao.findAccountByNickName(nickName);
		return account;
	}

	/**
	 * アカウントをIDで検索する。
	 * 
	 * @param id
	 *            検索対象のID
	 * @return 見つかったアカウント。見つからない場合null。
	 */
	public Account getAccountById(Long id) {
		try {
			return this.accountDao.get(id);
		} catch (final DataRetrievalFailureException e) {
			return null;
		}
	}

	/**
	 * アカウントを取得する。 TODO: LastAccess順でソート、件数制限、ページング
	 * 
	 * @return 取得したアカウントのリスト
	 */
	public List<Account> fetchAccounts() {
		return this.accountDao.getAll();
	}

	/**
	 * 指定されたアカウントと関連情報を削除する。
	 * 
	 * @param account
	 *            削除するアカウント。
	 */
	public void deleteAccount(Account account) {
		this.accountDao.remove(account.getAccountId());
	}

	/**
	 * @param firstSuperUser
	 *            the firstSuperUser to set
	 */
	public void setFirstSuperUser(String firstSuperUser) {
		this.firstSuperUser = firstSuperUser;
	}
}
